package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForOnlineClamp;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is the driver for the EPOS controller which controls the
 * Autochanger ONLINE clamps. The model is EPOS2 25/5.
 *
 *
 * @author virieux
 */
public class CanOpenEPOSOnlineClamp extends CanOpenEPOS implements EPOSControllerForOnlineClamp {

    private boolean homingDone = false;

    /**
     * build a new CanOpenEPOSOnlineClamp
      *
     * @param nodeID
     * @param serialNB
     */
    public CanOpenEPOSOnlineClamp(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    /**
     * Release brake to be able to open or close clamp. Only for tests for end
     * users.
     *
     * releaseBrake for an ONlineClamp controller consists in forcing to ZERO bit 15
     * of parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Release brake to be able to move clamp.")
    public void releaseBrake() {
        // If the controller is not enabled and we release brakes it could lead to a
        // lost of a filter.
        if (!isEnabled()) {
            throw new RejectedCommandException(name + " can't releaseBrake if not enabled.");
        }
        doReleaseBrake();
    }

    @Override
    public void activateBrakeAndDisable() {
        activateBrake();
        disableOperation();
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (isBooted()) {
            activateBrakeAndDisable();
        }
    }

    /**
     * return true if homing is done: the controller knows its absolute
     * position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description = "return true if homing is done: the controller knows its absolute position.")
    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     * homing of a ONLINE clamp in final products AC1 and AC2, the ONLINE clamps
     * have encoders. So we open and close them in PROFILE_POSITION mode. This
     * can be done when the homing of the controller has been done before. The
     * homing has to be done when fcs starts or each time the controller has
     * been shutdown. The homing is to be done when the clamp has been openend
     * in CURRENT mode.
     */
    @Override
    public void homing() {
        homingDone = false;
        this.defineAbsolutePosition(0);
        position = readPosition();
        if (position != 0) {
            throw new FcsHardwareException(name + " could not do homing.");
        }
        homingDone = true;
        activateBrakeAndDisable();
    }
}
