package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.LOCK_ONLINECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_ONLINECLAMPS;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.StrainGauge;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSystecIO;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps extends MobileItem implements ControlledBySensors {
    @LookupField(strategy = Strategy.ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = Strategy.TREE)
    protected Autochanger autochanger;

    @LookupField(strategy = Strategy.TREE, pathFilter = ".*onlineStrainGauge")
    private StrainGauge onlineStrainGauge;

    @LookupField(strategy = Strategy.TREE, pathFilter = ".*tempSensorsDevice")
    private CanOpenSystecIO tempSensorsDevice;

    @LookupField(strategy = Strategy.SIBLINGS, pathFilter = "lpmOnlineClampsStatus")
    private DigitalSensor lpmOnlineClampsStatus;

    protected final AutochangerOnlineClamp onlineClampXminus;
    protected final AutochangerOnlineClamp onlineClampXplus;
    protected final AutochangerOnlineClamp onlineClampYminus;
    protected final AutochangerOnlineClamp[] clampsList;

    private FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    /* Tools for current ramp */
    protected ScheduledFuture<?> currentRampHandle;

    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @ConfigurationParameter(description = "timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    protected int timeoutForLockingClamps = 15000;

    @ConfigurationParameter(description = "timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.")
    protected int timeoutForUnlockingClamps = 20000;

    @ConfigurationParameter(description = "maximum time in milliseconds to lock the 3 clamps")
    protected int maxTimeToLockAllClamps = 3000;

    @ConfigurationParameter(description = "maximum time in milliseconds to unlock the 3 clamps")
    protected int maxTimeToUnlockAllClamps = 3000;

    @ConfigurationParameter(description = "maximum time in milliseconds to open the 3 clamps")
    protected int maxTimeToOpenClampsX = 2000;

    @ConfigurationParameter(description = "maximum time in milliseconds to close the 3 clamps")
    protected int maxTimeToCloseClampsX = 2000;

    @ConfigurationParameter(description = "minimal period for current ramps. should be > 50")
    protected int minPeriod = 200;

    /**
     * **************************************************
     */
    /* configurable parameters for onlineStrainGauge */

    /* if strain < maxLockedStrain, clamps are LOCKED */
    @ConfigurationParameter
    private short maxLockedStrain = 1484;

    /* if strain < minLockedStrain, clamps may be in ERROR */
    @ConfigurationParameter
    private short minLockedStrain = 500;

    /* if maxLockedStrain <= strain < maxClosedStrain, clamps are CLOSED */
    /* if strain >= maxClosedStrain, clamps are OPENED */
    @ConfigurationParameter
    private short maxClosedStrain = 1572;
    /* end of configurable parameters for onlineStrainGauge */

    @ConfigurationParameter
    private double strainGain = 2.978;

    private short readStrain = 0;

    private short normalizedStrain = 0;

    /**
     * Used to know if lockStatus has been initialized from gauge strain. At
     * startup this value is false. At first updateState this value is set to
     * true.
     */
    private boolean lockStatusInitialized = false;

    /**
     * **************************************************
     */
    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClamp.
     *
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampYminus
     */
    public AutochangerThreeOnlineClamps(AutochangerOnlineClamp onlineClampXminus, AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampYminus;
        clampsList = new AutochangerOnlineClamp[]{onlineClampXminus, onlineClampXplus, onlineClampYminus};
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if lockStatus has been first initialized from strain.")
    public boolean isLockStatusInitialized() {
        return lockStatusInitialized;
    }

    /**
     *
     * @return true if homing has been done for the 3 controllers of clamps.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if homing has been done for the 3 ONLINE clamp controllers.")
    public boolean isHomingDone() {
        return onlineClampXminus.isHomingDone() && onlineClampXplus.isHomingDone()
                && onlineClampYminus.isHomingDone();
    }

    /**
     * set a minimal value for current ramps period
     *
     * @param minPeriod
     */
    @ConfigurationParameterChanger
    public void setMinPeriod(int minPeriod) {
        if (minPeriod < 50) {
            throw new IllegalArgumentException(minPeriod + " bad value for minPeriod. should be > 50");
        } else {
            this.minPeriod = minPeriod;
        }
    }

    /**
     *
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=LOCKED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are LOCKED.")
    public boolean isLocked() {
        return onlineClampXminus.isLocked() && onlineClampXplus.isLocked() && onlineClampYminus.isLocked();
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isClosed() {
        return onlineClampXminus.isClosed() && onlineClampXplus.isClosed() && onlineClampYminus.isClosed();
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are OPENED.")
    public boolean isOpened() {
        return onlineClampXminus.isOpened() && onlineClampXplus.isOpened() && onlineClampYminus.isOpened();
    }

    /**
     * @return true if the 3 clamps are in Travel between opened position and
     * closed position.
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel() && onlineClampXplus.isInTravel() && onlineClampYminus.isInTravel();
    }

    /**
     * Returns true if LockStatus=ERROR
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if one of the clamp is in error.")
    @Override
    public boolean isInError() {
        return onlineClampXminus.isInError() || onlineClampXplus.isInError() || onlineClampYminus.isInError();
    }

    /**
     * Return true if the 3 onlineClamps hardware is lockStatusInitialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
            + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized()
                && onlineClampYminus.isInitialized();
    }

    /**
     * Check that the 3 controllers of the clamps are lockStatusInitialized and
     * that parameters are correct.
     * @param action
     */
    protected void enableAndCheckControllersBeforeAction(MobileItemAction action) {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.enableAndCheckControllerBeforeAction(action);
        }
    }

    /**
     * do homing of the 3 clamps
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller")
    public void homing() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.openInCurrentModeAndHoming();
        }
    }

    /**
     * Closes and locks the online clamps. This command closes the 3 clamps.
     * Then when the clamps are closed, it lock the clamps. The closing is made
     * in PROFILE_POSITION mode, the locking in CURRENT mode.
     *
     * state before this action : OPENED ; state after this action : LOCKED
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION,
            level = Command.NORMAL,
            description = "Closes and locks the online clamps.",
            autoAck = false,
            timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeAndLockClamps() {
        updateStateAndCheckSensors();
        subs.helper()
                .precondition(isHomingDone(), " homing could not be done")
                .precondition(isOpened(), " clamps have to be opened before closeAndLockClamps command")
                .action(() -> {
            closeClamps();
            doLock();
            activateBrakesAndDisableOperation();
                });
    }



    /**
     * Unlock and open the online clamps. Unlock is done in CURRENT mode. open
     * is done in PROFILE_POSITION mode. initial state = LOCKED - final state =
     * OPENED
      *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unlock and open the online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockAndOpenClamps() {
        updateStateAndCheckSensors();
        autochanger.checkConditionsForActioningOnlineClamps();
        enableAndCheckControllersBeforeAction(UNLOCK_AND_OPEN_ONLINECLAMPS);
        // TODO test conditions for unlocking clamps
        // ce n'est pas une MobileItemAction parce que ça ne marchait pas.
        if (this.isLocked()) {
            autochanger.checkOnlineClampMotionAllowed();
            this.currentAction = UNLOCK_AND_OPEN_ONLINECLAMPS;
            long dur;
            long beginTime = System.currentTimeMillis();
            doUnlock();
            updateStateAndCheckSensors();
            if (!isClosed()) {
                throw new FailedCommandException(name + "couldn't unlock clamps");
            }
            openClamps();
            updateStateAndCheckSensors();
            if (!isOpened()) {
                throw new FailedCommandException(name + "couldn't open clamps");
            }
            dur = System.currentTimeMillis() - beginTime;
            FCSLOG.fine(name + " unlockAndOpenClamps duration = " + dur);

        } else if (this.isOpened()) {
            FCSLOG.info(name + " clamps already UNLOCKED and OPENED - nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be LOCKED before unlockAndOpen action.");
        }
    }

    /**
     * test periodicTask.
     */
    public void testPeriodTask() {
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask(name + "-updateStateAndCheckSensors", this::updateStateAndCheckSensors)
                .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofMillis(5000)));
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = " close clamps in mode PROFILE_POSITION for AC1 and AC2")
    public void closeClamps() {
        if (!isHomingDone()) {
            throw new RejectedCommandException(name + " homing not done yet. Can't close.");
        }
        updateStateAndCheckSensors();

        if (isOpened()) {
            autochanger.checkConditionsForActioningOnlineClamps();
            onlineClampYminus.close();
            /* See startAction */
            this.executeAction(CLOSE_ONLINECLAMPSX, timeoutForLockingClamps);

        } else if (isClosed()) {
            FCSLOG.info(name + " clamps already CLOSED nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be unlocked before.");
        }
    }



    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For
     * final pruducts AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Opens the 3 online clamps in mode PROFILE_POSITION.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClamps() {
        if (!isHomingDone()) {
            throw new RejectedCommandException(name + " homing not done yet for the 3 clamps. Can't open.");
        }
        updateStateAndCheckSensors();
        // TODO test conditions for opening clamps
        if (isClosed()) {
            checkReadyForAction();
            onlineClampYminus.open();
            this.executeAction(OPEN_ONLINECLAMPSX, timeoutForUnlockingClamps);

        } else if (isOpened()) {
            FCSLOG.info(name + " clamps already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be closed before.");
        }
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps
     * have to be CLOSED. At the end of this action, the clamps are CLOSED but a
     * strong pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Locks clamps : closed with a strong pressure (high current).", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void lockClamps() {
        updateStateAndCheckSensors();
        if (isClosed()) {
            this.executeAction(LOCK_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isLocked()) {
            FCSLOG.info(name + " is already LOCKED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " have to be CLOSED before to be locked.");
        }
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to
     * decrease pressure on the clamps. The clamps have to be LOCKED. At the end
     * of this action, the clamps are CLOSED with a small pressure of the clamp
     * hardware on the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "unlocks clamps : decreases current in controller to decrease pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockClamps() {
        updateStateAndCheckSensors();
        if (isLocked()) {
            this.executeAction(UNLOCK_ONLINECLAMPS, timeoutForLockingClamps);

        } else if (isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " have to be CLOSED before to be locked.");
        }
    }

    /**
     * Sends current to all controllers to closeClampInCurrentMode clamps and
     * save sentCurrent value. * Controllers have been enabled and brakes
     * released when this method is called. used for actions
     * CLOSE_AND_LOCK_ONLINECLAMPS and CLOSE_ONLINECLAMPS see startAction
     */
    private void sendCurrentToCloseAllClamps() {
        FCSLOG.info(name + " sending current to close all clamps");
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getFinalCurrentToClose());
        }
    }

    /**
     * Sends current to all controllers in order to lock clamps and save
     * sentCurrent value. Controllers have been enabled and brakes released when
     * this method is called.
     */
    private void sendCurrentToLockAllClamps() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.sendCurrentToControllerAndSaveValue(clamp.getCurrentToClamp());
        }
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready.")
    @Override
    public boolean myDevicesReady() {
        return onlineClampXminus.myDevicesReady() && onlineClampXplus.myDevicesReady()
                && onlineClampYminus.myDevicesReady();
    }

    /**
     * returns true if the action given as argument is completed. The completion
     * of action has been confirmed by sensors.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == UNLOCK_ONLINECLAMPS) {
            return isClosed();

        } else if (action == LOCK_ONLINECLAMPS) {
            return isLocked();

        } else if (action == OPEN_ONLINECLAMPSX) {
            return onlineClampXminus.isOpened() && onlineClampXplus.isOpened()
                    && onlineClampXminus.isAtPositionOpen()
                    && onlineClampXplus.isAtPositionOpen();

        } else if (action == CLOSE_ONLINECLAMPSX) {
            return onlineClampXminus.isClosed() && onlineClampXplus.isClosed()
                    && onlineClampXminus.isAtPositionClose()
                    && onlineClampXplus.isAtPositionClose();

        } else {
            throw new IllegalArgumentException(name + " invalid action:" + action);
        }
    }

    /**
     * reads sensors, computes new lockStatus. used in MobileItem to check that
     * action started in startAction is completed.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        enableAndCheckControllersBeforeAction(action);
        autochanger.checkConditionsForActioningOnlineClamps();
        autochanger.checkOnlineClampMotionAllowed();
        autochanger.increaseCurrentMonitoring();
        if (action == UNLOCK_ONLINECLAMPS) {
            /*
             * clamps are LOCKED with a high pressure (currentToClamp previously sent to
             * controllers)
             */
            doUnlock();

        } else if (action == LOCK_ONLINECLAMPS) {
            /* clamps are CLOSED */
            doLock();

        } else if (action == OPEN_ONLINECLAMPSX) {
            onlineClampXminus.getController().enableAndReleaseBrake();
            onlineClampXplus.getController().enableAndReleaseBrake();
            onlineClampXminus.getController().writeTargetPosition(onlineClampXminus.getTargetPositionToOpen());
            onlineClampXplus.getController().writeTargetPosition(onlineClampXplus.getTargetPositionToOpen());
            onlineClampXminus.getController().writeControlWord(0x3F);
            onlineClampXplus.getController().writeControlWord(0x3F);

        } else if (action == CLOSE_ONLINECLAMPSX) {
            onlineClampXminus.getController().enableAndReleaseBrake();
            onlineClampXplus.getController().enableAndReleaseBrake();
            onlineClampXminus.getController().writeTargetPosition(onlineClampXminus.getTargetPositionToClose());
            onlineClampXplus.getController().writeTargetPosition(onlineClampXplus.getTargetPositionToClose());
            onlineClampXminus.getController().writeControlWord(0x3F);
            onlineClampXplus.getController().writeControlWord(0x3F);

        } else {
            throw new IllegalArgumentException(name + " invalid action:" + action);
        }
    }

    /**
     * What to do to abort an action.
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        autochanger.decreaseCurrentMonitoring();
        if (action == OPEN_ONLINECLAMPSX || action == CLOSE_ONLINECLAMPSX) {
            onlineClampXminus.getController().stopAction();
            onlineClampXplus.getController().stopAction();
            onlineClampXminus.getController().activateBrake();
            onlineClampXplus.getController().activateBrake();

        } else {
            /* activate brakes on the 3 clamps controllers and disable operation */
            activateBrakesAndDisableOperation();
        }
    }

    /**
     * What to do to end an action.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        autochanger.decreaseCurrentMonitoring();
        if (action == OPEN_ONLINECLAMPSX || action == CLOSE_ONLINECLAMPSX) {
            onlineClampXminus.getController().activateBrake();
            onlineClampXplus.getController().activateBrake();
            onlineClampXminus.getController().disableOperation();
            onlineClampXplus.getController().disableOperation();

        } else {
            /* activate brakes on the 3 clamps controllers, set current to 0 and disableOperation*/
            activateBrakesAndDisableOperation();
        }

    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * activate brakes on the 3 clamps controllers, set current to 0 and disable
     * operation.
     */
    protected void activateBrakesAndDisableOperation() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.getController().activateBrake();
            clamp.getController().writeCurrent((short) 0);
            clamp.getController().disableOperation();
        }
    }

    /**
     * releases brakes on the 3 clamps controllers
     */
    private void doReleaseBrakes() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.getController().doReleaseBrake();
        }
    }

    protected void doLock() {
        /* first we have to send current to close to avoid an elastic reaction*/
        sendCurrentToCloseAllClamps();
        FcsUtils.sleep(250, name);
        doReleaseBrakes();
        executeCurrentRampToLockAllClamps();
        updateState();
    }

    /**
     * unlock clamps : clamps are LOCKED with a high pressure (currentToClamp
     * previously sent to controllers) avoid an elastic return we send
     * currentToClamp BEFORE the brakes are released.
     */
    protected void doUnlock() {
        sendCurrentToLockAllClamps();
        FcsUtils.sleep(250, name);
        doReleaseBrakes();
        executeCurrentRampToUnlockAllClamps();
        updateState();
    }

    private short computeNormalizedStrain() {
        /*
         * because computeNormalizedStrain is called in postStart we can't use
         * tempClampMotorXplus like temp = tempClampMotorXplus.getValue() / 10
         *
         */
        /* tempClampMotorXplus is always == 0 in postStart */
        double temp = 20;
        if (tempSensorsDevice.isBooted()) {
            temp = (double) tempSensorsDevice.readChannel(3) / 10;
        } else {
            this.raiseWarning(AC_SENSOR_ERROR, " can't read temperature because tempSensorsDevice is not booted,"
                    + " temperature is supposed to be 20°C", name);
        }
        if (onlineStrainGauge.isBooted()) {
            readStrain = onlineStrainGauge.readStrain();
        } else {
            this.raiseWarning(AC_SENSOR_ERROR, " can't read strainGain because onlineStrainGauge is not booted", name);
        }
        normalizedStrain = (short) ((double) readStrain - (strainGain * (temp - 20)));
        FCSLOG.info("temperature=" + temp + " / ONLINE read strain=" + readStrain + " / normalized strain="
                + normalizedStrain);
        return normalizedStrain;
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setLockStatus(lockStatus);
        status.setOnlineClampStrain(normalizedStrain);
        status.setOnlineClampRawStrain(readStrain);
        status.setHomingDone(isHomingDone());
        return status;
    }

    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByThreeClamps()));
    }

    protected void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update state from sensors values.")
    public void updateState() {
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.updateState();
        }
        normalizedStrain = computeNormalizedStrain();
        computeLockStatus();
        this.publishData();
    }

    /**
     * compute lockStatus from the 3 ONLINE clamps lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isClosed()) {
            this.lockStatus = CLOSED;

        } else if (isLocked()) {
            this.lockStatus = LOCKED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else {
            this.lockStatus = UNKNOWN;
        }

    }

    /**
     * return a lock status computed from strain read on onlineStrainGauge.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description = "Return a lock status computed from the normalizedStrain value")
    public FcsEnumerations.LockStatus computeStrainGaugeLockStatus() {
        normalizedStrain = computeNormalizedStrain();
        FCSLOG.info(name + " ONLINE strain = " + normalizedStrain);

        if (normalizedStrain > maxClosedStrain) {
            return OPENED;

        } else if (normalizedStrain > maxLockedStrain) {
            return CLOSED;

        } else if (normalizedStrain > minLockedStrain) {
            return LOCKED;

        } else {
            return ERROR;
        }
    }

    /**
     * Compute lockStatus in reading gauge strain. This is done only the first
     * time updateState is called. After computeLockStatus is called.
     */
    private void computeLockStatusFromStrain() {

        FcsEnumerations.LockStatus strainLockStatus = computeStrainGaugeLockStatus();
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else if (isClosed()) {
            if (strainLockStatus == LOCKED) {
                this.lockStatus = LOCKED;

            } else if (strainLockStatus == CLOSED) {
                this.lockStatus = CLOSED;

            } else if (strainLockStatus == OPENED || strainLockStatus == ERROR) {
                // The strainLockStatus is either OPENED or ERROR
                this.lockStatus = INTRAVEL;

            } else {
                this.lockStatus = ERROR;
            }

        } else {
            /*
             * if lockStatus is not the same for the 3 clamps we are in this case. It can
             * happen during the closing (or the opening) of the 3 clamps when a clamp is
             * already closed but not the 2 others.
             */
            this.lockStatus = UNKNOWN;
        }
    }


    /**
     * Sends current alternativaly to the 3 clamps from finalCurrentToClose to
     * currentToClamp and waits until the task is completed.
     *
     */
    private void executeCurrentRampToLockAllClamps() {
        FCSLOG.info(name + " executing current ramp to lock all clamps.");
        writeCurrentRampToLockAllClamps();
        waitForEndOfCurrentRamp(timeoutForLockingClamps);
    }

    protected void executeCurrentRampToUnlockAllClamps() {
        writeCurrentRampToUnlockAllClamps();
        waitForEndOfCurrentRamp(timeoutForUnlockingClamps);
    }

    /**
     * cancels current ramp send signal to all threads waiting for
     * currentRampEnded condition.
     */
    protected void cancelCurrentRamp() {
        FCSLOG.finest(" => stop writing current");
        this.currentRampHandle.cancel(true);
        FCSLOG.finest(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    protected void waitForEndOfCurrentRamp(int timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF CURRENT RAMP");
        try {
            currentRampHandle.get(timeout, TimeUnit.SECONDS);
        } catch (CancellationException ex) {
            if (currentRampHandle.isDone()) {
                FCSLOG.info(name + " CURRENT RAMP ENDED");
            } else {
                FCSLOG.error(name + " during action " + this.currentAction
                        + " waiting was cancelled before END of CURRENT RAMP " + ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.error(name + " CURRENT RAMP IS DURING TOO LONG - timeout=" + timeout, ex);

        } catch (InterruptedException ex) {
            FCSLOG.error(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);
            throw new FcsHardwareException(
                    name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);

        } catch (ExecutionException ex) {
            FCSLOG.error(name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction + " cause=" + ex);

        } finally {
            cancelCurrentRamp();
            FCSLOG.info(name + " STOP WAITING FOR END OF CURRENT RAMP");
        }
        ;
    }



    private void writeCurrentRampToLockAllClamps() {
        writeCurrentRampToAllClamps(
                onlineClampYminus.getFinalCurrentToClose(),
                onlineClampYminus.getCurrentToClamp(),
                onlineClampYminus.getIncrementCurrentToClamp(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getCurrentToClamp(),
                onlineClampXminus.getIncrementCurrentToClamp(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getCurrentToClamp(),
                onlineClampXplus.getIncrementCurrentToClamp(),
                this.maxTimeToLockAllClamps
        );
    }

    private void writeCurrentRampToUnlockAllClamps() {
        writeCurrentRampToAllClamps(
                onlineClampYminus.getCurrentToClamp(),
                onlineClampYminus.getFinalCurrentToClose(),
                onlineClampYminus.getIncrementCurrentToClamp(),
                onlineClampXminus.getCurrentToClamp(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getIncrementCurrentToClamp(),
                onlineClampXplus.getCurrentToClamp(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getIncrementCurrentToClamp(),
                this.maxTimeToUnlockAllClamps
        );
    }

    public void checkParametersToOpenClampsX() {
        int initialValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int finalValueXminus = onlineClampXminus.getCurrentToOpen();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToOpen();
        int initialValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int finalValueXplus = onlineClampXplus.getCurrentToOpen();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToOpen();

        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToOpenClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(
                    maxTimeToOpenClampsX + " : value too low for maxTimeToOpenClampsX. should be >= " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }

    public void checkParametersToCloseClampsX() {
        int initialValueXminus = onlineClampXminus.getInitialCurrentToClose();
        int finalValueXminus = onlineClampXminus.getFinalCurrentToClose();
        int incrementCurrentXminus = onlineClampXminus.getIncrementCurrentToClose();
        int initialValueXplus = onlineClampXplus.getInitialCurrentToClose();
        int finalValueXplus = onlineClampXplus.getFinalCurrentToClose();
        int incrementCurrentXplus = onlineClampXplus.getIncrementCurrentToClose();

        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = this.maxTimeToCloseClampsX / Math.min(nbStepXminus, nbStepXplus);
        if (period < minPeriod) {
            int newMaxTime = minPeriod * Math.min(nbStepXminus, nbStepXplus);
            throw new IllegalArgumentException(maxTimeToCloseClampsX + " : value too low. should be > " + newMaxTime);
        }
        if (nbStepXminus != nbStepXplus) {
            FCSLOG.finest(name + ": nbstep to open are not equal for the 2 clamps X");
        }
    }



    /**
     * Sends current alternativaly to the 3 clamps controllers with a ramp of
     * current Uses a sheduler.
     */
    private void writeCurrentRampToAllClamps(int initialValueYminus, int finalValueYminus, int incrementYminus,
            int initialValueXminus, int finalValueXminus, int incrementXminus, int initialValueXplus,
            int finalValueXplus, int incrementXplus, int maxTime) {
        int stepHeightYminus = FcsUtils.getSignedStepHeight(initialValueYminus, finalValueYminus, incrementYminus);
        int stepHeightXminus = FcsUtils.getSignedStepHeight(initialValueXminus, finalValueXminus, incrementXminus);
        int stepHeightXplus = FcsUtils.getSignedStepHeight(initialValueXplus, finalValueXplus, incrementXplus);
        final int nbStepYminus = Math.abs((finalValueYminus - initialValueYminus) / stepHeightYminus);
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / stepHeightXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / stepHeightXplus);
        final int period = maxTime / nbStepYminus;
        if (period < 200) {
            throw new IllegalArgumentException(period + " : bad value for period");
        }

        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampYminus=" + initialValueYminus);
        FCSLOG.finest("finalValue for clampYminus=" + finalValueYminus);
        FCSLOG.finest("initialValue for clampXminus=" + initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus=" + finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus=" + initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus=" + finalValueXplus);
        FCSLOG.finest("nbStepYminus=" + nbStepYminus + " nbStepXminus=" + nbStepXminus + " nbStepXplus=" + nbStepXplus);
        FCSLOG.finest("period=" + period);
        FCSLOG.finest("stepHeightYminus=" + stepHeightYminus);
        FCSLOG.finest("stepHeightXminus=" + stepHeightXminus);
        FCSLOG.finest("stepHeightXplus=" + stepHeightXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueYminus = initialValueYminus;
            private int newCurrentValueXminus = initialValueXminus;
            private int newCurrentValueXplus = initialValueXplus;
            boolean finalValueReachedYminus = false;
            boolean finalValueReachedXminus = false;
            boolean finalValueReachedXplus = false;

            @Override
            public void run() {
                newCurrentValueYminus = FcsUtils.computeNewCurrentValue(stepHeightYminus, newCurrentValueYminus,
                        finalValueYminus);
                newCurrentValueXminus = FcsUtils.computeNewCurrentValue(stepHeightXminus, newCurrentValueXminus,
                        finalValueXminus);
                newCurrentValueXplus = FcsUtils.computeNewCurrentValue(stepHeightXplus, newCurrentValueXplus,
                        finalValueXplus);
                try {
                    if (!finalValueReachedYminus) {
                        onlineClampYminus.sendCurrentToControllerAndSaveValue(newCurrentValueYminus);
                    }
                    if (!finalValueReachedXminus) {
                        onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    }
                    if (!finalValueReachedXplus) {
                        onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    }

                } finally {
                    finalValueReachedYminus = newCurrentValueYminus == finalValueYminus;
                    finalValueReachedXminus = newCurrentValueXminus == finalValueXminus;
                    finalValueReachedXplus = newCurrentValueXplus == finalValueXplus;
                }

                if (finalValueReachedYminus && finalValueReachedXminus && finalValueReachedXplus) {
                    cancelCurrentRamp();
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }

    /**
     * initialize AC online clamps hardware after initialization. to be executed
     * if during boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Initialize AC online clamps hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        this.postStart();
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.postStart();
        }
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (onlineStrainGauge.isBooted()) {
            onlineStrainGauge.initializeAndCheckHardware();
        } else {
            onlineStrainGauge.raiseAlarmIfMissing();
        }
        if (tempSensorsDevice.isBooted()) {
            computeLockStatusFromStrain();
            lockStatusInitialized = true;
            this.publishData();
        } else {
            tempSensorsDevice.raiseAlarmIfMissing();
        }
        FCSLOG.fine(name + " END postStart.");
    }

}
