package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.CarouselClamp;
import org.lsst.ccs.subsystems.fcs.Carousel;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * To simulate a controller which can open carousel clamps.
 * @author virieux
 */
public class SimuCarouselClampController extends SimuEPOSController {

    @LookupField(strategy=Strategy.TREE, pathFilter = ".*carousel")
    protected Carousel carousel;

    @LookupField(strategy=Strategy.TREE, pathFilter = ".*hyttc580")
    private SimuTTC580 hyttc580;

    /**
     * Build a new SimuCarouselClampController
     * @param nodeID
     * @param serialNB
     */
    public SimuCarouselClampController(
            int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    @Override
    public void init() {
        this.mode = CURRENT;
    }

    /**
     * Write current to the simulated controller.
     * @param currentValue
     * @throws SDORequestException
     */
    @Override
    public void writeCurrent(int currentValue) {

        FCSLOG.debug(getName() + "==========> ACTUATOR ON");

        if (this.carousel.getSocketAtStandby() == null) {
            throw new RejectedCommandException(getName() + " can't writeCurrent to clamp controller "
                    + "when there is no socket at standby position");
        }

        //the clamp is now UNLOCKED : we have to simulate that.
        CarouselClamp clamp;
        if (name.contains("Xminus")) {
            clamp = this.carousel.getSocketAtStandby().getClampXminus();
        } else if (name.contains("Xplus")) {
            clamp = this.carousel.getSocketAtStandby().getClampXplus();
        } else {
            throw new RejectedCommandException(name + ":couldn't find the clamp to unlock");
        }
        boolean atStandby = true;
        hyttc580.simulateClampIsUnlocked(atStandby, clamp);
        carousel.updateStateWithSensors();
        //we have to publish on status bus the sentValue (for the GUI)
        //not done in CanOPenEPOS
        this.publishData();
    }

    @Override
    public void stopAction()  {
        FCSLOG.debug(name + "==========> ACTUATOR OFF");
        //the clamp is now LOCKED : we have to simulate that.
        CarouselClamp clamp;
        if (name.contains("Xminus")) {
            clamp = this.carousel.getClampXminus();
        } else if (name.contains("Xplus")) {
            clamp = this.carousel.getClampXplus();
        } else {
            throw new FailedCommandException(name + ": unable to find the clamp to unlock.");
        }
        clamp.updateState();
        this.publishData();
    }
}
