package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 *
 * This is the driver class for the ADC we use in the FCS test bench at APC and
 * in SINGLE FILTER TEST CAN-CBX-AI814.
 *
 * @author virieux
 */
public class CanCBXAI814 extends CanOpenDevice implements ADCInterface {

    /**
     * One bit read on the input of the ADC represents 0.3125 mV
     */
    public static final double adcVoltsPerBit = 0.3125;
    public static final double waterFreezeTemp = 273.15;

    /**
     * The transmission types for CAN-CBX-AI814 are described in doc
     * CAN-CBX-AI814_Manual.pdf § 8.6.2
     * - acyclic, synchronous: The transmission is initiated if a SYNC-message has
     * been received (PDO- transmission type 0) and data has changed.
     * - cyclic, synchronous: The transmission is initiated if a defined number of
     * SYNC-messages have been received (PDO-transmission type 1...240).
     * - synchronous, remote request: The state of the inputs is latched
     * with each SYNC-message and is transmitted after the reception of a RTR-frame
     * (PDO-transmission type 252).
     * - asynchronous, remote request: After the reception of a RTR-frame the last
     * latched state of the inputs is transmitted (PDO-transmission type 253).
     * - event controlled, asynchronous: The transmission is initiated if the state
     * of selected inputs has changed (PDO-transmission type 254, 255).
     *
     */
    @ConfigurationParameter(description = "UNDEFINED", units = "unitless")
    protected volatile int transmissionType;

    public int[] inputs = new int[8];

    protected int cobid1 = nodeID + 0x280;
    protected int cobid2 = nodeID + 0x380;

    /**
     * Build a CanOpenADC with a CANopen node ID, a serial number and a transmission
     * type.
     *
     * @param nodeID
     * @param serialNB
     * @param transmissionType
     */
    public CanCBXAI814(int nodeID, String serialNB, int transmissionType) {
        super(nodeID, serialNB);
        this.transmissionType = transmissionType;
    }

    /**
     * @return
     */
    public int[] getInputs() {
        return inputs;
    }

    @Override
    public void doInitializePDOs() throws DriverException {
        tcpProxy.addReceivedPDO(cobid1);
        tcpProxy.addReceivedPDO(cobid2);
        FCSLOG.info(name + " pdos have been initialized with cobid1 = " + Integer.toHexString(cobid1) + " and cobid2 = "
                + Integer.toHexString(cobid2));
    }

    @Override
    public void initializeAndCheckHardware() {
        for (int i = 0; i < 8; i++) {
            inputs[i] = 0;
        }
        if (isBooted()) {
            this.writeTransmissionTypeToDevice(this.transmissionType);
            this.initialized = true;
            publishData();
        }
    }

    /**
     * Read the analog input whom number is given as a parameter
     *
     * @param inputNB
     * @return the value read
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     */
    @Override
    public int readAnalogInput(int inputNB) {

        if (!tcpProxy.isReady()) {
            throw new HardwareNotDetectedException(name + ": hardware not connected.");
        }

        if (isBooted()) {
            if ((inputNB < 1) || (inputNB > 8)) {
                throw new IllegalArgumentException("CAN-CBX-AI814 : input has to be 1..8");
            }
            return (int) readSDO(0x6401, inputNB);

        } else {
            String message = String.format("Module %s / nodeID=%s is UNREACHABLE", getName(), nodeID);
            FCSLOG.error(message);

            throw new HardwareNotDetectedException(name + " MISSING", name, this.nodeID, this.serialNB);
        }
    }

    /**
     * Read the voltage at the ADC input inputNB on the ADC and returns the value in
     * Volt
     *
     * @param inputNB : integer from 1 to 8
     * @return the value read (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     */
    @Override
    public double readVoltage(int inputNB) {
        return readAnalogInput(inputNB) * CanCBXAI814.adcVoltsPerBit;
    }

    /**
     * process PDOData to extract the values this device has sent.
     *
     * @param pdo
     */
    @Override
    public void updateFromPDO(PDOData pdo) {
        boolean updated = false;
        if (isBooted()) {
            if (pdo.getPDOs().containsKey(cobid1)) {
                FCSLOG.finest(name + " updating from pdo with cobid = 0x" + Integer.toHexString(cobid1));
                long pdoVal = pdo.getPDOs().get(cobid1);
                inputs[3] = (int) (pdoVal >> 48);
                inputs[2] = (int) (pdoVal >> 32 & 0xFFFF);
                inputs[1] = (int) (pdoVal >> 16 & 0xFFFF);
                inputs[0] = (int) (pdoVal & 0xFFFF);
                updated = true;
            }
            if (pdo.getPDOs().containsKey(cobid2)) {
                FCSLOG.info(name + " updating from pdo with cobid = 0x" + Integer.toHexString(cobid2));
                long pdoVal = pdo.getPDOs().get(cobid2);
                inputs[7] = (int) (pdoVal >> 48);
                inputs[6] = (int) (pdoVal >> 32 & 0xFFFF);
                inputs[5] = (int) (pdoVal >> 16 & 0xFFFF);
                inputs[4] = (int) (pdoVal & 0xFFFF);
                updated = true;
            }
            if (updated) {
                this.publishData();
            }
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print input values. For test and debug.")
    public String printInputs() {
        StringBuilder sb = new StringBuilder(super.toString());
        for (int i = 0; i < 8; i++) {
            sb.append("\n inputs[");
            sb.append(i);
            sb.append("]=");
            sb.append(inputs[i]);
        }
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description = "Return the string representation of CanCBXAI814.", alias = "print_cobid")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("cobid1=");
        sb.append(cobid1);
        sb.append("(hexa=");
        sb.append(Integer.toHexString(cobid1));
        sb.append(")/cobid2=");
        sb.append(cobid2);
        sb.append("(hexa=");
        sb.append(Integer.toHexString(cobid2));
        sb.append(")");
        return sb.toString();
    }

    /**
     * lifecycle commands
     */
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        this.initializeAndCheckHardware();
        FCSLOG.fine(name + " END postStart.");
    }
}
