
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * Represents a model for a force sensor.
 * A force sensor is used in loader subsystem to check if the clamp is CLAMPED or not.
 * It's connected to the gateway.
 * The value returned by the force sensor is coded on 1 byte.
 * The analog value is used mainly to be displayed on the GUI.
 * The gateway gives also a status to say if the clamp is CLAMPED or not.
 *
 * @author virieux
 */
public class ForceSensor implements SensorPluggedOnDevice {

    @LookupName
    private String name = "unset";

    /**
     * voltage value read on the gateway.
     * format = decimal
     * unit = mV (millivolts)
     */
    private int voltage;

    /**
     * name of the gateway this sensor is plugged on.
     */
    @ConfigurationParameter
    protected volatile String deviceName;

    /**
     * byte number on the gateway
     */
    @ConfigurationParameter
    protected volatile int byteNumero;


    /* from interface SensorPluggedOnDevice */
    @Override
    public String getName() {
        return name;
    }

    /**
     * return byteNumero
     *
     * @return
     */
    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    @Override
    public String getDeviceName() {
        return deviceName;
    }




    @Override
    public void updateValue(int newValue) {
        FCSLOG.info(name + " updating voltage = " + newValue);
        /*
         * voltage sent by sensor is given in 0.1V so it has to be multiplied by 100 to
         * be expressed in mV
         */
        voltage = newValue;
    }
    /* end from interface SensorPluggedOnDevice */

    /**
     * Returns voltage value read on force sensor in mV.
     *
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Returns force sensor analog Value. No unit.")
    public int getVoltage() {
        return voltage;
    }



    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
             description="Returns a string representation of ForceSensor")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/deviceName=");
        sb.append(this.deviceName);
        sb.append("/byteNumero=");
        sb.append(String.valueOf(this.byteNumero));
        sb.append("/voltage=");
        sb.append(String.valueOf(voltage));
        return sb.toString();
    }
}
