package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput4;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.LOCK_ONLINECLAMP;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMP_MODE_PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.REOPEN_ONLINECLAMP_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_ONLINECLAMP;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForOnlineClamp;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for online clamp which holds a filter when it is at ONLINE
 * position. The online clamps are part of the autochanger subsystem. There is 3
 * online clamps.
 *
 * @author virieux
 */
public class AutochangerOnlineClamp extends MobileItem implements MovedByEPOSController, ControlledBySensors {

    public static final String ONLINE_CLAMP_CONFIG_CATEGORY = "onlineClampCurrent";
    public static final String CURRENT_MONITOR_TASK_NAME = "-monitorCurrent";

    // TODO these have to be configurable parameters because behavior of Y is different from X's
    public static final int TIMEOUT_FOR_CLOSING = 6000;
    public static final int TIMEOUT_FOR_OPENING = 6000;

    @LookupField(strategy = Strategy.TREE)
    private Autochanger autochanger;

    private final EPOSControllerForOnlineClamp controller;

    /* couple of closeClampInCurrentMode sensors */
    private final ComplementarySensors closeSensors;

    /* couple of openClampInCurrentMode sensors */
    private final ComplementarySensors openSensors;

    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = Strategy.TREE)
    private PersistencyService persistenceService;

    /**
     * currentToClamp is different for each clamp so the value is initialized from
     * configuration system.
     */
    @ConfigurationParameter(description = "current to clamp ONLINE clamp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int currentToClamp = 0;

    @ConfigurationParameter(description = "current to open ONLINE clamp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int currentToOpen = 0;

    /**
     * sentCurrent represents the last current value sent to controller we want to
     * save this value of current between two instances of running FCS and each time
     * we sent current to the controller.
     *
     */
    @Persist
    private int sentCurrent = 0;

    /**
     * current read on the controller
     */
    private int currentReadOnController;

    /**
     * position read on controller
     */
    private int position;

    /**
     * increment of current in a current ramp, in mA
     */
    @ConfigurationParameter(description = "increment of current to lock in a current ramp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int incrementCurrentToClamp = 250;

    /**
     * increment of current to closeClampInCurrentMode a clamp in a current ramp, in
     * mA
     */
    @ConfigurationParameter(description = "increment of current to close clamp in a current ramp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int incrementCurrentToClose = 250;

    /**
     * increment of current to openClampInCurrentMode a clamp in a current ramp, in
     * mA
     */
    @ConfigurationParameter(description = "increment of current to open clamp in a current ramp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int incrementCurrentToOpen = 250;

    /**
     * initial current to close a clamp in CURRENT mode in a * current ramp, in mA
     */
    @ConfigurationParameter(description = "initial current to close clamp in a current ramp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int initialCurrentToClose = -currentToOpen;

    /**
     * final current to close a clamp In Current Mode a clamp in a * current ramp,
     * in mA
     */
    @ConfigurationParameter(description = "final current to close clamp in a current ramp", category = ONLINE_CLAMP_CONFIG_CATEGORY, units = "mA")
    private volatile int finalCurrentToClose = -currentToOpen * 2;


    /**
     * For the onlineClamp the lockStatus is OPENED, CLOSED, or LOCKED. Sensors say
     * if clamp is CLOSED or OPENED. To know if a clamp is LOCKED, we must know the
     * value of the current that was sent to the controller to
     * closeClampInCurrentMode the clamp. (field sentCurrent).
     */
    private FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    private volatile boolean initialized;
    // During tests we don't change configuration of clamps controller.
    private boolean controllerConfigured = true;

    /* Tools for current ramp */
    private ScheduledFuture<?> currentRampHandle;

    /**
     * ******************************************** parameters for FINAL PRODUCTS
     * only *********************************************
     */
    /**
     * position when clamps is opened
     */
    @ConfigurationParameter(description = "absolute target position to open clamp")
    private volatile int targetPositionToOpen;

    /**
     * position when clamps is closed
     */
    @ConfigurationParameter(description = "absolute target position to close clamp")
    private volatile int targetPositionToClose;

    @ConfigurationParameter(description = "delta of current between target current "
            + "and current read on controller. Used to know if action is completed for homing.")
    private volatile int deltaCurrent = 50;

    /**
     * ******************************************** end of parameters for FINAL
     * PRODUCTS only *********************************************
     */
    /**
     * Builds an AutochangerOnlineClampModule with a controller, 4 sensors and 2
     * maps of parameters for the controller.
     *
     * @param controller
     * @param closeSensors
     * @param openSensors
     */
    public AutochangerOnlineClamp(EPOSControllerForOnlineClamp controller, ComplementarySensors closeSensors,
            ComplementarySensors openSensors) {
        this.controller = controller;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
    }

    /* for simulation and tests */
    @ConfigurationParameterChanger
    public void setCurrentToClamp(int currentToClamp) {
        this.currentToClamp = currentToClamp;
    }

    /* for simulation & tests */
    @ConfigurationParameterChanger
    public void setCurrentToOpen(int currentToOpen) {
        this.currentToOpen = currentToOpen;
    }

    @ConfigurationParameterChanger
    public void setInitialCurrentToClose(int initialCurrentToClose) {
        this.initialCurrentToClose = initialCurrentToClose;
    }

    @ConfigurationParameterChanger
    public void setFinalCurrentToClose(int finalCurrentToClose) {
        this.finalCurrentToClose = finalCurrentToClose;
    }

    /**
     *
     * @param sentCurrent
     */
    public void setSentCurrent(int sentCurrent) {
        this.sentCurrent = sentCurrent;
    }

    /* for simulation and tests */
    public void setTargetPositionToOpen(int targetPositionToOpen) {
        this.targetPositionToOpen = targetPositionToOpen;
    }

    /* for simulation and tests */
    public void setTargetPositionToClose(int targetPositionToClose) {
        this.targetPositionToClose = targetPositionToClose;
    }


    /**
     *
     * @param stepHeightAbsValue
     */
    @ConfigurationParameterChanger
    public void setIncrementCurrentToClamp(int stepHeightAbsValue) {
        FcsUtils.checkPositive(stepHeightAbsValue);
        this.incrementCurrentToClamp = stepHeightAbsValue;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToOpen(int incrementCurrentToOpen) {
        FcsUtils.checkPositive(incrementCurrentToOpen);
        this.incrementCurrentToOpen = incrementCurrentToOpen;
    }

    @ConfigurationParameterChanger
    public void setIncrementCurrentToClose(int incrementCurrentToClose) {
        FcsUtils.checkPositive(incrementCurrentToClose);
        this.incrementCurrentToClose = incrementCurrentToClose;
    }

    public int getInitialCurrentToClose() {
        return initialCurrentToClose;
    }

    public int getFinalCurrentToClose() {
        return finalCurrentToClose;
    }

    public int getIncrementCurrentToOpen() {
        return incrementCurrentToOpen;
    }

    public int getIncrementCurrentToClamp() {
        return incrementCurrentToClamp;
    }

    public int getIncrementCurrentToClose() {
        return incrementCurrentToClose;
    }

    /**
     * returns last value of current sent to controller
     *
     * @return
     */
    public int getSentCurrent() {
        return sentCurrent;
    }

    /**
     * Return ONLINE clamp controller
     *
     * @return
     */
    @Override
    public EPOSControllerWithBrake getController() {
        return controller;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * For simulator
     *
     * @return
     */
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    /**
     * returns closeSensors for tests
     *
     * @return
     */
    public ComplementarySensors getCloseSensors() {
        return closeSensors;
    }

    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }

    /**
     * returns openSensors for tests
     *
     * @return
     */
    public ComplementarySensors getOpenSensors() {
        return openSensors;
    }

    /**
     * Returns true if closeSensor and closeSensorC return the same value. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if lockSensor and lockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isCloseSensorsInError() {
        return closeSensors.isInError();
    }

    /**
     * Returns true if openSensor and openSensorC retun the same value. Doesn't read
     * again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if unlockSensor and unlockSensorC retun the same value. "
            + "Doesn't read again sensors.")
    public boolean isOpenSensorsInError() {
        return openSensors.isInError();
    }

    /**
     * Returns true if LockStatus=ERROR, this means closeSensor or openSensor is in
     * ERROR or openSensor and closeSensor return non consistant values. Doesn't
     * read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if in ERROR which means that "
            + "closeSensor or openSensor is in ERROR or that"
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return lockStatus == ERROR;
    }

    /**
     * Returns true if LockStatus=CLOSED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if CLOSED. Doesn't read again sensors.")
    public boolean isClosed() {
        return lockStatus == CLOSED;
    }

    /**
     *
     * @return true if homing of controller has been done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if homing of controller has been done.")
    public boolean isHomingDone() {
        return controller.isHomingDone();
    }

    /**
     * Returns true if LockStatus=LOCKED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if LOCKED. Doesn't read again sensors.")
    public boolean isLocked() {
        return lockStatus == LOCKED;
    }

    /**
     * Returns true if LockStatus=OPENED. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if OPENED. Doesn't read again sensors.")
    public boolean isOpened() {
        return lockStatus == OPENED;
    }

    /**
     * Returns true if LockStatus=INTRAVEL. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if INTRAVEL. Doesn't read again sensors.")
    public boolean isInTravel() {
        return lockStatus == INTRAVEL;
    }

    /**
     * Returns true if clamp is initialized : hardware is ready (at least booted)
     * and clamp controller is initialized (parameters in the controller CPU have
     * been opened and controller has been configured).
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if clamp is initialized : controller is booted and parameters in the controller "
            + "CPU have been opened and controller is configured.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        persistenceService.setAutomatic(true, false);
    }

    /**
     *
     */
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerOnlineClamp.class, name);

        /**
         * Monitor current read on controller.
         */
//        periodicTaskService
//                .scheduleAgentPeriodicTask(new AgentPeriodicTask(name + CURRENT_MONITOR_TASK_NAME, this::monitorCurrent)
//                        .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofSeconds(60)));

    }

    /**
     * *** end of lifecycle methods
     * *************************************************
     */
    /**
     * Return true when ONLINE clamp hardware is ready. That is when controller is
     * initialized and configured.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if controller is initialized and configured.")
    @Override
    public boolean myDevicesReady() {
        return controller.isInitialized() && controllerConfigured;
    }

    /**
     * Check if it's safe to openClampInCurrentMode the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForOpening() {
        autochanger.checkConditionsForActioningOnlineClamps();

        if (!autochanger.isHoldingFilter()) {
            throw new RejectedCommandException(name + " can't be OPENED if autochanger is not HOLDING filter.");
        }
    }

    /**
     * Check if it's safe to closeClampInCurrentMode the clamp.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the onlineClamp can be locked.")
    public void checkConditionsForClosing() {
        autochanger.checkConditionsForActioningOnlineClamps();
    }

    /**
     * This method is called during INITIALIZATION phase. It configures the
     * controller for the digital inputs and outputs. It changes it to CURRENT mode
     * if it's not in this mode and it writes in the CPU of the controller the
     * values of configuration for CURRENT mode parameters.
     */
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (controller.isBooted()) {
            initializeController();
        } else {
            FCSLOG.error(name + " could not initialize controller in postStart. Do it by hand.");
        }
        FCSLOG.fine(name + " END postStart.");
    }

    /**
     * check that controllers are correctly configured.
     */
    private void initializeController() {
        try {
            /*
             * check that parameters on CPU are those on configuration
             */
            controller.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", controller.getName(), ex);
        }
    }

    /**
     * Enables controller, change mode depending on action and checkFault.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read in the CPU of the controller parameters for mode CURRENT."
            + "If a parameter has a different value than in configuration " + "throws an exception.")
    public void enableAndCheckControllerBeforeAction(MobileItemAction action) {
        controller.enable();
        if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION || action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION
                || action == OPEN_ONLINECLAMPSX || action == CLOSE_ONLINECLAMPSX) {
            controller.changeMode(PROFILE_POSITION);
        } else {
            controller.changeMode(CURRENT);
        }
        controller.checkFault();
    }

    public void checkControllerWhenGoingFromEngineeringModeToNormal() {
        controller.checkParameters(CURRENT);
        // TODO to be done in checkPArameters
        if (!controller.isParametersOK()) {
            String msg = name + " Some parameter values are not" + " the same in CPU and configuration system.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Configure ONLINE clamp controller.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Configure controller.")
    public void configureController() {
        try {
            controllerConfigured = false;
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (SDORequestException ex) {
            FCSLOG.warning(name, ex);
        }
    }

    /**
     * read actual current on controller, update field currentReadOnController and
     * publish it on STATUS bus.
     */
    @Override
    public void updateCurrent() {
        try {
            this.currentReadOnController = controller.readCurrent();
            this.publishData();

            /*
             * we don't want to have an ALARM when monitor-current could not read
             * currentReadOnController.
             */
        } catch (Exception ex) {
            raiseWarning(SDO_ERROR, " could not updateCurrent", controller.getName(), ex);
        }
    }

    public void updatePosition() {
        try {
            this.position = controller.readPosition();
            this.publishData();

        } catch (Exception ex) {
            raiseWarning(SDO_ERROR, " could not updatePosition", controller.getName(), ex);
        }
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in
     * error.
     *
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update state from sensors values.")
    public void updateState() {
        boolean closed = this.closeSensors.isOn();
        boolean opened = this.openSensors.isOn();
        boolean someSensorsInError = closeSensors.isInError() || openSensors.isInError();
        boolean inError = someSensorsInError || (closed && opened);

        if (inError) {
            lockStatus = ERROR;
        } else if (closed) {
            lockStatus = CLOSED;
        } else if (opened) {
            lockStatus = OPENED;
        } else {
            lockStatus = INTRAVEL;
        }
        /*
         * if the clamp is closed and a currentToClamp has been previously sent to
         * controller then clamp is LOCKED
         */
        if (closed && this.sentCurrent == this.currentToClamp) {
            lockStatus = LOCKED;
        }
        updatePosition();
        this.publishData();
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
        position = controller.readPosition();
        currentReadOnController = controller.readCurrent();
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (action == UNLOCK_ONLINECLAMP) {
            return sentCurrent == finalCurrentToClose;

        } else if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
            FCSLOG.info(name + "isClosed()=" + isClosed() + " position=" + position);
            return isClosed() && isAtPosition(targetPositionToClose);

        } else if (action == REOPEN_ONLINECLAMP_MODE_CURRENT
                || action == OPEN_ONLINECLAMP_MODE_CURRENT) {
            /* for homing*/
            FCSLOG.info(name + "isOpened()=" + isOpened() + " current=" + currentReadOnController);
            return isOpened() && isCurrentReached(currentToOpen);

        } else if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION) {
            FCSLOG.info(name + "isOpened()=" + isOpened() + " position=" + position);
            return isOpened() && isAtPosition(targetPositionToOpen);

        } else if (action == LOCK_ONLINECLAMP) {
            return isLocked();

        } else {
            throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
        }
    }

    private boolean isCurrentReached(int currentToReach) {
        FCSLOG.debug(name + " readCurrent=" + currentReadOnController + " currentToReached=" + currentToReach);
        return Math.abs(currentReadOnController - currentToReach) <= deltaCurrent;
    }

    /**
     * return true if clamp position is around targetPositionToClose
     *
     * @return
     */
    public boolean isAtPositionClose() {
        return isAtPosition(targetPositionToClose);
    }

    /**
     * return true if clamp position is around targetPositionToOpen
     *
     * @return
     */
    public boolean isAtPositionOpen() {
        return isAtPosition(targetPositionToOpen);
    }

    /**
     * return true if position of online clamp is around pos given as argument
     *
     * @param pos
     * @return
     */
    public boolean isAtPosition(int pos) {
        return Math.abs(position - pos) < 500;
    }

    /**
     * Closes the ONLINE clamp. For Proto only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Closes the ONLINE clamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void closeClampInCurrentMode() {
        updateStateAndCheckSensors();
        if (this.isOpened() || isInTravel()) {
            checkConditionsForClosing();
            this.executeAction(CLOSE_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_CLOSING);
            checkCloseSensors();
            updateStateAndCheckSensors();

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a close action.");
        }
    }


    /**
     * Closes the ONLINE clamp in mode PROFILE_POSITION.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Closes the ONLINE clamp in mode PROFILE_POSITION.", timeout = TIMEOUT_FOR_CLOSING)
    public void close() {
        if (!controller.isHomingDone()) {
            throw new RejectedCommandException(name + " homing not done yet. Can't close.");
        }
        updateStateAndCheckSensors();
        if (this.isOpened() || isInTravel()) {
            checkConditionsForClosing();
            this.executeAction(CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION, TIMEOUT_FOR_CLOSING);
            checkCloseSensors();
            updateStateAndCheckSensors();

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a close action.");
        }
    }

    private void checkCloseSensors() {
        if (isClosed()) {
            FCSLOG.info(name + " is CLOSED");
        } else {
            throw new FailedCommandException(name + " is not CLOSED after close or unlock command.");
        }
    }

    private void checkOpenSensors() {
        if (isOpened()) {
            FCSLOG.info(name + " is OPEN");
        } else {
            throw new FailedCommandException(name + " is not OPEN after open or openAndHoming command."
                    + " current action : " + currentAction.toString());
        }
    }

    /**
     * Opens the ONLINE clamp.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Opens the ONLINE clamp.", timeout = TIMEOUT_FOR_OPENING)
    public void openClampInCurrentMode() {
        updateStateAndCheckSensors();
        if (this.isClosed()) {
            checkConditionsForOpening();
            this.executeAction(OPEN_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_OPENING);

        } else if (this.isOpened() || this.isInTravel()) {
            FCSLOG.info(name + " is already OPENED. Send again currentToOpen.");
            this.executeAction(REOPEN_ONLINECLAMP_MODE_CURRENT, TIMEOUT_FOR_OPENING);

        } else {
            throw new RejectedCommandException(name + " has to be CLOSED before an open action.");
        }
        controller.checkFault();
        checkOpenSensors();
        updateStateAndCheckSensors();
    }

    /**
     * openClampInCurrentMode clamp in mode PROFILE_POSITION condition : homing must
     * have been done before, and latches must be closed. for FINAL products AC1 and
     * AC2.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Opens the ONLINE clamp in PROFILE_POSITION.", timeout = TIMEOUT_FOR_OPENING)
    public void open() {
        if (!controller.isHomingDone()) {
            throw new RejectedCommandException(name + " homing not done yet. Use openInCurrentModeAndHoming.");
        }
        updateStateAndCheckSensors();
        if (this.isClosed() || this.isInTravel()) {
            checkConditionsForOpening();
            this.executeAction(OPEN_ONLINECLAMP_MODE_PROFILE_POSITION, TIMEOUT_FOR_OPENING);
            checkOpenSensors();
            updateStateAndCheckSensors();

        } else if (this.isOpened()) {
            FCSLOG.info(name + " is already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be CLOSED before an open action.");
        }
    }

    /**
     * openClampInCurrentMode clamp in CURRENT mode and do homing.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "open clamp in CURRENT mode and do homing",
            timeout = 5000)
    public void openInCurrentModeAndHoming() {
        openClampInCurrentMode();
        controller.homing();
        updatePosition();
    }

    /**
     * Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp
     * of current from currentToClose to currentToRamp. The clamp has to be CLOSED *
     * before this action. At the end of this action, the clamp is LOCKED.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Locks the ONLINE clamp : sends currentToClamp to the controller with a ramp of current"
            + "from currentToClose to currentToRamp.", timeout = TIMEOUT_FOR_CLOSING)
    public void lock() {
        updateStateAndCheckSensors();
        if (this.isClosed()) {
            this.executeAction(LOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);
            updateStateAndCheckSensors();

        } else if (this.isLocked()) {
            FCSLOG.info(name + " is already LOCKED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be CLOSED before lock action.");
        }
    }

    /**
     * Unlocks the ONLINE clamp : sends currentToClamp to the controller with a ramp
     * of current from currentToClamp to currentToClose. The clamp has to be LOCKED
     * * before this action. At the end of this action, the clamp is CLOSED. The
     * goal if this action is to slow down the pressure exerted by the clamp on the
     * filter. If we decrease too quickly the pressure the clamp could be ejected
     * out by elastic return.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Unlocks the ONLINE clamp : sends currentToClamp to the controller "
            + " with a ramp of current from currentToClamp to currentToClose.", timeout = TIMEOUT_FOR_OPENING)
    public void unlock() {
        updateStateAndCheckSensors();
        // TODO check conditions for unlocking
        if (this.isLocked()) {
            this.executeAction(UNLOCK_ONLINECLAMP, TIMEOUT_FOR_CLOSING);
            checkCloseSensors();
            updateStateAndCheckSensors();

        } else if (this.isClosed()) {
            FCSLOG.info(name + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be LOCKED before unlock action.");
        }
    }

    /**
     * Start action of openClampInCurrentMode or closeClampInCurrentMode or lock or
     * unlock.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        this.enableAndCheckControllerBeforeAction(action);
        autochanger.checkConditionsForActioningOnlineClamps();
        //TODO uncomment when monitoring will be reactivated
//        autochanger.increaseCurrentMonitoring();
        autochanger.checkOnlineClampMotionAllowed();

        if (action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
            controller.changeMode(PROFILE_POSITION);
            controller.doReleaseBrake();
            controller.enable();
            controller.writeTargetPosition(targetPositionToClose);
            controller.writeControlWord(0x3F);

        } else if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION) {
            controller.enableAndReleaseBrake();
            controller.writeTargetPosition(targetPositionToOpen);
            controller.writeControlWord(0x3F);

        } else if (action == OPEN_ONLINECLAMP_MODE_CURRENT) {
            /* for homing when isClosed*/
            sendCurrentToControllerAndSaveValue(finalCurrentToClose);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            executeCurrentRamp(finalCurrentToClose, currentToOpen, incrementCurrentToOpen);

        } else if (action == REOPEN_ONLINECLAMP_MODE_CURRENT) {
            /* for homing when opened or inTravel*/
            controller.enableAndReleaseBrake();
            sendCurrentToControllerAndSaveValue(currentToOpen);

        } else if (action == LOCK_ONLINECLAMP) {
            sendCurrentToControllerAndSaveValue(finalCurrentToClose);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            /*
             * write a ramp from finalCurrentToClose to currentToClamp within maxTimeToLock
             */
            executeCurrentRamp(this.finalCurrentToClose, this.currentToClamp, incrementCurrentToClamp);

        } else if (action == UNLOCK_ONLINECLAMP) {
            /*
             * the clamp is LOCKED so we send currentToClamp before releasing brake to avoid
             * an elastic return of the clamp
             */
            sendCurrentToControllerAndSaveValue(this.currentToClamp);
            FcsUtils.sleep(500, name);
            controller.doReleaseBrake();
            /*
             * write a ramp from currentToClamp to finalCurrentToClose within
             * maxTimeToUnlock
             */
            executeCurrentRamp(this.currentToClamp, this.finalCurrentToClose, incrementCurrentToClamp);

        } else {
            throw new IllegalArgumentException(name + " invalid action for ONLINE clamp:" + action);
        }
    }

    /**
     * Writes current to controller and saves the * current value in the
     * ConfigurationParameter sentCurrent.
     *
     * The goal of saving the value is to be able to restart next time and know
     * which value of current has been sent to controller. To know this value of
     * current is important because the pressure exerted on the filter by the online
     * clamps depends on the value of this current. Higher is the current value,
     * higher is the pressure. And the process of opening or closing of a online
     * clamp depends on this previous sent current value.
     *
     * @param current to send to controller
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "sent current to control and save this value")
    public void sendCurrentToControllerAndSaveValue(int current) {
        FCSLOG.finest(name + " current to write to controller " + getControllerName() + ":" + current);
        controller.changeMode(CURRENT);
        controller.writeCurrent((short) current);
        this.sentCurrent = current;
        persistenceService.persistNow();
        this.publishData();
    }

    /**
     * What to do to abort an action.
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        if (action == OPEN_ONLINECLAMP_MODE_PROFILE_POSITION || action == CLOSE_ONLINECLAMP_MODE_PROFILE_POSITION) {
            controller.stopAction();
            controller.activateBrake();
        } else {
//            autochanger.decreaseCurrentMonitoring();
            controller.activateBrake();
            controller.writeCurrent((short) 0);
            controller.disableOperation();
        }
    }

    /**
     * Ending action OPEN / CLOSE / CLAMP / UNCLAMP
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());

//        autochanger.decreaseCurrentMonitoring();

        if (action == OPEN_ONLINECLAMP_MODE_CURRENT || action == REOPEN_ONLINECLAMP_MODE_CURRENT) {
            // in case of homing we have time to wait clamp is completely open
            FcsUtils.sleep(1000, name);
            controller.activateBrake();
            FcsUtils.sleep(20, name);
            controller.writeCurrent((short) 0);

        } else {
            controller.activateBrake();
        }
        FcsUtils.sleep(autochanger.getWaitTimeForBrakeOC(), name);
        controller.disableOperation();
        controller.disableVoltage();
        FCSLOG.debug(name + " END of action " + action.toString());
        controller.checkFault();
        updateState();
    }

    /**
     * This method is used to configure the controllers of the autochanger online
     * clamps. ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital Inputs
     * activated for OnlineClamps controllers are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ QuickStop Input=01 Mask=Enabled
     * Polarity=HighActive HE_Sensor General_A Input=02 Mask=Enabled
     * Polarity=HighActive HE_Sensor General_B Input=03 Mask=Enabled
     * Polarity=HighActive Check Brake Status
     * ******************************************** 1 - Configure the Digital Input
     * Functionnality index 0x2070 sindex input_ID Functionnality tab Value = 15
     * General Purpose A Value = 14 General Purpose B Value = 13 General Purpose C
     * Value = 12 General Purpose D Value = 11 General Purpose E Value = 10 General
     * Purpose F Value = 9 General Purpose G Value = 8 General Purpose H Value = 7
     * General Purpose I Value = 6 General Purpose J Value = 5 QuickStop Value = 4
     * Drive enable Value = 3 Position marker Value = 2 Home switch Value = 1
     * Positive Limit switch Value = 0 Negative Limit switch
     * *************************************************************** 2 - Configure
     * the Digital Input Polarity index 0x2071 subindex 0x03 Value = 0 High Active
     * Value = 1 Low Active
     * **************************************************************** 3 -
     * Configure the Digital Input Mask index 0x2071 subindex 0x02 Value = 0
     * Functionnality state will not be displayed Value = 1 Functionnality state
     * will be displayed Digital Input Functionnality tab value Bit_15 General
     * Purpose A 1 Bit_14 General Purpose B 1 Bit_13 General Purpose C 0 Bit_12
     * General Purpose D 0 Bit_11 General Purpose E 0 Bit_10 General Purpose F 0
     * Bit_9 General Purpose G 0 Bit_8 General Purpose H 0 Bit_7 General Purpose I 0
     * Bit_6 General Purpose J 0 Bit_5 QuickStop 1 Bit_4 Drive enable 0 Bit_3
     * Position marker 0 Bit_2 Home switch 0 Bit_1 Positive Limit switch 0 Bit_0
     * Negative Limit switch 0
     * ********************************************************************* 4 -
     * Configure the Digital Input Execution Mask index 0x2071 subindex 0x04 Digital
     * Input Excecution Mask Value Bit_15 to Bit_6 reserved 0 Bit_5 QuickStop 1
     * Bit_4 Drive enable 1 Bit_3 Position marker 0 Bit_2 Home switch 0 Bit_1
     * Positive Limit switch 0 Bit_0 Negative Limit switch 0
     * **********************************************************************
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps() {
        // 1-Configure Digital Input Fonctionnality
        // QuickStop (2070,01)
        controller.writeParameter(ConfigurationOfDigitalInput1, 0x5);

        // 2-Configure the Digital Input Polarity (2071,03)
        controller.writeParameter(DigitalInputFonctionnalityPolarity, 0);

        // 3-Configure the Digital Input Mask 8C20:1000110000100000 (2071,02)
        controller.writeParameter(DigitalInputFonctionnalityMask, 0x8C20);

        // 4 - Configure the Digital Input Execution Mask 30:0011 0000 (2071,04)
        controller.writeParameter(DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    /**
     * msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_2
     * General_A,wsdo,%s,2070,02,02,8000\n' % (str(cobID))
     * self.com_hal.executeOperation(client,msg) msg = 'Set DigIn_3
     * Genral_B,wsdo,%s,2070,02,02,4000\n' % (str(cobID))
     */
    /**
     * This method is used to configure the autochanger onlineClamps EPOS
     * controllers. Digital Outputs activated for OnlineClamps controller are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Ready/Fault
     * Output=01 Mask=Enabled Polarity=HighActive General_A Output=0 Mask=Enabled
     * Polarity=HighActive ******************************************* 1 - Configure
     * the Digital Output Functionnality index 0x2079 subindex input_ID
     * Functionnality tab : Value = 15 General Purpose Out_A Value = 14 General
     * Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General Purpose
     * Out_D Value = 11 General Purpose Out_E Value = 10..8 not used Value = 7..3
     * reserved Value = 2 Holding Brake Value = 1 Position compare Value = 0
     * Ready/Fault
     * ********************************************************************************
     * 2 - Configure the Digital Output Functionnality Mask index 0x2078 subindex
     * 0x02 Value = 0 functionnality not activatd Value = 1 functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General
     * Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7..3 reserved
     * 0 Bit_2 Holding Brake 0 (read only) Bit_1 Position Compare (read only) Bit_0
     * Ready/Fault (read only)
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value = 1
     * associated output inverted => LowActive
     * ********************************************************************************
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps() {
        // 1 - Configure the Digital Output Functionnality
        // Ready/Fault Active on Device Ready / Inactive on Fault
        controller.writeParameter(ConfigurationOfDigitalOutput1, 0);
        // General_A
        controller.writeParameter(ConfigurationOfDigitalOutput4, 0xF); // pas bon

        // 2 - Configure the Digital Output Functionnality Mask 8001=1000 0000 0000 0001
        controller.writeParameter(DigitalOutputFonctionnalityMask, 0x8001);

        // 3 - Configure the Digital Output Functionnality Polarity 8001=1000 0000 0000
        // 0001
        controller.writeParameter(DigitalOutputFonctionnalityPolarity, 0);
    }

    /**
     * Creates and returns the object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerOnlineClamp createStatusDataPublishedByOnlineClamp() {
        StatusDataPublishedByAutochangerOnlineClamp status = new StatusDataPublishedByAutochangerOnlineClamp();
        status.setLockSensorValue(closeSensors.isOn());
        status.setUnlockSensorValue(openSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorInError(closeSensors.isInError());
        status.setUnlockSensorInError(openSensors.isInError());
        status.setInError(lockStatus == ERROR);
        status.setControllerBooted(controller.isBooted());
        status.setControllerInFault(controller.isInError());
        status.setSentCurrent(sentCurrent);
        status.setCurrent(currentReadOnController);
        status.setPosition(position);
        status.setHomingDone(controller.isHomingDone());
        return status;
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByOnlineClamp()));
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        controller.quickStop();
    }

    /**
     * ********************************************************** Methods to write
     * current to the controller with a ramp
     * /************************************************************
     */
    /**
     * Cancel currentReadOnController ramp
     */
    private void cancelCurrentRamp() {
        // since this is called from within the runnable, a mayInterrupt=true seems a
        // bit risky.
        // no need to interrupt a new current application (during a sleep).
        this.currentRampHandle.cancel(false);
        FCSLOG.debug(" => current ramp ended");
    }

    /**
     * This method waits until the newCurrentValue ramp is completed.
     */
    private void waitForEndOfCurrentRamp(int timeout) {
        FCSLOG.info(name + " BEGIN WAITING FOR END OF CURRENT RAMP");
        try {
            currentRampHandle.get(timeout, TimeUnit.SECONDS); // will never return, only throw exception
        } catch (CancellationException ex) {
            if (currentRampHandle.isDone()) {
                FCSLOG.info(name + " CURRENT RAMP ENDED");
            } else {
                FCSLOG.error(name + " during action " + this.currentAction
                        + " waiting was cancelled before END of CURRENT RAMP " + ex);
            }
        } catch (TimeoutException ex) {
            FCSLOG.error(name + " CURRENT RAMP IS DURING TOO LONG - timeout=" + timeout, ex);
            cancelCurrentRamp();

        } catch (InterruptedException ex) {
            FCSLOG.error(name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);
            throw new FcsHardwareException(
                    name + " interrupted while waiting end of current ramp during action=" + currentAction, ex);

        } catch (ExecutionException ex) {
            FCSLOG.error(name + " error during action=" + currentAction, ex);
            throw new FcsHardwareException(name + " error during action=" + currentAction, ex);

        } finally {
            FCSLOG.info(name + " STOP WAITING FOR END OF CURRENT RAMP");
        }
    }

    /**
     * Sends currentReadOnController value to controller with a
     * currentReadOnController ramp from initialValue to finalValue. Uses a
     * scheduler to schedule the task.
     *
     * @param initialValue first value to send to controller
     * @param finalValue   last value to send to controller
     * @param nbStep       number of steps between initialValue to finalValue.
     */
    private void writeCurrentRamp(final int initialValue, final int finalValue, final int increment) {
        FCSLOG.debug("############################");
        FCSLOG.debug(name + "writeCurrentRamp");
        FCSLOG.debug("initialValue=" + initialValue);
        FCSLOG.debug("finalValue=" + finalValue);
        FCSLOG.debug("|increment|=" + increment);
        FCSLOG.debug("period=" + 250);
        FCSLOG.debug("waitTime=" + 250);
        FCSLOG.debug("############################");

        final Runnable currentRamp = new Runnable() {
            private final int adjustedStepHeight = FcsUtils.getSignedStepHeight(initialValue, finalValue, increment);
            private int currentValue = initialValue;

            public boolean finalValueReached() {
                boolean reachedUp = adjustedStepHeight > 0 && currentValue >= finalValue;
                boolean reachedDown = adjustedStepHeight < 0 && currentValue <= finalValue;
                return reachedUp || reachedDown;
            }

            @Override
            public void run() {
                // extra safety
                if (!finalValueReached()) {
                    currentValue += adjustedStepHeight;
                    FCSLOG.info("ramp write current " + currentValue);
                    sendCurrentToControllerAndSaveValue(currentValue);
                }
                if (finalValueReached()) {
                    cancelCurrentRamp();
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 250, 250, TimeUnit.MILLISECONDS);
    }

    /**
     * Sends currentReadOnController to controller with a ramp : from initialCurrent
     * to finalCurrent in a given number of steps (nbStep) and waits until the final
     * value is reached. It uses method writeCurrentRamp which uses a scheduler.
     *
     * @param initialCurrent   first value to be sent to controller
     * @param finalCurrent     last value to send to controller
     * @param incrementCurrent
     */
    public void executeCurrentRamp(int initialCurrent, int finalCurrent, int incrementCurrent) {
        writeCurrentRamp(initialCurrent, finalCurrent, incrementCurrent);
        waitForEndOfCurrentRamp(TIMEOUT_FOR_OPENING);
    }

    /**
     * ************************************************************** end of Methods
     * to write wurrent to the controller with a ramp
     * /****************************************************************
     */
}
