package org.lsst.ccs.subsystems.fcs.common;

import java.util.Map;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.HOMING;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_VELOCITY;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.VELOCITY;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux cf doc EPOS2_Firmware_Specification § 3.2 Device Control
 */
public interface EPOSController extends PieceOfHardware, HasLifecycle, ClearAlertHandler {

    /****************************************************************************************/
    /****************************************************************************************/
    /*
     * Methods to enable, shutdownController, shutdown or check status controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * *************************************************************************************
     */
    /*
     Statusword :
     bit 0: ready to switch on
     bit 1: switched on
     bit 2: operation enabled
     bit 3: fault
     bit 4: voltage enabled
     bit 5: quick stop
     bit 6: switch on disabled
     bit 7: warning
     bit 8: manufacturer-specific
     bit 9: remote
     bit 10: target reached
     bit 11: internal limit active
     bit 12-13: mode-specific
     bit 14-15: manufacturer-specific
     /****************************************************************************************/
    /**
     * *************************************************************************************
     */
    /**
     * For GUI. Doesn't read again controller CPU. Return true if controller is
 disabled.
     *
     * @return
     */
    boolean isEnabledToPublish();

    void setEnabledToPublish(boolean enabledToPublish);

    /**
     * Read StatusWord on controller CPU.
     *
     * @return
     */
    int readStatusWord();

    short readControlWord();

    /**
     * Write controlWord to the controller. Use to send or start commands. See EPOS
     * documentation.
     *
     * @param w
     * @throws FcsHardwareException
     */
    void writeControlWord(int w);

    /**
     * print status word bit by bit for end user and tests
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print status word in binary.")
    default String printStatusWordInBinary() {
        int sw = readStatusWord();
        StringBuilder sb = new StringBuilder("0b");
        for (int i = 0; i < 16; i++) {
            sb.append((sw >> 15 - i) & 1);
            if (i == 7) {
                sb.append(" ");
            }
        }
        return sb.toString();
    }

    /***************************************************************************
     * Commands to change CANopen state
     ***************************************************************************
     */

    /**
     * shutdowns controller cf doc EPOS2_Firmware_Specification § 3.2 Device Control
     * lower byte of control word: 0xxx x110
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Shutdown the controller.")
    default void shutdownController() {
        int cw = readControlWord();
        int mask = 0b1111111101111110;
        int cw2 = cw & mask;
        int mask2 = 0b00000110;
        writeControlWord(cw2 | mask2);
        setEnabledToPublish(false);
        FCSLOG.debug(getName() + ": is SHUTDOWN.");
        publishData();
    }

    /**
     * switch controller ON cf doc EPOS2_Firmware_Specification § 3.2 Device Control
     * lower byte of control word: 0xxx x111
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Switch ON the controller.")
    default void switchOn() {
        int cw = readControlWord();
        int mask = 0b1111111101111111;
        int cw2 = cw & mask;
        int mask2 = 0b00000111;
        writeControlWord(cw2 | mask2);
        FCSLOG.debug(getName() + ": is SWITCH ON.");
    }

    /**
     * cf doc EPOS2_Firmware_Specification § 3.2 Device Control lower byte of
     * control word: 0xxx xx0x
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Disable voltage. This command does not activate holding brake. "
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void disableVoltage() {
        int cw = readControlWord();
        int mask = 0b1111111101111101;
        writeControlWord(cw & mask);
        setEnabledToPublish(false);
        FCSLOG.info(getName() + " High-level power is switched off.");
        publishData();
    }

    /**
     * cf doc EPOS2_Firmware_Specification § 3.2 Device Control lower byte of
     * control word: 0xxx 0111
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Disable operation. This command activates holding brake. "
            + "cf doc EPOS2_Firmware_Specification § 3.2 Device Control")
    default void disableOperation() {
        int cw = readControlWord();
        int mask = 0b1111111101110111;
        int cw2 = cw & mask;
        int mask2 = 0b00000111;
        writeControlWord(cw2 | mask2);
        setEnabledToPublish(false);
        FCSLOG.info(getName() + " Drive function is disabled.");
        publishData();
    }

    /**
     * Switch on and enable controller. cf doc EPOS2_Firmware_Specification § 3.2
     * Device Control lower byte of control word: 0xxx1111
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Switch on and enable controller.")
    default void switchOnEnableOperation() {
        int cw = readControlWord();
        int mask = 0b1111111101111111;
        int cw2 = cw & mask;
        int mask2 = 0b00001111;
        writeControlWord(cw2 | mask2);
        setEnabledToPublish(true);
        FCSLOG.info(getName() + " Drive function is enabled.");
        publishData();
    }

    /***************************************************************************
     * End of Commands to change CANopen state
     ***************************************************************************
     */

    /**
     * Stop action and shutdownController(shutdown) controller.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Stop action and disable(shutdown) controller.")
    default void stopAction() {
        if (getMode() == HOMING) {
            this.writeControlWord(0x11F);

        } else if (getMode() == CURRENT) {
            /* stop current */
            writeCurrent((short) 0);

        } else if (getMode() == PROFILE_POSITION) {
            this.stopPosition();

        } else if (getMode() == PROFILE_VELOCITY) {
            this.stopVelocity();
        }
        disableOperation();
        FCSLOG.info(getName() + " ACTION STOPPED");
    }

    /**
     * Send a quickStop command to the controller.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "QuickStop.")
    default void quickStop() {
        FCSLOG.debug(getName() + " running QUICKSTOP command.");
        if (getMode().equals(HOMING) || getMode().equals(PROFILE_POSITION)
                || getMode().equals(VELOCITY))
            writeControlWord(0xB);
        else if (getMode().equals(CURRENT)) {
            writeControlWord(0x2);
        } else
            throw new IllegalArgumentException(getName() + " has invalid Epos mode:" + getMode());
    }

    /**
     * This methods enables the controller : i.e. this makes it able to receive
     * commands.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "This methods enables the controller : i.e. this makes it able to receive commands.")
    default void enable() {
        shutdownController();
        switchOnEnableOperation();
        checkEnabled();
        setEnabledToPublish(true);
        FCSLOG.debug(getName() + ": is ENABLED.");
        publishData();
    }

    /**
     * Check if controller is enable otherwise, throw an Exception.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Check if controller is enabled otherwise, throw an Exception.")
    default void checkEnabled() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean enabled = false;
        while (!enabled && duration <= timeout) {
            FCSLOG.info(getName() + " is not enabled yet. duration = " + duration);
            duration = System.currentTimeMillis() - timeStart;
            enabled = isEnabled();
        }
        if (!enabled) {
            FCSLOG.error(getName() + " NOT ENABLED" + " StatusWord=" + readStatusWord());
            checkFault();
            throw new FcsHardwareException(
                    getName() + " couldn't be enabled during time allocated of " + timeout + " ms");
        }
    }

    /**
     * Check if controller is disable otherwise, throw an Exception.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Check if controller is enabled otherwise, throw an Exception.")
    default void checkDisabled() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean disabled = false;
        while (!disabled && duration <= timeout) {
            FCSLOG.info(getName() + " is not enabled yet. duration = " + duration);
            duration = System.currentTimeMillis() - timeStart;
            disabled = !isDisabled();
        }
        if (!disabled) {
            FCSLOG.error(getName() + " NOT DISABLED" + " StatusWord=" + readStatusWord());
            checkFault();
            throw new FcsHardwareException(
                    getName() + " couldn't be disabled during time allocated of " + timeout + " ms");
        }
    }

    /**
     * Returns true if the controller is enabled.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if the controller is enabled.")
    // TODO check Firmware Specification p3-15
    default boolean isEnabled() {
        try {
            int statusWord = readStatusWord();
            int bit0 = statusWord & 1;
            int bit1 = (statusWord >> 1) & 1;
            int bit2 = (statusWord >> 2) & 1;
            int bit3 = (statusWord >> 3) & 1;
            FCSLOG.debug(getName() + ": statusWord = " + statusWord + " => bit0=" + bit0 + ",bit1=" + bit1 + ",bit2="
                    + bit2 + ",bit3=" + bit3);
            return (bit0 == 1 && bit1 == 1 && bit2 == 1 && bit3 == 0);

        } catch (SDORequestException ex) {
            String msg = getName() + " couldn't read status word because SDORequestException :" + ex.getMessage();
            FCSLOG.error(msg);
            FCSLOG.error(ex);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /**
     * Returns true if the controller is disabled.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if the controller is disabled.")
    // TODO check Firmware Specification p3-15
    default boolean isDisabled() {
        try {
            int statusWord = readStatusWord();
            int bit0 = statusWord & 1;
            int bit1 = (statusWord >> 1) & 1;
            int bit2 = (statusWord >> 2) & 1;
            int bit3 = (statusWord >> 3) & 1;
            FCSLOG.debug(getName() + ": statusWord = " + statusWord + " => bit0=" + bit0 + ",bit1=" + bit1 + ",bit2="
                    + bit2 + ",bit3=" + bit3);
            // TODO: find a better condition for checking if controller is disabled
            return (bit2 == 0);

        } catch (SDORequestException ex) {
            String msg = getName() + " couldn't read status word because SDORequestException :" + ex.getMessage();
            FCSLOG.error(msg);
            FCSLOG.error(ex);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /****************************************************************************************/
    /****************************************************************************************/
    /*
     * end of Methods to enable, shutdownController, shutdown controller /
     ****************************************************************************************/
    /****************************************************************************************/

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to read, write or check EPOS mode in the CPU of the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * This methods changes the mode to the new mode given as an argument. It writes
     * the new mode in the CPU of the EPOS controller and updates the field mode.
     *
     * @param newMode
     * @throws SDORequestException
     */
    void changeMode(EposMode newMode);

    /**
     * read the EPOS mode on the controller CPU.
     *
     * @return mode
     * @throws FcsHardwareException
     */
    EposMode readMode();

    /**
     * Return the EPOS mode stored in field mode. Doesn't read again controller CPU.
     *
     * @return
     */
    EposMode getMode();

    /**
     * return true if this controller is in the mode given as argument.
     *
     * @param aMode
     * @return
     */
    default boolean isInMode(EposMode aMode) {
        return aMode.equals(getMode());
    }

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * end of Methods to read, write or check EPOS mode in the CPU of the controller
     * /
     ****************************************************************************************/
    /****************************************************************************************/

    /**
     * Read actual current on the controller CPU. (object 6078)
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    int readCurrent();

    /**
     * Read Average current on controller CPU. (object 2027)
     *
     * @return
     */
    int readCurrentAverageValue();

    /**
     * read velocity on the controller CPU.
     *
     * @return
     */
    int readVelocity();

    /**
     * read Following Error on controller CPU.
     *
     * @return
     */
    int readFollowingError();

    /**
     * In current mode this methods send a current to the motor. This make run the
     * motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "In current mode this methods send a current to the motor.")
    void writeCurrent(int aValue) throws EPOSConfigurationException;

    /**
     * Enable, set mode CURRENT and write current. For end user at console.
     *
     * @param aValue
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Enables controller and sets the current value in the EPOS CPU.")
    default void enableAndWriteCurrent(int aValue) {
        enable();
        writeCurrent(aValue);
    }

    /**
     * Defines the actual position as the absolute position which value is given as
     * an argument. Writes value 35 to set the Homing method as Actual (See EPOS
     * documentation)
     *
     * @param position
     * @throws SDORequestException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Define the actual position as position given as argument.")
    default void defineAbsolutePosition(int position) {
        FCSLOG.debug(getName() + " Defining Absolute Position:" + position);
        int sleepTime = 50;
        this.changeMode(HOMING);
        // this.writeParameters(HOMING);
        this.writeParameter(EPOSEnumerations.Parameter.HomePosition, position);
        this.writeParameter(EPOSEnumerations.Parameter.HomingMethod, 35);
        /* this.enable() doesn't work for loader clamp homing so we go back to doc Application_Notes_Collection $8 */
        this.writeControlWord(6);
        FcsUtils.sleep(sleepTime, getName());
        this.writeControlWord(0xF);
        FcsUtils.sleep(sleepTime, getName());
        /* start homing */
        this.writeControlWord(0x1F);
        checkHomingDone();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Check if homing has been done otherwise throws an Exception.")
    default void checkHomingDone() {
        long timeout = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " homing target not yet reached. duration=" + duration);
            duration = System.currentTimeMillis() - timeStart;
        }
        if (isTargetReached()) {
            writeControlWord(0xF);
            FCSLOG.info(getName() + " ==> END homing");
        } else {
            throw new FcsHardwareException(getName() + " couldn't do homing : "
                    + "target is not reached within timeout of " + timeout + "milliseconds.");
        }
    }

    default void definePositionFromNegativeLimitSwitch() {
        FCSLOG.debug(getName() + " Homing with homing method negative limit switch");
        this.changeMode(HOMING);
        this.writeParameter(EPOSEnumerations.Parameter.HomePosition, 0);
        this.writeParameter(EPOSEnumerations.Parameter.HomingMethod, 17);
        this.enable();
        /* start homing */
        this.writeControlWord(0x1F);
    }

    /**
     * Reads in the CPU the value of the parameter PositionActualValue and returns
     * it in a decimal format.
     *
     * @return value in decimal format
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readPositionActualValue", description = "In PROFILE_POSITION Mode this methods returns the actual position.")
    default int readPosition() {
        return (int) this.readParameter(EPOSEnumerations.Parameter.PositionActualValue);
    }

    int getPosition();

    int getCurrent();

    int getVelocity();

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read the position returned by the absolute encoder (single serial data).")
    default int readSSIPosition() {
        return (int) this.readParameter(EPOSEnumerations.Parameter.SSIEncoderActualPosition);
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter ProfileVelocity.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "In PROFILE_POSITION returns the value of the parameter ProfileVelocity.")
    default long readProfileVelocity() {
        long profileVelocity = this.readParameter(EPOSEnumerations.Parameter.ProfileVelocity);
        FCSLOG.debug(getName() + ":readProfileVelocity=" + profileVelocity);
        return profileVelocity;
    }

    /**
     * Read parameter PositionSensorType in CPU of controller and return read value.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read PositionSensorType.")
    default int readPositionSensorType() {
        return (int) readParameter(EPOSEnumerations.Parameter.PositionSensorType);
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "In PROFILE_POSITION mode this methods set the target position.")
    void writeTargetPosition(int aValue);

    /**
     * set the target velocity.
     *
     * @param velocity
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "In PROFILE_VELOCITY mode this methods set the target velocity.")
    void writeTargetVelocity(int velocity);

    /**
     * change ProfileVelocity to a new value. Used by loader carrier controller to
     * reduce speed.
     *
     * @param newVelocity
     */
    default void changeProfileVelocity(int newVelocity) {
        FCSLOG.info(getName() + " changing ProfileVelocity to " + newVelocity);
        try {
            writeParameter(EPOSEnumerations.Parameter.ProfileVelocity, newVelocity);
        } catch (Exception ex) {
            String msg = getName() + " : could not change ProfileVelocity to value " + newVelocity;
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg, ex);
        }
    }

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to read or write EPOS parameters in the CPU of the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * This methods returns true if the values of parameters stored in CPU are the
     * same than those stored in the configuration system.
     *
     * @return
     */
    boolean isParametersOK();

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param parameter
     * @return
     */
    long readParameter(EPOSEnumerations.Parameter parameter);

    /**
     * Write on the CPU of the controller a value given in hexadecimal format for a
     * parameter.
     *
     * @param parameter
     * @param val       the value.
     * @throws FcsHardwareException
     */
    void writeParameter(EPOSEnumerations.Parameter parameter, int val);

    /**
     * Check parameters for a given EPOS mode : compare the values stored in the
     * Configuration System and the values stored in controller CPU. Raise a
     * FcsHardwareException if some values are not equals.
     *
     * @param aMode
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void checkParameters(EposMode aMode);

    /**
     * This methods writes in the CPU of the EPOS devis the values of the parameters
     * set for the mode. Usually the values of the parameters are given by the
     * configuration system.
     *
     * @param mode
     * @throws SDORequestException
     */
    void writeParameters(EposMode mode);

    /**
     * Write in the CPU of the EPOS device the values of the parameters set for the
     * mode in configuration system.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Write in the CPU of the EPOS device the values "
            + "of the parameters set for the mode.")
    default void writeParameters() {
        readMode();
        writeParameters(getMode());
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is
     * given as argument.
     *
     * @param parameterName name of the parameter to read
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Reads in the EPOS CPU the decimal value of the Parameter "
            + "which parameter name is given as argument.")
    default long readParameter(String parameterName) {
        EPOSEnumerations.Parameter param = EPOSEnumerations.Parameter.valueOf(parameterName);
        return readParameter(param);
    }

    /**
     * Read the parameters for the actual mode.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read the parameters for the actual mode.")
    default String readParameters() {
        return readParameters(readMode().toString());
    }

    /**
     * This methods reads in the CPU of the EPOS the values of the parameters for a
     * given mode.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "This methods reads in the CPU of the EPOS the values "
            + "of the parameters for a given mode.")
    default String readParameters(String modeInString) {
        return readParameters(EposMode.valueOf(modeInString));
    }

    /**
     * Read all the value of parameters for a given EPOS mode and return a printed
     * list of this values.
     *
     * @param aMode
     * @return
     * @throws FcsHardwareException
     */
    default String readParameters(EposMode aMode) {
        String modeIS = aMode.toString();
        StringBuilder sb = new StringBuilder(getName() + " parameters VALUES in decimal format for mode ");
        sb.append(modeIS);
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = aMode.getParameters();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.toString());
            sb.append("=");
            long valueInt = readParameter(param);
            sb.append(valueInt);
            sb.append("\n");
        }
        FCSLOG.debug(getName() + sb.toString());
        return sb.toString();
    }

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    default void writeParameters(Map<String, Integer> paramMap) {
        for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue();
            writeParameter(EPOSEnumerations.Parameter.valueOf(paramName), value);
        }
    }

    /**
     * values transmitted by EPOSController can be : UNSIGNED32, UNSIGNED16,
     * UNSIGNED8 for UNSIGNED parameter INTEGER32, INTEGER16, INTEGER8 for SIGNED
     * parameter This method convert a value sent by device to the value to be used
     * in fcs. Exemple : PolePairNumber is an UNSIGNED8. Real values are always > 0.
     * Value sent by device should not be changed and be always > 0 : 255 => 255
     *
     * HomingMethod is an INTEGER8 Real values can be negative. So value sent by
     * device is casted as byte to have the real value. 255 => -1
     *
     * @param param
     * @param value
     * @param myName
     * @return
     */
    static long convertEPOSValue(EPOSEnumerations.Parameter param, long value, String myName) {
        if (!param.isSigned()) {
            return value;
        } else if (param.getSize() == 4) {
            return (int) value;
        } else if (param.getSize() == 2) {
            return (short) value;
        } else if (param.getSize() == 1) {
            return (byte) value;
        } else {
            throw new EPOSConfigurationException(myName + ":\f" + param.getSize() + " => bad value for size");
        }
    }

    /****************************************************************************************/
    /****************************************************************************************/
    /*
     * END of Methods to read or write EPOS parameters in the CPU of the controller
     * /
     ****************************************************************************************/
    /****************************************************************************************/

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * Methods to check fault or reset fault in the controller /
     ****************************************************************************************/
    /****************************************************************************************/
    /**
     * Return true if controller is in fault.
     *
     * @return
     */
    boolean isInError();

    /**
     *
     * @return error register in printed format
     */
    String getErrorRegister();


    /**
     *
     * @return last code error occured on that device
     */
    int getLastErrorCode();

    /**
     *
     * @return last error name occured on that device
     */
    String getLastErrorName();

    /**
     * Return the number of errors that occured on this controller.
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    int getErrorHistoryNB();

    /**
     * Return a printed list of errors that occured on this controller.
     *
     * @return
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    String displayErrorHistory();

    /**
     * Check if the controller is in fault. Throw an Exception if the controller is
     * in fault.
     *
     * @throws SDORequestException
     * @throws FcsHardwareException
     */
    void checkFault();

    void faultReset();

    /****************************************************************************************/

    /****************************************************************************************/
    /*
     * End of Methods to check fault or reset fault in the controller /
     ****************************************************************************************/
    /****************************************************************************************/

    /**
     * In HOMING mode and PROFILE_POSITION mode this indicates that the position is
     * reached.
     *
     * @return true if the position is reached.
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "In HOMING mode and PROFILE_POSITION mode this indicates that the position is reached.")
    default boolean isTargetReached() {
        /* target is reached when bit10 of status word is 1 */
        return ((readStatusWord() >> 10) & 1) == 1;
    }

    /**
     * check if target is reached in HOMING or PROFILE_POSITION commands and throws
     * an Exception if not.
     *
     * @param timeout
     */
    default void checkTargetReached(long timeout) {
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        while (!isTargetReached() && duration <= timeout) {
            FCSLOG.info(getName() + " target is not yet reached.");
            duration = System.currentTimeMillis() - timeStart;
        }
        if (!isTargetReached()) {
            String msg = getName() + " couldn't reach target during time allocated of " + timeout + " ms";
            FCSLOG.info(msg);
            throw new FcsHardwareException(msg);
        }
    }

    /**
     * Enable controller and go to absolute position. Doesn't check condition.
     * DANGER !!!!
     *
     * @param pos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Enable controller and go to absolute position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteAbsolutePosition(int pos) {
        enable();
        writeTargetPosition(pos);
        writeControlWord(0x3F);
    }

    /**
     * Enable controller and go to relative position. Doesn't check condition.
     * DANGER !!!!
     *
     * @param pos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Enable controller and go to relative position. Doesn't check condition. DANGER !!!! ")
    default void enableAndWriteRelativePosition(int pos) {
        enable();
        FcsUtils.sleep(250, getName());
        writeTargetPosition(pos);
        writeControlWord(0x7F);
    }

    /**
     *
     * Stop motion when in mode PROFILE_POSITION.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Stop motion when in mode PROFILE_POSITION.")
    default void stopPosition() {
        if (isInMode(PROFILE_POSITION)) {
            this.writeControlWord(0x010F);
        } else {
            throw new RejectedCommandException(getName() + " is not in PROFILE_POSITION mode.");
        }
    }

    /**
     * Stop motion when in mode PROFILE_VELOCITY.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Stop motion when in mode PROFILE_VELOCITY.")
    default void stopVelocity() {
        if (isInMode(PROFILE_VELOCITY)) {
            this.writeControlWord(0x0107);
        } else {
            throw new RejectedCommandException(getName() + " is not in PROFILE_VELOCITY mode.");
        }
    }

    @Override
    default ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
    }

    /**
     * Creates an object to be published on the STATUS bus by an EPOSController.
     *
     * @return
     */
    default StatusDataPublishedByEPOSController createStatusDataPublishedByEPOSController() {
        StatusDataPublishedByEPOSController status = new StatusDataPublishedByEPOSController(
                isBooted(), isInitialized(), isInError(), getErrorRegister(),
                getErrorHistoryNB(), getLastErrorCode(), getLastErrorName());
        status.setEnabled(isEnabledToPublish());
        status.setMode(getMode());
        return status;
    }

}
