
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;

/**
 * This is the model for a single hook in the loader.
 * Each one is checked in CLOSED and OPENED position by 2 sensors.
 * These sensors have to give a consistent response otherwise the hook is in ERROR.
 * There is 4 hooks in the loader to hold a filter.
 *
 * @author virieux
 */
public class LoaderHook {

    @LookupName
    private String name;

    private final DigitalSensor closeSensor;
    private final DigitalSensor openSensor;

    private FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    private boolean inError;

    public LoaderHook(DigitalSensor closeSensor, DigitalSensor openSensor) {
        this.closeSensor = closeSensor;
        this.openSensor = openSensor;
    }

    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update state from sensors values.")
    public void updateState() {
        inError = this.closeSensor.isOn() && this.openSensor.isOn();

        if (closeSensor.isOn() && openSensor.isOn()) {
            lockStatus = ERROR;
        } else if (closeSensor.isOn()) {
            lockStatus = CLOSED;
        } else if (openSensor.isOn()) {
            lockStatus = OPENED;
        } else {
            lockStatus = INTRAVEL;
        }
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "display sensors info")
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("#closeSensor=");
        sb.append(this.closeSensor.toString());
        sb.append("#openSensor=");
        sb.append(this.openSensor.toString());
        return sb.toString();
    }

    /**
     * To create an object StatusDataPublishedByLoaderHook for publication on the
     * STATUS bus.
     *
     * @return
     */
    StatusDataPublishedByLoaderHook createStatusDataPublishedByLoaderHook() {
        return new StatusDataPublishedByLoaderHook(closeSensor.isOn(), openSensor.isOn(), inError, lockStatus);
    }

}
