
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class's goal is to manage the filters.
 *
 * @author virieux
 */
public class FilterManager implements HasLifecycle {

    @LookupName
    private String name;

    @LookupField(strategy = Strategy.CHILDREN)
    private final List<Filter> filters = new ArrayList<>();

    /**
     * A map to store the filters that FCS manages. The key of this map is the
     * filter name.
     *
     */
    @LookupField(strategy = Strategy.CHILDREN)
    protected final TreeMap<String, Filter> filtersMapByName = new TreeMap<>();

    /**
     * A map to store the filters that FCS manages. The key of this map is the
     * filter ID.
     *
     */
    private final TreeMap<Integer, Filter> filtersMapByID = new TreeMap<>();

    /**
     * A getter for the Filters Map sorted by filter name.
     *
     * @return
     */
    public TreeMap<String, Filter> getFiltersMapByName() {
        @SuppressWarnings("unchecked")
        TreeMap<String, Filter> filtersMap = (TreeMap<String, Filter>) filtersMapByName.clone();
        return filtersMap;
    }

    /**
     *
     * @return
     */
    public TreeMap<Integer, Filter> getFiltersMapByID() {
        @SuppressWarnings("unchecked")
        TreeMap<Integer, Filter> filtersMap = (TreeMap<Integer, Filter>) filtersMapByID.clone();
        return filtersMap;
    }

    @Override
    public void init() {
        /**
         * Creation of the HashMap filtersMap. To move this initialization to the
         * constructor is a bad idea. It has been tested and it doesn't work because in
         * the construction phase the environment of the objects Filter is not set, so
         * the method getName() returns null.
        */
        for (Filter f : filters) {
            FCSLOG.info("Add filter to filterManager:" + f.getName());
            filtersMapByID.put(f.getFilterID(), f);
        }
    }

    /**
     * Return a filter which name is given as argument.
     *
     * @param filterName
     * @return
     */
    public Filter getFilterByName(String filterName) {
        return filtersMapByName.get(filterName);
    }

    /**
     * Return Return filter which id is given as argument.
     *
     * @param id
     * @return
     */
    public Filter getFilterByID(int id) {
        return filtersMapByID.get(id);
    }

    /**
     * Return true if a filterID is in the map.
     *
     * @param id
     * @return true if filtersMapByID contains this id.
     */
    public boolean containsFilterID(int id) {
        return filtersMapByID.containsKey(id);
    }

    /**
     * Return Return filter name which correspond to ID.
     *
     * @param id
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return filter name which correspond to ID.")
    public String getFilterNameByID(int id) {
        if (filtersMapByID.containsKey(id)) {
            return filtersMapByID.get(id).getName();
        } else {
            if (id != 0) {
                FCSLOG.warning(name + ": " + id + "  unknown filter ID. Please update filterManager configuration.");
            }
            return "filterID_" + id;
        }
    }

    /**
     * Returns a list of filter names. This list is sorted by filter ID. Useful for
     * initialization of the GUI.
     *
     * @return
     */
    public List<String> getFilterNamesSortedByFilterID() {
        ArrayList<String> namesList = new ArrayList<>();
        TreeMap<Integer, String> filterNamesByID = new TreeMap<>();
        for (Map.Entry<String, Filter> entry : filtersMapByName.entrySet()) {
            Filter filter = entry.getValue();
            filterNamesByID.put(filter.getFilterID(), filter.getName());
        }
        namesList.addAll(filterNamesByID.values());
        return namesList;
    }

    /**
     * This method controls if the filter name given at the console by an operator
     * is correct.
     *
     * @param filterName
     * @throws IllegalArgumentException in the following cases :
     * - the name is not a String,
     * - the name is unknown in the map filters,
     * - the name is known in the map filters, but the filter location is
     * unknown,
     * - the name is known in the map filters, but the filter is out of the
     * camera.
     *
     */
    protected void checkFilterName(String filterName) {

        if (this.getFilterByName(filterName) == null) {
            throw new IllegalArgumentException("Unknown filter name : " + filterName);
        }
    }

    // TODO print a human readable String
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return a printed list of filters.", alias = "printFilterList")
    @Override
    public String toString() {
        return filtersMapByName.toString();
    }

}
