
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOSCarousel;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * To simulate the controller which rotates carousel.
 * @author virieux
 */
public class SimuCarouselController extends CanOpenEPOSCarousel implements EPOSControllerForCarousel {

    @LookupField(strategy=Strategy.TREE)
    private Carousel carousel;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = Strategy.TREE)
    private SimuTTC580 hyttc580;

    protected int targetPosition;
    protected int ssiPosition;
    protected int targetCurrent;
    protected int actualCurrent;

    /**
     * Build a new SimuCarouselController
     * @param nodeID
     * @param serialNB
     */
    public SimuCarouselController(
            int nodeID, String serialNB) {
        super(nodeID, serialNB);
        this.position = 0;
        this.mode = PROFILE_POSITION;
        this.enabledToPublish = true;
    }

    /**
     * To initialize a simulated ssiPosition.
     * @param ssiPosition
     */
    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
        this.position = ssiPosition;
        simulatePosition(ssiPosition);
    }

    @Override
    public void init() {
        updateFakePDOData(cobid1, 0);
        updateFakePDOData(cobid2, 0);
    }


    /**
     * This methods does nothing but save the relativePosition given as argument.
     * @param absolutePosition
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    @Override
    public void writeTargetPosition(int absolutePosition) throws FcsHardwareException {
        FCSLOG.info(name + " => carousel is rotating.");
        FCSLOG.info(name + " => initial position=" + this.position);
        FCSLOG.info(name + " => target position=" + absolutePosition);
        int iterNB = 10;
        int initialPosition = carousel.getPosition();
        int step = (absolutePosition - initialPosition) / iterNB;
        FCSLOG.info("step=" + step);
        try {
            for (int i = 1; i < iterNB; i++) {
                int newPos = initialPosition + (i * step);
                setSsiPosition(newPos);
                FCSLOG.info(name + " i=" + i + ", position=" + position);
                try {
                    Thread.sleep(50);
                    if (carousel.getHaltRequired().get()) {
                        FCSLOG.fine(name + " STOP simulated carousel motion.");
                        return;
                    }
                } catch (InterruptedException ex) {
                    throw new FcsHardwareException(name + " sleep was interrupted.",ex);
                }
            }
            int finalPos = absolutePosition % carousel.getFullTurn();
            setSsiPosition(finalPos);
            hyttc580.simulateSocketSensors(finalPos);
            carousel.updateStateWithSensors();

        } catch (FcsHardwareException  ex) {
            FCSLOG.error(name+" should not raise an Exception:",ex);
        }
    }

    /**
     * Update PDOdata
     * @param cobid
     * @param newPdo
     */
    public void updateFakePDOData(int cobid, long newPdo) {
        if (cobid == cobid1) {
            setPdo1(newPdo);
        } else if (cobid == cobid2) {
            setPdo2(newPdo);
        }
        ((SimuCanOpenInterface) tcpProxy.getCanInterface()).simulatePDOData(cobid, newPdo);
    }

    /**
     * set controller position in PDOData to simulate a carousel position.
     * @param pos
     */
    public void simulatePosition(int pos) {
        System.out.println("simulate carousel position : " + pos);
        long pdo = (long) pos << 32;
        long mask = 0x00000000FFFFFFFFL;
        long newpdo = pdo | (getPdo1() & mask);
        System.out.println("newpdo1 = " + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    }

    /**
     * **************************************************************************
     *
     * Methods copied from SimuEPOSController. This is uggly to copy methods.
     * //TODO see how we can extract an interface with this methods.
     *
     * **************************************************************************
     */
    /**
     * set a position for simulation
     *
     * @param actualPosition
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "For simulator only : Update position with a position given as argument.")
    public void setPosition(int actualPosition) {
        this.position = actualPosition;
    }

    /**
     * overriden for simulation
     *
     * @return
     */
    @Override
    public boolean isTargetReached() {
        return this.position == this.targetPosition
                || this.actualCurrent == this.targetCurrent;
    }

    /**
     * We assum that simulated hardware'subs controller has always correct
     * parameters.
     *
     * @return
     */
    @Override
    public boolean isParametersOK() {
        return true;
    }

    @Override
    public void defineAbsolutePosition(int position) {
        this.position = position;
    }

    @Override
    public void writeParameters(EPOSEnumerations.EposMode mode) {
    }

    @Override
    public void writeParameter(EPOSEnumerations.Parameter param, int value) {
        FCSLOG.debug(name + "writeParameter");
    }

    @Override
    public void writeParameters(Map<String, Integer> paramMap) {
    }

    @Override
    public void writeCurrent(int aValue) {
        this.actualCurrent = aValue;
    }

    @Override
    public void stopAction() {
    }

    @Override
    public int readPosition() {
        return this.position;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     */
    @Override
    public int readSSIPosition() {
        return this.ssiPosition;
    }

    @Override
    public int readCurrent() {
        return actualCurrent;
    }

    @Override
    public EPOSEnumerations.EposMode readMode() {
        return mode;
    }

    @Override
    public int readStatusWord() {
        return statusWord;
    }

    @Override
    public void quickStop() {
        FCSLOG.finest(name + " quickStop done.");
    }

    @Override
    public int getErrorHistoryNB() {
        return 0;
    }

    @Override
    public int readErrorRegister() {
        return 0;
    }

    @Override
    public void updateErrorHistory() {
        errorHistory = new int[0];
    }

    @Override
    public String displayErrorHistory() {
        return "No ERROR";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault() {
    }

    @Override
    public String readParameters(EPOSEnumerations.EposMode mode) {
        return getName() + ":no parameter to read.";
    }

    @Override
    public void checkParameters(EPOSEnumerations.EposMode aMode) {
        /*nothing to do because no controller CPU*/
    }

    @Override
    public void writeControlWord(int w) {
        FCSLOG.debug(name + " writeControlWord : 0x" + Integer.toHexString(w) + " (in decimal:" + w + ")");
        this.statusWord = w;
    }

    @Override
    public void switchOnEnableOperation() {
        writeControlWord(7);
    }
}
