package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import static org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode.*;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.*;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.*;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.NOFILTER;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCKCLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.RELEASECLAMPS;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.SlaveModuleStatus;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.TTC580Utils;

/**
 * This is a socket on the carousel : there is 5 sockets on a carousel. When a
 * filter is on the carousel, it is attached at a socket. A socket has a clamp
 * on each side : - clampX- (clampXminus) - clampX+ (clampXplus) clampX+ and
 * clampX- are the same except for the method isLocked() See
 * FilterClampXminusModule and FilterClampXplusModule
 *
 * The main job of this class CarouselSocket is to synchronize the actions on
 * the 2 clamps. The state of the CarouselSocket is the state of the clamps when
 * both clamps state are identical.
 *
 * During an action on both clamps, we want to lock the CarouselSocket so no
 * other thread can try to access this object before the completion of action.
 * We do that with a Lock and Condition. (see java.util.concurrent).
 *
 * CarouselSocket extends MobileItem because we want to lock or release the
 * clamps when the carousel is at standby position and we want to wait until the
 * action is completed. We know if the action is completed in reading the clamps
 * sensors. So we use the general mechanism provided by the MobileItem.
 *
 *
 * @author virieux
 *
 */
public class CarouselSocket extends MobileItem implements ClearAlertHandler {

    private int id;

    @LookupField(strategy = Strategy.TREE)
    private MainModule mainModule;

    @LookupField(strategy = Strategy.TREE)
    private Carousel carousel;

    private CarouselClamp clampXminus;

    private CarouselClamp clampXplus;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*clampXminusController")
    private EPOSController clampXminusController;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*clampXplusController")
    private EPOSController clampXplusController;

    @LookupField(strategy = Strategy.CHILDREN)
    private SensorPluggedOnTTC580 slaveModuleSensor;

    private SlaveModuleStatus slaveModuleStatus;

    /*To be able to know if the autochanger holds a filter. */
    /*This can't be a Autochanger because when the carousel is in Standalone mode
     /*there is not autochanger.
     /*FilterHolder is an Interface. */
    @LookupField(strategy = Strategy.TREE, pathFilter=".*autochanger")
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.TREE)
    private FilterManager filterManager;

    @LookupField(strategy = Strategy.TREE)
    private PersistencyService persistenceService;

    private final boolean loadFilterLocationAtStartup = true;
    private final boolean saveFilterLocationAtShutdown = true;

    /**
     * carousel positionOnCarousel (angle) when this socket is at standby
     * standbyPosition should be : 360 - positionOnCarousel, but could be different
     */
    @ConfigurationParameter(range = "-4500000..4500000",
            description = " carousel position when this socket is at standby")
    private int standbyPosition;

    @Persist
    private int filterID;

    private volatile FilterClampState clampsState = UNDEFINED;

    private long timeoutForUnlocking = 4000;
    private long timeoutForReleasing = 4000;

    /**
     * Build a new CarouselSocket with a tickMillis of 5000.
     *
     * @param id
     * @param clampXminus
     * @param clampXplus
     * @param standbyPosition
     */
    public CarouselSocket(int id, CarouselClamp clampXminus, CarouselClamp clampXplus, int standbyPosition) {
        this.id = id;
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
        this.standbyPosition = standbyPosition;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerData(new KeyValueData(name, createStatusDataPublishedByCarouselSocket()));
    }

    public int getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    public CarouselClamp getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClamp clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClamp getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClamp clampXplus) {
        this.clampXplus = clampXplus;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns filterID if a filter is in the socket.")
    public int getFilterID() {
        return filterID;
    }

    /**
     *
     * @return NO_FILTER if no filter at standby or filter name
     */
    public String getFilterName() {
        return filterManager.getFilterNameByID(filterID);
    }

    public void setFilterID(int filterID) {
        this.filterID = filterID;
        publishData();
    }

    public int getStandbyPosition() {
        return standbyPosition;
    }

    public SlaveModuleStatus getSlaveModuleStatus() {
        return slaveModuleStatus;
    }

    /**
     * Return true if this socket is at STANDBY position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if this socket is at STANDBY position on the carousel.")
    public boolean isAtStandby() {
        if (carousel.isHomingDone()) {
            return (Math.abs(standbyPosition - carousel.getPosition()) <= 1000) && carousel.getSocketAtStandbyID() == id;
        } else {
            this.raiseWarningOnlyIfNew(HARDWARE_ERROR, "carousel homing is not done", name);
            return carousel.getSocketAtStandbyID() == id;
        }
    }

    /**
     * ***********************************************************************************************
     *  END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        persistenceService.setAutomatic(loadFilterLocationAtStartup, saveFilterLocationAtShutdown);

        this.clampXminus.setController(clampXminusController);
        this.clampXplus.setController(clampXplusController);
    }

    /**
     * Checking if autochanger is holding the filter is delegated to autochanger.
     *
     * @return true if autochanger is at STANDBY and holds a filter.
     */
    private boolean isAutochangerHoldingFilter() {
        return autochanger.isAtStandby() && autochanger.isHoldingFilter();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return clampXminusController.isInitialized() && clampXplusController.isInitialized();
    }

    /**
     * Returns the state of the clamps. If the state is being updated and waiting
     * for a response from a sensor, this methods waits until the state is updated.
     * If the state is not being updated, it returns immediatly the state.
     *
     * @return state
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return clamps state - doesn't read again sensors.")
    public FilterClampState getClampsState() {
        return clampsState;
    }

    /**
     * update state from sensors values.
     *
     */
    public void updateState() {
        FCSLOG.info(name + " updating state.....");
        long beginTimeCmd = System.currentTimeMillis();

        if (slaveModuleSensor.getValue() >= 1 && slaveModuleSensor.getValue() <= 7) {
            slaveModuleStatus = SlaveModuleStatus.getStatusByCode(slaveModuleSensor.getValue());
        } else {
            slaveModuleStatus = SlaveModuleStatus.UNKNOWN_STATUS;
        }
        if (slaveModuleSensor.getValue() < 4 && slaveModuleSensor.getValue() > 0) {
            updateClampsState();
        } else {
            clampsState = UNDEFINED;
            //TODO: Has to be raiseAlarm for final carousel.
            this.raiseWarning(CA_SENSOR_ERROR,
                    " slave module status = " + slaveModuleSensor.getValue() + " => " + slaveModuleStatus, name);
        }
        publishData();
        long duration = System.currentTimeMillis() - beginTimeCmd;
        FCSLOG.info(name + " updateState duration = " + duration);
    }

    private void updateClampsState() {
        clampXminus.updateState();
        clampXplus.updateState();
        boolean inError = clampXminus.getClampState() == FilterClampState.ERROR
                || clampXplus.getClampState() == FilterClampState.ERROR;

        if (clampXminus.getClampState() == clampXplus.getClampState()) {
            this.clampsState = clampXminus.getClampState();
            FCSLOG.info(name + " : clamps are updated");

        } else if (clampXminus.getClampState() == UNCLAMPEDEMPTY && clampXplus.getClampState() == READYTOCLAMP) {
            if (clampXminus.isReleased() && clampXplus.isReleased()) {
                this.clampsState = READYTOCLAMP;

            } else {
                this.clampsState = UNCLAMPEDEMPTY;
            }

        } else if (!inError) {
            FCSLOG.fine(name + " clampXminus is " + clampXminus.getClampState()
                    + " and clampXplus is " + clampXplus.getClampState() + " clampState = " + UNDEFINED);
            this.clampsState = UNDEFINED;

        } else {
            this.clampsState = FilterClampState.ERROR;
            String msg = name + ": clampState at Xminus side is different from clampState at Xplus side. ";
            msg = msg + clampXminus.getName() + " state=" + clampXminus.getClampState() + " ";
            msg = msg + clampXplus.getName() + " state=" + clampXplus.getClampState();
            this.raiseWarning(CA_SENSOR_ERROR, msg, name);
        }
    }

    /**
     * update field filterID
     * to be used only with whole FCS.
     * In standalone mode, filters can't be removed neither loaded into carousel so filterID never changes.
     */
    public void updateFilterID() {
        if (this.isEmpty()) {
            filterID = 0;
        } else if (autochanger.isAtStandby()) {
            filterID = autochanger.getFilterID();
        }
    }

    /**
     * check and update presence sensor offset2 for the 2 clamps
     *
     * @param sdo
     */
    public void checkAndUpdateOffset2(long sdo) {
        clampXminus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xminus(sdo));
        clampXplus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xplus(sdo));
    }


    /*
     * The carousel socket is empty if there is no filter in the socket. That
     * appends when clamps of both side of the socket are empty. A clamp is empty
     * when no filter is engaged in the clamp.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is no filter in the socket.")
    public boolean isEmpty() {
        return clampXminus.getFilterPresenceStatus() == NOFILTER && clampXplus.getFilterPresenceStatus() == NOFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are LOCKED.")
    public boolean isClampedOnFilter() {
        return clampsState == CLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedOnFilter() {
        return clampsState == UNCLAMPEDONFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is NO filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedEmpty() {
        return clampsState == UNCLAMPEDEMPTY;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if this socket is ready to clamp a filter.")
    public boolean isReadyToClamp() {
        return clampsState == READYTOCLAMP;
    }

    /**
     * Releases the 2 clamps of the socket.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Release the 2 clamps of this socket if the socket is at STANDBY position.")
    public void releaseClamps() {
        carousel.updateStateWithSensors();
        FCSLOG.info("Checking conditions for release clamp " + name + " on socket " + "at standby position.");
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
        }
        if (clampXminus.isFilterEngaged() || clampXplus.isFilterEngaged()) {
            throw new RejectedCommandException(
                    name + ": Can't release clamps because a filter is engaged in clamps.");
        }
        this.timeoutForReleasing = java.lang.Math.max(clampXminus.timeoutForReleasing, clampXplus.timeoutForReleasing);
        FCSLOG.info(name + ": Releasing clamps at standby position.");
        this.executeAction(RELEASECLAMPS, timeoutForReleasing);
    }

    /**
     * This method unclamps the 2 clamps at standby position.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Unlock the 2 clamps of this socket if the socket is at STANDBY position.")
    public void unlockClamps() {
        carousel.updateStateWithSensors();
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
        }
        if (!this.isAutochangerHoldingFilter()) {
            throw new RejectedCommandException("CANNOT UNLOCK CLAMPS if FILTER is not HELD by autochanger.");
        }
        this.timeoutForUnlocking = java.lang.Math.max(this.clampXminus.timeoutForUnlocking,
                this.clampXplus.timeoutForUnlocking);
        this.executeAction(UNLOCKCLAMPS, timeoutForUnlocking);
        FCSLOG.info("Command unlockClamps completed");
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns a String representation of a CarouselSocket.")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append(",FilterID in socket : ");
        sb.append(filterID).append("\n");
        return sb.toString();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == UNLOCKCLAMPS) {
            return this.clampsState == UNCLAMPEDONFILTER;

        } else if (action == RELEASECLAMPS) {
            return this.clampsState == READYTOCLAMP;

        } else {
            throw new IllegalArgumentException("Action on clamps on socket must " + "be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateSocketAtStandbyWithSensors();
    }

    /**
     * Start action of unlocking or releasing clamps.
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(MobileItemAction action) {
        if (action == UNLOCKCLAMPS) {
            this.clampXminusController.enable();
            this.clampXplusController.enable();

            // Following a request from Guillaume Daubard, we first send a small current
            // to the clamp to prepare it and prevent an unstable state when we send the
            // full current to unlock.
            this.clampXminusController.writeCurrent((short) -100);
            this.clampXplusController.writeCurrent((short) -100);

            this.clampXminusController.writeCurrent((short) this.clampXminus.getCurrentToUnlock());
            this.clampXplusController.writeCurrent((short) this.clampXplus.getCurrentToUnlock());

        } else if (action == RELEASECLAMPS) {
            this.clampXminusController.disableOperation();
            this.clampXplusController.disableOperation();

        } else {
            throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        // TODO complete this
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        FCSLOG.debug(name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        carousel.updateStateWithSensors();
        FCSLOG.debug(name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByCarouselSocket()));
    }

    /**
     * Create an object to be published on the STATUS bus by a CarouselSocket.
     *
     * @return
     */
    public StatusDataPublishedByCarouselSocket createStatusDataPublishedByCarouselSocket() {
        StatusDataPublishedByCarouselSocket status = new StatusDataPublishedByCarouselSocket();
        status.setAtStandby(this.isAtStandby());
        status.setSlaveStatus(slaveModuleStatus);
        status.setClampsState(clampsState);
        status.setEmpty(this.isEmpty());
        status.setFilterID(filterID);
        status.setSocketID(id);
        if (this.isEmpty()) {
            status.setFilterName(NO_FILTER);
        } else {
            status.setFilterName(getFilterName());
        }
        return status;
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet."); // To change body of generated methods, choose
                                                                       // Tools | Templates.
    }

    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        String alertId = name + getAlertSeparator() + CA_SENSOR_ERROR.name();
        if (alert.getAlertId().equals(alertId)) {
            // TODO check that alert is no more valid (sensors are OK / slave module is OK,
            // etc...)
            return CLEAR_ALERT;
        } else {
            return UNKNOWN_ALERT;
        }
    }

    @Override
    public void shutdown() {
        /* to save filter location in a file */
        persistenceService.persistNow();
    }

}
