package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import static org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.*;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus.*;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.*;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.RELEASE;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * A class to model a clamp that holds a filter on the carousel. Each clamp on
 * the carousel is coupled with 2 sensors which detect the presence of a filter,
 * or if the clamp is locked or unlocked. This class provides a method to lock,
 * unlock or release the clamps, and to compute the clampState of the clamp. The
 * clampState of the clamp is computed, updated and published on the status bus
 * each tick of the timer.
 *
 *
 * @author virieux
 */
public class CarouselClamp extends MobileItem implements MovedByEPOSController, ClearAlertHandler {

    /**
     * to be able to known if autochanger hold a filter or not at STANDBY can't be
     * an instance of Autochanger because in standalone it's not an instance of
     * class Autochanger can't be strategy = Strategy.TREE because in fcs__hardware
     * there are 2 FilterHolder : autochanger and loader
    */
    @LookupField(strategy = Strategy.TREE, pathFilter = ".*autochanger")
    private FilterHolder autochanger;

    /* to be able to known if autochanger hold a filter or not at STANDBY */
    @LookupField(strategy = Strategy.TREE)
    private Carousel carousel;

    @LookupField(strategy = Strategy.TREE)
    private PersistencyService persistenceService;

    private static boolean loadOffsetAtStartup = true;
    private static boolean saveOffsetAtShutdown = true;

    /**
     * The motor controller which controls this clamp. This is initialized in the
     * initModule method of CarouselSocket.
     */
    protected EPOSController controller;

    /**
     * The filterPresenceSensor detects where if the filter is in the clamp.
     *
     * @See enum PresenceFilterStateOnClamp
     */
    private final SensorPluggedOnTTC580 filterPresenceSensor;

    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final SensorPluggedOnTTC580 lockSensor;

    private FilterClampState clampState = UNDEFINED;
    private FilterPresenceStatus filterPresenceStatus = NOT_LOCKABLE;
    private LockStatus lockStatus = LockStatus.UNKNOWN;

    @ConfigurationParameter
    protected int currentToUnlock = -4660;

    @ConfigurationParameter
    protected int currentToMaintainUnlocked = 200;

    @ConfigurationParameter(description = "for command unlock : time between little current to "
            + "prepare hardware and currentToLock ")
    protected int timeToPrepareUnlock = 200;

    /**
     * if lockSensor returns a value < lockSensorMinValue, it's an ERROR.
     */
    @ConfigurationParameter(range = "0..32768")
    private Integer lockSensorMinValue = 200;

    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA: the sensor
     * is unlocked. If the lock sensor returns a value between lockSensorValueA and
     * lockSensorValueB: we don't know. If the sensor returns a value between
     * lockSensorValueB and lockSensorMaxValue: the clamp is locked. If the sensor
     * returns a value greater than lockSensorMaxValue the sensor is in ERROR.
     *
     * This value is read on hyttc580 when fcs starts.
     */
    @ConfigurationParameter
    protected Integer lockSensorValueA = 5000; /* 6000 for clampXplus */

    /**
     * If sensor returns a value between lockSensorValueB and lockSensorMaxValue,
     * the clamp is locked.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorValueB = 9500;

    /**
     * If sensor returns a value above lockSensorMaxValue, the sensor is in error.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorMaxValue = 11000;

    /**
     * if filter presence sensor returns a value between 0 and filterPresenceOffset1
     * : sensor is in error. The value of filterPresenceOffset1 is read on httc580
     * when fcs starts and it is persisted. If the new value read is different from
     * the last value read it may means that the clamp has been replaced.
     *
     */
    protected Integer filterPresenceOffset1 = 200;

    /**
     * if filter presence sensor returns a value between filterPresenceOffset1 and
     * filterPresenceOffset2 : the filter is engaged and lockable. This value is
     * read on hyttc580 when fcs starts. It is persisted. If the new value read is
     * different from the precedent value, it means that the clamp has been
     * replaced, and an ALERT must be launched.
     */
    @Persist
    protected Integer filterPresenceOffset2 = 3000;

    @ConfigurationParameter(description = "an offset for tests after carousel has been transported")
    protected Integer filterPresenceOffset3 = 0;
    /**
     * if filter presence sensor returns a value between filterPresenceOffset2 and
     * filterPresenceValueB filter is not lockable. And if the value is greater than
     * * filterPresenceMaxValue, there's no filter.
     */
    @ConfigurationParameter(range = "0..12000")
    protected Integer filterPresenceValueB = 9200;

    @ConfigurationParameter(range = "0..15000")
    private Integer filterPresenceMaxValue = 12000;

    @ConfigurationParameter(range = "0..10000")
    protected long timeoutForUnlocking = 1500;

    @ConfigurationParameter(range = "0..10000")
    protected long timeoutForReleasing = 1500;

    private boolean atStandby = false;

    private int actualCurrent = 0;

    /**
     * Build a new CarouselClampModule with a tickMillis value:5000;
     *
     * @param filterPresenceSensor
     * @param lockSensor
     */
    public CarouselClamp(SensorPluggedOnTTC580 filterPresenceSensor, SensorPluggedOnTTC580 lockSensor
    // Thermometer thermometer
    ) {
        this.filterPresenceSensor = filterPresenceSensor;
        this.lockSensor = lockSensor;
        // this.thermometer = thermometer;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerData(new KeyValueData(name, createStatusDataPublishedByClamp()));
    }

    public boolean isAtStandby() {
        return atStandby;
    }

    /**
     * Returns the amount of current needed to unlock this clamp. This is defined in
     * the ConfigurationSystem.
     *
     * @return
     */
    public short getCurrentToUnlock() {
        return (short) currentToUnlock;
    }

    /**
     * In the initialization phase, this method is used to initialize the
     * controller. cf initModule in CarouselSocket
     *
     * @param actuator
     */
    protected void setController(EPOSController actuator) {
        this.controller = actuator;
    }

    /**
     * @return the lockSensorValueA for simulation
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }

    public int getLockSensorMaxValue() {
        return lockSensorMaxValue;
    }

    /**
     * @return the filterPresenceOffset1
     */
    public int getFilterPresenceOffset1() {
        return filterPresenceOffset1;
    }

    /**
     * @return the filterPresenceOffset2
     */
    public int getFilterPresenceOffset2() {
        return filterPresenceOffset2;
    }

    /**
     * @return the filterPresenceValueB
     */
    public int getFilterPresenceValueB() {
        return filterPresenceValueB;
    }

    public Integer getFilterPresenceOffset3() {
        return filterPresenceOffset3;
    }

    /**
     * For simulation.
     *
     * @return the filterPresenceSensor
     */
    public SensorPluggedOnTTC580 getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * For simulation.
     *
     * @return the lockSensor
     */
    public SensorPluggedOnTTC580 getLockSensor() {
        return lockSensor;
    }

    /**
     * @return the lockStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return the lockStatus")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     *
     * @return filterPresenceStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    /**
     * for tests
     *
     * @param lockSensorValueA
     */
    public void setLockSensorValueA(Integer lockSensorValueA) {
        this.lockSensorValueA = lockSensorValueA;
    }

    /**
     * for tests
     *
     * @param lockSensorValueB
     */
    public void setLockSensorValueB(Integer lockSensorValueB) {
        this.lockSensorValueB = lockSensorValueB;
    }

    public void setLockSensorMaxValue(Integer maxValue) {
        this.lockSensorMaxValue = maxValue;
    }

    public void setFilterPresenceOffset1(Integer value) {
        this.filterPresenceOffset1 = value;
    }

    public void setFilterPresenceOffset2(Integer value) {
        this.filterPresenceOffset2 = value;
    }

    public void setFilterPresenceValueB(Integer filterPresenceValue) {
        this.filterPresenceValueB = filterPresenceValue;
    }

    /**
     * ***********************************************************************************************
     * * END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     *
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        persistenceService.setAutomatic(loadOffsetAtStartup, saveOffsetAtShutdown);
    }

    /**
     * *** end of lifecycle methods
     * *************************************************
     */

    /**
     * *** ClearAlertHandler methods
     * *************************************************
     */
    /**
     * to be able to clear alerts
     *
     * @param alert
     * @param alertState
     * @return
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        return CLEAR_ALERT;
    }

    /**
     * *** end of ClearAlertHandler methods
     * *************************************************
     */

    /**
     * *
     *
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return controller.isInitialized();
    }

    /**
     * Returns the clampState of the clamp. If the clampState is being updated and
     * waiting for a response from a sensor, this methods waits until the clampState
     * is updated. If the clampState is not being updated, it returns immediatly the
     * clampState.
     *
     * @return clampState
     *
     */
    public FilterClampState getClampState() {
        return clampState;
    }

    /**
     * This methods returns true if the clamp is locked. In the simulator, it is
     * overriden.
     *
     * @return true if the clamp is locked
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if the clamp is locked")
    public boolean isLocked() {
        return this.lockStatus == LOCKED;
    }

    /**
     *
     * @return true if filter is engaged on the clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if filter is engaged on the clamp : filter presence sensors sees it ")
    public boolean isFilterEngaged() {
        return this.filterPresenceStatus == LOCKABLE;
    }

    /**
     * compare offset2 value given as argument with persisted value.
     *
     * @param newOffset2
     */
    public void checkAndUpdateOffset2(int newOffset2) {
        if (newOffset2 != filterPresenceOffset2) {
            filterPresenceOffset2 = newOffset2;
            this.raiseAlarm(CA_SENSOR_ERROR, " offset2 has changed; new value = " + newOffset2, name);
        }
    }

    /**
     *
     * @return true if this clamp is clampXminus.
     */
    public boolean isXminus() {
        return name.contains("Xminus");
    }

    /**
     * update filterPresenceStatus from value returned by filterPresenceSensor
     */
    public void updateFilterPresenceStatus() {
        int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
        if (newFilterPresenceSensorValue < this.filterPresenceOffset1) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue + " must be >= " + this.filterPresenceOffset1, name);

        } else if (newFilterPresenceSensorValue < this.filterPresenceOffset2 + this.filterPresenceOffset3) {
            this.filterPresenceStatus = LOCKABLE;

        } else if (newFilterPresenceSensorValue < this.filterPresenceValueB) {
            this.filterPresenceStatus = NOT_LOCKABLE;

        } else if (newFilterPresenceSensorValue < this.filterPresenceMaxValue) {
            this.filterPresenceStatus = NOFILTER;

        } else {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue + " must be <=" + this.filterPresenceMaxValue, name);
        }
    }

    /**
     * Update lockStatus from value returned by lockSensor.
     *
     * @throws FcsHardwareException
     */
    public void updateLockStatus() {
        int mechaValue = lockSensor.getValue();
        if (mechaValue < lockSensorMinValue) {
            this.lockStatus = LockStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + " ERROR new read value FOR LOCK SENSOR  = " + mechaValue
                    + " should be >= " + lockSensorMinValue, name);

        } else if (mechaValue < lockSensorValueA) {
            this.lockStatus = UNLOCKED;

            /* for clampXminus, lockSensor can sent a value > lockSensorValueA but it is opened */
        } else if (mechaValue <= lockSensorValueA + 500 && isXminus()) {
            this.lockStatus = RELAXED;

        } else if (mechaValue <= lockSensorValueB) {
            this.lockStatus = LockStatus.UNKNOWN;

        } else if (mechaValue <= lockSensorMaxValue) {
            this.lockStatus = LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + " ERROR new read value FOR LOCK SENSOR  = " + mechaValue
                    + " should be <= " + lockSensorMaxValue, name);
        }
    }

    /**
     * Update clampState from values sent by sensors. This method updates the clamp
     * clampState regarding the value returned by the filter presence sensor and the
     * value returned by the method isLocked(). When the update is completed, it
     * sends a signal to threads waiting to get the new value of clampState.
     *
     * @throws FcsHardwareException
     */
    public void updateState() {
        synchronized (this) {
            this.updateFilterPresenceStatus();
            this.updateLockStatus();
            this.computeClampState();
            this.updateCurrent();
        }
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByCarouselClamp status = this.createStatusDataPublishedByClamp();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }

    /**
     * Create an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByCarouselClamp createStatusDataPublishedByClamp() {
        StatusDataPublishedByCarouselClamp status = new StatusDataPublishedByCarouselClamp();
        status.setClampState(clampState);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterPositionSensorValue(filterPresenceSensor.getValue());
        status.setLockSensorValue(lockSensor.getValue());
        status.setLockStatus(lockStatus);
        status.setFilterPresenceOffset1(filterPresenceOffset1);
        status.setFilterPresenceOffset2(filterPresenceOffset2);
        status.setActualCurrent(actualCurrent);
        if (controller == null) {
            status.setControllerInFault(false);
            status.setControllerEnabled(false);
        } else {
            status.setControllerInFault(controller.isInError());
            status.setControllerEnabled(controller.isEnabledToPublish());
        }
        return status;
    }

    /**
     * Compute the global state of the clamp given the lock sensor and the presence
     * filter sensor state. This has to be overriden for the clamp X-.
     *
     */
    private void computeClampState() {

        if (this.filterPresenceStatus == FilterPresenceStatus.ERROR || this.lockStatus == LockStatus.ERROR) {
            clampState = FilterClampState.ERROR;

        } else if (this.filterPresenceStatus == NOT_LOCKABLE) {
            clampState = UNLOCKABLE;

        } else if (this.filterPresenceStatus == LOCKABLE) {
            // a filter is in the socket

            if (this.lockStatus == LOCKED) {
                clampState = CLAMPEDONFILTER;

            } else if (this.lockStatus == UNLOCKED) {
                clampState = UNCLAMPEDONFILTER;

            } else {
                FCSLOG.fine(name + " is " + LOCKABLE + " and " + lockStatus
                        + " lockValue = " + lockSensor.getValue() + " clampState = " + UNDEFINED);
                clampState = UNDEFINED;
            }

        } else if (this.filterPresenceStatus == NOFILTER) {
            // nofilter for clampXplus (when no filter is detected)
            // impossible to have no filter and clamp LOCKED for clampXminus.

            if (this.lockStatus == LOCKED) {
                clampState = READYTOCLAMP;

                // clampXminus can be RELAXED
            } else if (this.lockStatus == UNLOCKED || this.lockStatus == RELAXED) {
                clampState = UNCLAMPEDEMPTY;

            } else {
                FCSLOG.fine(name + " is " + NOFILTER + " and " + lockStatus
                        + " lockValue = " + lockSensor.getValue() + " clampState = " + UNDEFINED);
                clampState = UNDEFINED;
            }

        } else {
            // NOT_LOCKABLE (UNDEFINED)
            FCSLOG.fine(name + " is " + filterPresenceStatus + " and " + lockStatus
                    + " lockValue = " + lockSensor.getValue() + " clampState = " + UNDEFINED);
            clampState = UNDEFINED;
        }
    }

    /**
     * The clamps on the carousel are locked automaticaly when the filter comes at
     * the standby position. To be able to lock automaticaly again, it has to be
     * released after each time it has been unlocked.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING3, type = Command.CommandType.ACTION, description = "Release clamp in order to get ready to clamp a filter again")
    public void release() {
        FCSLOG.info("Checking conditions for release clamp " + name + " on socket at standby position.");

        carousel.updateSocketAtStandbyWithSensors();
        FCSLOG.info("Releasing clamp " + name + " on socket at standby position.");
        this.executeAction(RELEASE, timeoutForUnlocking);
    }

    /**
     *
     * Check if controller has been disabled.
     *
     * @return true if controller is not enabled.
     */
    @Command(level = Command.ENGINEERING3, type = Command.CommandType.ACTION, description = "Check if controller has been disabled.")
    public boolean isReleased() {
        return !controller.isEnabled();
    }

    /**
     * Unlock the clamp when a filter is locked by the clamp. This is used only in
     * engineering mode.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Unlock the clamp")
    public void unlock() {
        FCSLOG.info(name + ": " + "UNLOCK State1 = " + clampState.toString());
        carousel.updateSocketAtStandbyWithSensors();

        if (autochanger.isAtStandby() && !autochanger.isHoldingFilter()) {
            throw new RejectedCommandException("CANNOT UNLOCK A CLAMP if FILTER is not HELD by autochanger.");
        }
        this.executeAction(UNLOCK, timeoutForUnlocking);
    }

    public void checkIsAtStandby() {
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + " is not at STANDBY. Can't be moved.");
        }
    }

    @Override
    public void startAction(MobileItemAction action) {
        if (action == UNLOCK) {
            this.controller.enable();
            // Following a request from Guillaume Daubard, we first send a small current
            // to the clamp to prepare it and prevent an unstable state when we send the
            // full current to unlock.
            this.controller.writeCurrent((short) -100);

            // a little sleep to let time to the hardware to prepare
            FcsUtils.sleep(this.timeToPrepareUnlock, name);

            this.controller.writeCurrent((short) this.currentToUnlock);

        } else if (action == RELEASE) {
            controller.disableOperation();

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK " + "or RELEASE");
        }

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == UNLOCK) {
            return this.clampState == UNCLAMPEDONFILTER;

        } else if (action == RELEASE) {
            // computeClampStateWhenReleasing();
            // return this.clampState == FilterClampState.READYTOCLAMP;
            return !controller.isEnabled();

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK " + "or RELEASE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateSocketAtStandbyWithSensors();
    }

    /**
     * This methods read the thermometer, update the field temperature and returns
     * the value sent by the thermometer;
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public double updateTemperature() {
        // TODO implement Monitoring
        // double temp = this.thermometer.readTemperature();
        // this.temperature = temp;
        // return temp;
        return 0;
    }

    // TODO check with Guillaume what is to be done to stop the actions.
    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        FCSLOG.debug(name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.debug(name + " is ENDING action " + action.toString());
        carousel.updateStateWithSensors();
        FCSLOG.debug(name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public EPOSController getController() {
        return controller;
    }

    @Override
    public void updateCurrent() {
        if (atStandby) {
            /* current has been updated by PDO */
            actualCurrent = controller.getCurrent();
        } else {
            actualCurrent = 0;
        }
    }

}
