package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_TIMEOUT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.EMCY;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByCanOpenDevice;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public class CanOpenDevice implements PieceOfHardware, HasLifecycle {

    @ConfigurationParameter(isFinal = true, description = "CANOpen node ID of this CANOpen device")
    protected int nodeID;

    @ConfigurationParameter(description = "serial number of this CANOpen device")
    protected String serialNB;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = Strategy.ANCESTORS)
    protected CanOpenProxy tcpProxy;

    protected boolean initialized = false;
    protected boolean booted = false;

    protected boolean inError = false;
    protected int[] errorHistory = new int[0];
    protected String errorRegister = NO_ERROR;

    // est renseigné dès que l'objet est contruit, pas accessible dans le
    // constructeur
    @LookupName
    protected String name;

    @LookupField(strategy = Strategy.TOP)
    protected Subsystem subs;

    /**
     * Create a new CanOpenDevice with a node ID and a serial number.
     *
     * @param nodeID
     * @param serialNB
     */
    public CanOpenDevice(int nodeID, String serialNB) {
        this.nodeID = nodeID;
        this.serialNB = serialNB;
    }

    public CanOpenDevice() {
    }


    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * Return name. Need this method because EPOSController uses getName() to log
     * events which concerns controllers.
     *
     * @return
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns this CANopen device's serial number.")
    @Override
    public String getSerialNB() {
        return this.serialNB;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns this CANopen device's node ID and print it in decimal format.")
    @Override
    public int getNodeID() {
        return this.nodeID;
    }

    /**
     * Returns true if this CANopen node is booted.
     *
     * @return
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if this CANopen node is booted.")
    public boolean isBooted() {
        return booted;
    }

    @Override
    public void setBooted(boolean isBooted) {
        this.booted = isBooted;
    }

    /**
     * Returns true if this CANopen node is initialized.
     *
     * @return
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if this CANopen node is initialized.")
    public boolean isInitialized() {
        return this.initialized;
    }

    /**
     * Return tcpProxy
     *
     * @return
     */
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }

    /**
     * For the GUI
     *
     * @return a list of Errors - can be empty
     */
    public int[] getErrorHistory() {
        return errorHistory.clone();
    }

    /**
     * For the GUI
     *
     * @return
     */
    public String getErrorRegister() {
        return errorRegister;
    }

    /**
     * For the GUI
     *
     * @return
     */
    public boolean isInError() {
        return inError;
    }

    /**
     * This method is to be used when receiving an EmergencyMessage from the CANbus
     * for this device.
     *
     * An emergency message for a controller can be received from the CANbus in 2
     * cases: - a faultReset has been previously sent to the controller, in this
     * case the error code of the emergency message is 0, - or when a new error is
     * detected by the controller device, in this case the error code is different
     * from 0.
     *
     * This method raises an ALARM, updates the boolean inError and publishes data.
     * It's useful for the GUI and the monitoring of the hardware.
     *
     * @param emcyMsg the emergency message to be processed.
     */
    @Override
    public void onEmergencyMessage(EmergencyMessage emcyMsg) {
        if (agentStateService.isInState(PhaseState.INITIALIZING)) {
            FCSLOG.error(name + " received emergency message during INITIALIZATION : " + emcyMsg);

        } else if (0 != emcyMsg.getErrorRegisterCode()) {
            FCSLOG.error(name + " emergency message received: " + emcyMsg.toString());
            this.inError = true;
            try {
                this.errorRegister = emcyMsg.getErrorRegisterName();
                this.errorHistory = readErrorHistory();
                this.raiseAlarm(EMCY, emcyMsg.toString(), name);
                publishData();
            } catch (FcsHardwareException ex) {
                this.raiseWarning(CAN_BUS_TIMEOUT, " couldn't read EPOS error register ", name, ex);
            }
        }
    }

    /**
     * Reset fields which contains error messages.
     */
    public void resetError() {
        this.inError = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new int[0];
        publishData();
    }

    /**
     * This methods has to be overridden if there is some initialization to do for
     * the device.
     *
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Initialize the device.")
    public void initializeAndCheckHardware() {
        if (this.isBooted()) {
            this.initialized = true;
        }
    }

    /**
     * This methods writes the transmission type given as argument on the device.
     *
     * @param transmissionType
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Write transmission type to the device.")
    protected void writeTransmissionTypeToDevice(int transmissionType) {
        if (this.isBooted()) {
            this.tcpProxy.writeSDO(this.nodeID, 0x1801, 0x2, 0x1, transmissionType);

            this.tcpProxy.writeSDO(this.nodeID, 0x1802, 0x2, 0x1, transmissionType);

            FCSLOG.info(name + " transmission type is now " + transmissionType);
        }
    }

    /**
     * Sends a writeSDO request to the CANopen device.
     *
     * @param index
     * @param subindex
     * @param size
     * @param newValue
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Write on the Can Open Device with a command wsdo.")
    public void writeSDO(int index, int subindex, int size, int newValue) {
        tcpProxy.writeSDO(this.nodeID, index, subindex, size, newValue);
    }

    /**
     * Sends a readSDO request to the CANopen device.
     *
     * @param index
     * @param subindex
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Read Can Open Device with a command rsdo.")
    public long readSDO(int index, int subindex) {
        return tcpProxy.readSDO(this.nodeID, index, subindex);
    }

    /**
     * return a Hexadecimal String representation of serial number. For end user.
     * 
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "return a Hexadecimal representation of serial number.")
    public String printSerialNumber() {
        return Integer.toHexString((int) readSDO(0x1018, 04));
    }

    /*************************************************************************
     * Methods to read or display the errors that have occured on the device The
     * errors are stored in 2 places : - error register at index 1001 - error
     * history at index 1003 /
     **************************************************************************/

    /**
     * Read Error Register on the controller (index 1001).
     *
     * @return error code
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Reads Error Register on the controller (index 1001).")
    public int readErrorRegister() {
        return (int) readSDO(0x1001, 0);
    }

    /**
     * Read Error Register on the controller (index 1001), retrieve error name and
     * return a String representation of the error register. For end user usage at
     * console.
     * 
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3, description = "Display Error Register on the controller (index 1001).")
    public String displayErrorRegister() {
        int error = readErrorRegister();
        String errorName = CanOpenErrorsTable.getErrorRegisterNameByCode(error);
        FCSLOG.debug("error register=" + Integer.toHexString(error) + " error name=" + errorName);
        return errorName;
    }

    /**
     * read error history on the device and return a list of device error code. The
     * error history holds errors that have occurred on the device and have been
     * signalled via the emergency object.
     *
     * @return errors : a list of device error code
     */
    public int[] readErrorHistory() {
        int numberOfErrors = readNumberOfErrors();
        int[] errors = new int[numberOfErrors];
        for (int i = 0; i < numberOfErrors; i++) {
            try {
                errors[i] = (int) readSDO(0x1003, i + 1);
            } catch (SDORequestException ex) {
                String msg = " no response to command: readErrorHistory - POWER FAILURE ? ";
                this.raiseWarning(CAN_BUS_TIMEOUT, msg, name, ex);
            }
        }
        return errors;
    }

    /**
     * Read error history on the device and return its String representation.
     * 
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Display Error Register on the controller (index 1001).")
    public String displayErrorHistory() {
        errorHistory = this.readErrorHistory();
        StringBuilder sb = new StringBuilder("Error history contains " + errorHistory.length + " errors.");
        if (!(errorHistory.length == 0)) {
            sb.append("\nList of errors in history," + "the newest is the first, the oldest the last :\n");
            for (int ix = 0; ix < errorHistory.length; ix++) {
                sb.append("Error code = 0x");
                sb.append(Integer.toHexString(errorHistory[ix]));
                sb.append(" / error name = ");
                sb.append(displayError(errorHistory[ix]));
                sb.append("\n");
            }
        }
        return sb.toString();
    }

    /**
     * read an error table to return an error name corresponding to the error code
     * given as argument.
     * 
     * @param err
     * @return string representation of error code given as argument
     */
    public String displayError(int err) {
        return CanOpenErrorsTable.getDeviceErrorNameByCode(err);
    }

    /**
     * Read and return the number of errors registred in the Error History of the
     * device.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Read and returns the number of errors registred in the Error History.")
    public int readNumberOfErrors() {
        return (int) readSDO(0x1003, 0);
    }

    /***********************************************************************************
     * End of methods to read errors on the device
     * /***********************************************************************************
     * 
     * /** This method saves the parameters in the controller memory.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "This method saves the parameters in the controller memory.")
    public void saveParameters() {
        this.writeSDO(0x1010, 1, 4, 0x65766173);
    }

    /**
     * Configure a node as a hearbeat producer. Every heartbeatTime milliseconds
     * this node will produce a message like: 0x700+nodeID: 05 - 1340268404.980000
     *
     * @param nodeID        node ID
     * @param heartbeatTime heartbeat in milliseconds
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Configure a CANopen device as a heartbeat producer.")
    public void configAsHeartbeatProducer(int nodeID, int heartbeatTime) {
        if (heartbeatTime < 0 || heartbeatTime > FcsUtils.MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException(
                    "heartbeat time is coded on 2 bytes" + " can't be > " + FcsUtils.MAX_VALUE_2BYTES);
        }
        tcpProxy.writeSDO(nodeID, 0x1017, 0, 2, heartbeatTime);
    }

    /**
     * Configure a node as a hearbeat consumer. Every heartbeatTime milliseconds
     * this node will produce a message like: 0x700+nodeID: 05 - 1340268404.980000
     *
     * @param nodeID         node ID in decimal
     * @param producerNodeID
     * @param heartbeatTime  FORMAT=decimal UNIT=milliseconds
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeatTime) {
        if (nodeID < 0 || nodeID > FcsUtils.MAX_NODE_ID) {
            throw new IllegalArgumentException("nodeID must be > 0 and <=" + FcsUtils.MAX_NODE_ID);
        }
        if (producerNodeID < 0 || producerNodeID > FcsUtils.MAX_NODE_ID) {
            throw new IllegalArgumentException("producerNodeID must be > 0 and <=" + FcsUtils.MAX_NODE_ID);
        }
        if (heartbeatTime < 0 || heartbeatTime > FcsUtils.MAX_VALUE_2BYTES) {
            throw new IllegalArgumentException(
                    "heartbeat time is coded on 2 bytes" + " can't be > " + FcsUtils.MAX_VALUE_2BYTES);
        }
        int value = producerNodeID + heartbeatTime;
        tcpProxy.writeSDO(nodeID, 0x1016, 1, 4, value);
    }

    /**
     * Do nothing if this device doesn't send PDOs except log for debug purpose. To
     * be overriden if this device sends PDOS.
     * 
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Override
    public void doInitializePDOs() throws DriverException {
        FCSLOG.info(name + " no PDO to be initialized.");
    }

    /**
     * Do nothing if this device doesn't send PDOs except log for debug purpose. To
     * be overriden if this device sends PDOS.
     * 
     * @param pdo
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Override
    public void updateFromPDO(PDOData pdo) throws DriverException {
        FCSLOG.info(name + " no PDO to be updated.");
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Print CANopen device.")
    public String toString() {
        StringBuilder sb;
        if (name == null) {
            sb = new StringBuilder();
        } else {
            sb = new StringBuilder(name);
        }
        sb.append("/SerialNumber=");
        sb.append(this.serialNB);
        sb.append("/NodeID=0x");
        sb.append(Integer.toHexString(nodeID));
        sb.append("/NodeID(in_decimal)=");
        sb.append(this.nodeID);
        sb.append("/");
        // because the configuration system can call toString before the object is
        // completely built
        // for tracing purpose.
        if (tcpProxy == null) {
            return sb.toString();
        }
        if (isBooted()) {
            sb.append("BOOTED/");

        } else {
            sb.append("NOT YET BOOTED/");
        }
        return sb.toString();
    }

    /**
     * Creates an object to be published on the STATUS bus by a CanOpenDevice.
     *
     * @return
     */
    public StatusDataPublishedByCanOpenDevice createStatusDataPublishedByCanOpenDevice() {
        StatusDataPublishedByCanOpenDevice status = new StatusDataPublishedByCanOpenDevice(isBooted(), initialized);
        status.setInError(inError);
        status.setErrorRegister(errorRegister);
        status.setErrorHistory(errorHistory);
        if (errorHistory.length > 0) {
            status.setLastError(displayError(errorHistory[0]));
        }
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "refreshGUI", description = "Publish data for the device on the status bus.")
    @Override
    public void publishData() {
        KeyValueData kvd = new KeyValueData(name, createStatusDataPublishedByCanOpenDevice());
        subs.publishSubsystemDataOnStatusBus(kvd);
    }

}
