package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.LO_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is model for the clamps mechanism in the loader. 4 hooks are used for
 * the clamping of the filter in the loader. This hooks are moved all together
 * by the hooksController.
 *
 * Raises alert : - CAN_BUS_TIMEOUT in
 * updateStateWithSensorsToCheckIfActionIsCompleted
 *
 * @author virieux
 */
public class LoaderClamp extends MobileItem implements MovedByEPOSController, ControlledBySensors {

    private final LoaderHook hook1;
    private final LoaderHook hook2;
    private final LoaderHook hook3;
    private final LoaderHook hook4;

    private final ForceSensor forceSensor0;
    private final ForceSensor forceSensor1;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="unclampedStatusSensor")
    private DigitalSensor unclampedStatusSensor;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="underClampedStatusSensor")
    private DigitalSensor underClampedStatusSensor;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="clampedStatusSensor")
    private DigitalSensor clampedStatusSensor;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="overClampedStatusSensor")
    private DigitalSensor overClampedStatusSensor;

    /* a list of hooks to make easier the computation of the LockStatus */
    @LookupField(strategy = Strategy.CHILDREN)
    private LoaderHook[] hooks;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*hooksController")
    private EPOSController hooksController;

    @LookupField(strategy = Strategy.TREE)
    private Loader loader;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*loaderTcpProxy")
    private BridgeToHardware loaderTcpProxy;

    private FcsEnumerations.LockStatus lockStatus = LockStatus.UNKNOWN;

    /**
     * computed from the 2 force sensors forceStatus.
     */
    private FcsEnumerations.LockStatus forceStatus;

    @ConfigurationParameter(description = "Trigger to know when action clamp is completed. "
            + "If force sensor voltage is over this limit then action clamp is completed. Unit = 0.1 Volt")
    private int minClampedVoltage = 25;

    @ConfigurationParameter(description = "timeout in milliseconds : if closing the clamp last more than "
            + "this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForClosingHooks = 60000;

    @ConfigurationParameter(description = "timeout in milliseconds : if closing strongly the clamp last "
            + "more than this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForClampingHooks = 30000;

    @ConfigurationParameter(description = "timeout in milliseconds : if opening the clamp last more than"
            + " this amount of time, then the subsystem goes in ERROR.")
    private int timeoutForOpeningHooks = 60000;

    private static final int TIMEOUT_FOR_MOVING_CLAMP = 60000;

    /**
     * only for simulation and GUI
     */
    @ConfigurationParameter(description = "only for simulation and LoaderClampPanel")
    @Deprecated
    private int targetPositionToOpen = 0;

    @ConfigurationParameter(description = "target encoder absolute value in qc when hooks are CLOSED")
    private int absolutePositionToClose = 340000;

    /**
     * used only by simulation
     */
    @Deprecated
    private int targetPositionToClamp = 345000;

    @ConfigurationParameter(description = "relative position in qc to unclamp when hooks are CLAMPED")
    private int relativePositionToUnclamp = -5000;

    @ConfigurationParameter(description = "current to clamp hooks, in mA")
    private int currentToClamp = 320;

    @ConfigurationParameter(description = "current to open hooks, in mA")
    private int currentToOpen = -150;

    /**
     * Represents the last clamp position read on hooks controller. We want to save
     * this position between two instances of running FCS. It's only to be displayed
     * on the loader GUI. Because when loader is closed on filter and fcs starts,
     * hooks controller homing is not done. So we don't know its position until
     * homing is done, but homing can be done only if there is no filter in the
     * loader.
     *
     */
    @Persist
    private int lastPosition = 0;

    private int positionToReach = 0;

    private volatile boolean homingDone = false;

    private int position;

    private int readCurrent;

    /**
     * Build a new LoaderClampModule with 4 hooks and the parameters to configure
     * the EPOS controller in mode CURRENT and in mode HOMING.
     *
     * @param hook1
     * @param hook2
     * @param hook3
     * @param hook4
     * @param forceSensor0
     * @param forceSensor1
     */
    public LoaderClamp(LoaderHook hook1, LoaderHook hook2, LoaderHook hook3, LoaderHook hook4, ForceSensor forceSensor0,
            ForceSensor forceSensor1) {
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.hooks = new LoaderHook[] { hook1, hook2, hook3, hook4 };
        this.forceSensor0 = forceSensor0;
        this.forceSensor1 = forceSensor1;
    }

    /**
     * Returns controller name.
     *
     * @return
     */
    @Override
    public String getControllerName() {
        return hooksController.getName();
    }

    // for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }

    // for the simulator
    public int getRelativePositionToClose() {
        return absolutePositionToClose;
    }

    // for the simulator
    public int getRelativePositionToUnclamp() {
        return relativePositionToUnclamp;
    }

    // for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }

    // for the simulator
    public int getCurrentToClamp() {
        return currentToClamp;
    }

    // for the simulator
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return position for end user. Do not read again controller.
     *
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Display position for end user."
            + "Do not read again controller.")
    public int getPosition() {
        return position;
    }

    /**
     * Returns true if loader clamp is homingDone.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if homing of loader clamp has been done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     *
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if LockStatus=CLOSED")
    public boolean isClosed() {
        return lockStatus == LockStatus.CLOSED;
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if LockStatus=OPENED")
    public boolean isOpened() {
        return lockStatus == LockStatus.OPENED;
    }

    /**
     * Returns true if LockStatus=CLAMPED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if LockStatus=CLAMPED")
    public boolean isClamped() {
        return lockStatus == LockStatus.CLAMPED;
    }

    /**
     * Returns true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN. That
     * means that one hook is in ERROR status or one or more of the 4 hooks has a
     * different LockStatus than the other.
     *
     * @return true if clamp LockStatus is in ERROR or LockStatus is UNKNOWN.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns true if LockStatus=ERROR or UNKNOWN.")
    @Override
    public boolean isInError() {
        return lockStatus == LockStatus.ERROR;
    }

    public boolean isUnderLoad() {
        return lockStatus == LockStatus.UNDER_CLAMPED;
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and homingDone.
     *
     * @return
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }

    @Override
    public void postStart() {
        FCSLOG.info(name + " postStart");
        if (hooksController.isBooted()) {
            initializeController();
        }
    }

    public void initializeController() {
        try {
            hooksController.initializeAndCheckHardware();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize loader clamp controller", ex);
        }
    }

    /**
     * Check if clamp controller is initialized.
     *
     * @throws RejectedCommandException if clamp is not homingDone
     */
    public void checkInitialized() {
        if (!hooksController.isInitialized()) {
            String msg = getName() + ": clamp is not intialized.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    /**
     *
     */
    public void checkClamped() {
        if (!this.isClamped()) {
            throw new RejectedCommandException(name + " has to be CLAMPED");
        }
    }

    /**
     *
     */
    public void checkUnclamped() {
        if (this.isClamped()) {
            throw new RejectedCommandException(name + " has to be UNCLAMPED");
        }
    }

    /**
     *
     */
    public void checkClosed() {
        if (!this.isClosed()) {
            throw new RejectedCommandException(name + " has to be CLOSED");
        }
    }

    /**
     * Open clamp in order to release filter.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open clamp.", timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void open() {
        loader.updateStateAndCheckSensors();
        loader.checkConditionsForOpeningHooks();
        this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS, timeoutForOpeningHooks);
        homing();
        loader.updateFCSStateToReady();
        this.publishData();
    }

    /**
     * Closes clamp in order to hold the filter. This action closes softly the clamp
     * on the filter. The filter can't fall although it is not tight fermely. When
     * the clamp is closed, the filter is held by loader so autochanger can open its
     * latches and move trucks back without the filter. Closing hooks consists in
     * setting controller mode to PROFILE_POSITION, and going to relative position
     * absolutePositionToClose.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Closes clamp.", timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void close() {

        loader.updateStateAndCheckSensors();
        loader.checkLoaderNotEmpty();
        if (!this.isHomingDone()) {
            throw new RejectedCommandException(getName() + " can't be CLOSED because homing is not done.");
        }
        if (this.isOpened() || lockStatus == LockStatus.INTRAVEL || lockStatus == LockStatus.UNKNOWN) {
            positionToReach = absolutePositionToClose;
            this.executeAction(MobileItemAction.CLOSELOADERHOOKS, timeoutForClosingHooks);


        } else if (this.isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be OPENED before a close action.");
        }
    }

    /**
     * Clamps hooks. When clamp is clamped, the filter is held tightly and carrier
     * can go to STORAGE position safely. Clamping consists in setting controller in
     * mode CURRENT and send currentToClamp to controller.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clamp to hold tightly a filter.", timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void clamp() {

        loader.updateStateAndCheckSensors();

        if (isClosed()) {
            loader.checkLoaderNotEmpty();
            loader.checkConditionsForClampingHooks();
            this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS, timeoutForClampingHooks);
            afterActionClamp();

        } else if (isClamped()) {
            FCSLOG.info(getName() + " is already CLAMPED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be CLOSED before a clamp action.");
        }
    }

    private void afterActionClamp() {
        if (overClampedStatusSensor.isOn()) {
            String msg = " force sensor value is higher than max force limit.";
            this.raiseAlarm(HARDWARE_ERROR, msg);
            throw new FcsHardwareException(name + msg);
        }
        loader.updateFCSStateToReady();
        this.publishData();
    }

    /**
     * unclamp Loader clamp. At the end of this action, hooks are still CLOSED, but
     * doesn't hold tightly filter.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Unclamp filter and return to CLOSED position.", timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void unclamp() {

        loader.updateStateAndCheckSensors();

        if (isClamped() || isUnderLoad()) {
            loader.checkLoaderNotEmpty();
            loader.checkConditionsForUnclampingHooks();
            FCSLOG.info("position=" + position);
            FCSLOG.info("relativePositionToUnclamp=" + relativePositionToUnclamp);
            positionToReach = position + relativePositionToUnclamp;
            FCSLOG.info("positionToReach=" + positionToReach);
            this.executeAction(FcsEnumerations.MobileItemAction.UNCLAMPLOADERHOOKS, timeoutForClampingHooks);

        } else if (isClosed()) {
            FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");

        } else {
            throw new RejectedCommandException(getName() + " has to be CLAMPED before an unclamp action.");
        }
    }

    /**
     * Check if the action is completed. This method is called after
     * updateStateWithSensorsToCheckIfActionIsCompleted where the different new
     * values of current or position or sensors values have been updated.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        /* A loader clamp motion is completed when the position is in a range of */
        /* 10 microns around the target position. */
        boolean actionCompleted = false;
        if (action == MobileItemAction.OPENLOADERHOOKS) {
            actionCompleted = isCurrentReached(currentToOpen) && isOpened();

        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            FCSLOG.info(name + " position = " + position);
            actionCompleted = hooksController.isTargetReached() && isPositionReached(this.positionToReach)
                    && isClosed();
            /**
             * It's tricky to find out when action clamp is completed.
             * isCurrentReached(currentToClamp) doesn't work because it doesn't let enough
             * time for a real clamping. this.readCurrent == currentToClamp works 9/10 times
             * but sometimes this exact value of current is never reached so the command
             * goes into TimeoutException.
             */
        } else if (action == MobileItemAction.CLAMPLOADERHOOKS) {
            actionCompleted = clampedStatusSensor.isOn()
                    && forceSensor0.getVoltage() >= minClampedVoltage && forceSensor1.getVoltage() >= minClampedVoltage;

        } else if (action == MobileItemAction.UNCLAMPLOADERHOOKS) {
            actionCompleted = isPositionReached(this.positionToReach) && !clampedStatusSensor.isOn()
                    && hooksController.isTargetReached();
        }

        return actionCompleted;
    }

    private boolean isCurrentReached(int currentToReach) {
        /* with DELTA = 2 command open is never completed. */
        int DELTA = 5;
        FCSLOG.debug("readCurrent=" + readCurrent + " currentToReached=" + currentToReach);
        return currentToReach - DELTA <= this.readCurrent && this.readCurrent <= currentToReach + DELTA;
    }

    private boolean isPositionReached(int positionToReach) {
        int DELTA = 5;
        FCSLOG.debug("position=" + position + " positionToReached=" + positionToReach);
        return positionToReach - DELTA <= this.position && this.position <= positionToReach + DELTA;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            hooksController.checkFault();
            loader.updateStateWithSensors();
            if (currentAction == MobileItemAction.OPENLOADERHOOKS
                    || currentAction == MobileItemAction.CLAMPLOADERHOOKS) {
                this.readCurrent = this.hooksController.readCurrent();
            }
        } catch (SDORequestException ex) {
            this.raiseWarning(SDO_ERROR, "error in updateStateWithSensorsToCheckIfActionIsCompleted: ", name, ex);
        }
    }

    @Override
    public void startAction(MobileItemAction action) {
        hooksController.checkFault();

        if (action == MobileItemAction.OPENLOADERHOOKS) {
            hooksController.changeMode(CURRENT);
            hooksController.enableAndWriteCurrent((short) this.currentToOpen);

        } else if (action == MobileItemAction.CLOSELOADERHOOKS) {
            hooksController.changeMode(PROFILE_POSITION);
            hooksController.enableAndWriteAbsolutePosition(this.absolutePositionToClose);

        } else if (action == MobileItemAction.CLAMPLOADERHOOKS) {
            hooksController.changeMode(CURRENT);
            hooksController.enableAndWriteCurrent((short) this.currentToClamp);

        } else if (action == MobileItemAction.UNCLAMPLOADERHOOKS) {
            hooksController.changeMode(PROFILE_POSITION);
            hooksController.enableAndWriteRelativePosition(this.relativePositionToUnclamp);
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is STOPPING action " + action.toString() + " within delay " + delay);
        FcsUtils.sleep(500, name);
        // because we don't want to let the controller on power
        this.hooksController.disableOperation();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is STOPPING action " + action.toString() + " within delay " + delay);
        this.hooksController.stopAction();
    }

    private void homing() {
        this.hooksController.defineAbsolutePosition(0);
        // for the GUI
        updatePosition();
        this.homingDone = true;
        this.hooksController.disableOperation();
        this.publishData();
        loader.updateFCSStateToReady();
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in
     * error.
     *
     */
    public void updateStateAndCheckSensors() {
        loader.updateStateWithSensors();
        checkSensors(LO_SENSOR_ERROR, name);
    }

    /**
     * This methods updates the lockStatus and forceStatus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateState() {
        for (LoaderHook hook : hooks) {
            hook.updateState();
        }
        updateForceStatus();

        if (oneHookInError()) {
            this.lockStatus = LockStatus.ERROR;

        } else if (allHooksInState(LockStatus.OPENED)) {
            computeStatusHooksOpened();

        } else if (allHooksInState(LockStatus.CLOSED)) {
            computeStatusHooksClosed();

        } else if (allHooksInState(LockStatus.INTRAVEL)) {
            this.lockStatus = LockStatus.INTRAVEL;

        } else {
            this.lockStatus = LockStatus.UNKNOWN;
        }
        this.position = hooksController.readPosition();

        this.publishData();
    }

    private boolean oneHookInError() {
        boolean bool = false;
        for (LoaderHook hook : hooks) {
            bool = bool || hook.getLockStatus() == LockStatus.ERROR;
        }
        return bool;
    }

    private boolean allHooksInState(LockStatus status) {
        boolean bool = true;
        for (LoaderHook hook : hooks) {
            bool = bool && hook.getLockStatus() == status;
        }
        return bool;
    }

    private void updateForceStatus() {
        if (unclampedStatusSensor.isOn()) {
            forceStatus = LockStatus.UNCLAMPED;
        } else if (underClampedStatusSensor.isOn()) {
            forceStatus = LockStatus.UNDER_CLAMPED;
        } else if (overClampedStatusSensor.isOn()) {
            forceStatus = LockStatus.OVER_CLAMPED;
        } else if (clampedStatusSensor.isOn()) {
            forceStatus = LockStatus.CLAMPED;
        } else {
            forceStatus = LockStatus.ERROR;
        }
    }

    private void computeStatusHooksClosed() {
        /* forceStatus is only to display on GUI */
        if (clampedStatusSensor.isOn()) {
            this.lockStatus = LockStatus.CLAMPED;

        } else {
            this.lockStatus = LockStatus.CLOSED;
        }
    }

    private void computeStatusHooksOpened() {
        /* hooks opened - no more load on hooks */
        if (forceStatus == LockStatus.UNCLAMPED) {
            this.lockStatus = LockStatus.OPENED;

            /* hooks opened - forceSensor CLAMPED : weird */
        } else if (forceStatus == LockStatus.CLAMPED || forceStatus == LockStatus.OVER_CLAMPED) {
            this.lockStatus = LockStatus.ERROR;

            /* hooks opened - too much load on hooks to be opened */
        } else {
            this.lockStatus = forceStatus;
        }
    }

    /**
     * Updates the field readCurrent of the clamp in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamp current in reading controller.")
    @Override
    public void updateCurrent() {
        try {
            this.readCurrent = hooksController.readCurrent();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update clamp position in reading controller.")
    public void updatePosition() {
        try {
            this.position = hooksController.readPosition();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }

    /**
     * To display position for end user. Updates carrier position in reading
     * controller and returns it.
     *
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "To update and display position for end user."
            + "Updates loader clamp position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }

    /**
     * Return a printed list of hardware with initialization information. For debug
     * purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Return a printed list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (homingDone) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/timeoutForClosingHooks=");
        sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");
        sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");
        sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }

    /**
     * Creates and returns the object to be published on the STATUS bus by the
     * LoaderClamp.
     *
     * @return
     */
    public StatusDataPublishedByLoaderClamp createStatusDataPublishedByLoaderClamp() {
        StatusDataPublishedByLoaderClamp status = new StatusDataPublishedByLoaderClamp();
        status.setPosition(position);
        status.setCurrent(readCurrent);
        status.setClampState(lockStatus);
        status.setForceSensorVoltage0(forceSensor0.getVoltage());
        status.setForceSensorVoltage1(forceSensor1.getVoltage());
        status.setForceClampedStatus(clampedStatusSensor.isOn());
        status.setForceUnclampedStatus(unclampedStatusSensor.isOn());
        status.setForceUnderClampedStatus(underClampedStatusSensor.isOn());
        status.setForceOverClampedStatus(overClampedStatusSensor.isOn());
        status.setForceStatus(forceStatus);
        status.setHomingDone(homingDone);
        status.setStatusPublishedByHook1(hook1.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook2(hook2.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook3(hook3.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook4(hook4.createStatusDataPublishedByLoaderHook());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        KeyValueData kvd = new KeyValueData("loaderClamp", this.createStatusDataPublishedByLoaderClamp());
        s.publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (hooksController.isBooted()) {
            hooksController.disableOperation();
        }
        FCSLOG.info(name + " is shutdown.");
    }

}
