package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.LO_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is the model for the loader in the Filter Exchange System. The loader is
 * used to load a filter into the camera or to unload a filter from the camera.
 *
 * CAMERA PROTECTION SYSTEM
 * signal coming from autochanger :
 * AP2: Autochanger at HANDOFF position
 * AF0: Autochanger latches opened
 * AF3: Filter locked in Autochanger
 *
 * @author virieux
 */
public class Loader implements HardwareController, FilterHolder, AlertRaiser, HasLifecycle {

    private static final int TIMEOUT_FOR_MOVING_CARRIER = LoaderCarrier.TIMEOUT_FOR_MOVING_CARRIER;

    @LookupField(strategy = Strategy.TOP)
    private Subsystem s;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private LoaderCarrier carrier;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private LoaderClamp clamp;

    @LookupField(strategy = Strategy.TREE)
    private MainModule main;

    @LookupField(strategy = Strategy.SIBLINGS, pathFilter="autochanger")
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="loaderFilterPresenceSensors")
    private RedondantSensors loaderFilterPresenceSensors;

    @LookupField(strategy = Strategy.CHILDREN, pathFilter="loaderOnCameraSensors")
    private RedondantSensors loaderOnCameraSensors;

    /**
     * CPS signal : autochanger_AP2
     * Autochanger in HANDOFF position
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="acAP2")
    private ComplementarySensors acAP2;

    /**
     * CPS signal : autochanger_AF0
     * Autochanger latches opened
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="acAF0")
    private ComplementarySensors acAF0;

    /**
     * CPS signal : autochanger_AF1
     * Filter detected in Autochanger
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="acAF1")
    private ComplementarySensors acAF1;

    /**
     * CPS signal : autochanger_AF3
     * Filter locked in autochanger
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="acAF3")
    private ComplementarySensors acAF3;

    /**
     * CPS signal : keyLock/permit
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="keyLockSensors")
    private ComplementarySensors keyLockSensors;

    /**
     * CPS signal : enginnering key
     */
    @LookupField(strategy = Strategy.CHILDREN, pathFilter="keyEngSensors")
    private ComplementarySensors keyEngSensors;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*loaderPlutoGateway")
    private PlutoGatewayInterface loaderPlutoGateway;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*hooksController")
    private EPOSController hooksController;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*carrierController")
    private EPOSController carrierController;

    @LookupField(strategy = Strategy.SIBLINGS, pathFilter=LOADER_TCPPROXY_NAME)
    private BridgeToHardware bridge;

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return s;
    }

    /**
     * Returns carrier.
     *
     * @return carrier
     */
    public LoaderCarrier getCarrier() {
        return carrier;
    }

    /**
     * Returns clamp.
     *
     * @return
     */
    public LoaderClamp getClamp() {
        return clamp;
    }

    boolean isCANbusConnected() {
        return bridge.isCanbusConnected();
    }

    /**
     *
     * @return true if hardware (controllers and plutoGateway) is correctly
     *         initialized and checkHardwareStateAndDoHomingIfPossible of the
     *         controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if hardware (controllers and plutoGateway) is correctly initialized"
            + "and homing of the controllers is done.")
    public boolean isInitialized() {
        boolean devicesInitialized = loaderPlutoGateway.isInitialized() && hooksController.isInitialized()
                && carrierController.isInitialized();
        return devicesInitialized && clamp.isHomingDone() && carrier.isInitialized();
    }

    /**
     * Returns the boolean field empty. If the empty boolean is being updated and
     * waits for a response from a sensor, this methods waits until empty is
     * updated. If the field empty is not being updated, it returns immediatly the
     * field empty.
     *
     * @return empty
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if there is no filter in the loader. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return !loaderFilterPresenceSensors.isOn();
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atHandoff is updated. If the field atHandoff is not being updated, it returns
     * immediatly the field atHandoff.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        return loaderOnCameraSensors.isOn();
    }

    /**
     * Return true if the autochanger is holding the filter. This command doesn't
     * read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        return autochanger.isHoldingFilter();
    }

    /**
     * Return true if a filter is present and it is held by the loader clamp.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if a filter is present and it is held by the loader clamp.")
    @Override
    public boolean isHoldingFilter() {
        if (this.isCANbusConnected()) {
            this.updateStateWithSensors();
            this.clamp.updatePosition();
            return this.clamp.isClamped() && !this.isEmpty();
        } else {
            return false;
        }
    }

    /**
     * Return true if loader clamp is openened.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader clamp is openened.")
    @Override
    public boolean isNotHoldingFilter() {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return this.clamp.isOpened();
    }

    /**
     * This command has to be executed after the initialization phase and after the
     * checkHardware command. It can't be automaticaly executed during
     * initialization phase because it's not compliant with the CCS requirement
     * which says that the initialization of the subsystem can't make move hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Attention ! this commands does the homing of the controllers and might move hardware. "
            + "Initialize loader hardware.", alias = "homing")
    public void locateHardware() {
        /*
         * check that plutoGateway and controllers are booted and initialized, and
         * controllers not in fault.
         */
        /* read sensors and update state */
        /* check that position read on carrier controller is confirmed by sensors. */
        carrier.initializeHardware();
        boolean devicesInitialized = loaderPlutoGateway.isInitialized() && hooksController.isInitialized()
                && carrierController.isInitialized();
        if (!devicesInitialized) {
            throw new FailedCommandException(
                    name + " couldn't locate hardware because devices " + "are not initialized.");
        }
        this.updateStateAndCheckSensors();
        /* check that clamp is not empty and closed. */
        checkClosedOnVoid();
        if (isEmpty() || (carrier.isAtHandoff() && autochanger.isHoldingFilter())) {
            /*
             * no filter in the loader /* OR a filter is in the loader, carrier is at
             * HANDOFF and autochanger is holding filter
             */
            clamp.open();

        } else {
            /* a filter is detected by filterPresenceSensor */
            if (!clamp.isClamped()) {
                clamp.clamp();
            }
        }
        this.updateFCSStateToReady();
    }

    public void checkClosedOnVoid() {
        /*
         * If the carrier is empty and the clamp is CLOSED, we can't start the loader
         * subsystem.
         */
        if (this.isEmpty() && clamp.isClosed()) {
            String msg = name + ": carrier is empty and clamp is CLOSED - can't start.";
            this.raiseAlarm(HARDWARE_ERROR, msg, name);
            throw new FcsHardwareException(msg);
        }
    }

    public void checkHaltRequired() {
        if (main.isHaltRequired() || main.isStopRequired()) {
            throw new FailedCommandException(name + ": received HALT or STOP command.");
        }
    }

    /**
     * Check that plutoGateway is booted and initialize plutoGateway. Read sensors
     * and update clamp's lockStatus.
     */
    @Override
    public void postStart() {
        FCSLOG.info(name + " postStart");
        if (loaderPlutoGateway.isBooted()) {
            initializeGateway();
            if (this.isEmpty() && carrier.isAtStorage()) {
                // open clamp to do the homing. To be done whith real signals coming from AC.
                // clamp.open();
            }
        } else if (bridge.isCanbusConnected()) {
            // raise an Alert if loader CANbus is connected and plutoGateway is not booted.
            loaderPlutoGateway.raiseAlarmIfMissing();
        }
    }

    public void initializeGateway() {
        try {
            this.loaderPlutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initialize loaderPlutoGateway", ex);
        }
    }

    /**
     * Check if loader is connected on camera.
     *
     * @throws RejectedCommandException if not connected on camera
     */
    public void checkConnectedOnCamera() {
        if (!isConnectedOnCamera()) {
            throw new RejectedCommandException(name + " Loader not connected - can't execute commands.");
        }
    }

    /**
     * Checks if the loader carrier can move.
     *
     * @param targetPosition
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the carrier can move.")
    public void checkConditionsForCarrierMotion(int targetPosition) {
        FCSLOG.info(name + " checking pre-conditions for carrier motion");

        updateStateAndCheckSensors();

        /* loader can't move if no filter and clamp is closed. */
        checkClosedOnVoid();

        carrier.updatePosition();

        if (this.isEmpty()) {
            FCSLOG.info(name + " no filter. Carrier can move.");
        } else {

            /*
             * if a filter is at HANDOFF in the loader, carrier can't move if it's held both
             * by loader and autochanger
             */
            if (carrier.isAtHandoff() && isAutochangerHoldingFilter() && (clamp.isClamped() || clamp.isClosed())) {
                throw new RejectedCommandException(name + " carrier can't move because a filter is in the loader"
                        + " and it's held by loader AND autochanger.");
            }
            /* motion between STORAGE and ENGAGED position with a filter */
            if (targetPosition <= carrier.getEngagedPosition()
                    && carrier.getPosition() < carrier.getEngagedPosition() + 10) {
                clamp.checkClamped();
                /* motion between HANDOFF and ENGAGED with a filter */
            } else if (targetPosition >= carrier.getEngagedPosition()
                    && carrier.getPosition() > carrier.getEngagedPosition() - 10) {
                clamp.checkUnclamped();

                /* motion from HANDOFF to STORAGE without a filter */
            } else if (carrier.isAtHandoff() && clamp.isOpened()) {
                FCSLOG.info(name + " carrier can move empty to targetPosition=" + targetPosition);

                /* motion between HANDOFF and STORAGE with a filter */
            } else {
                throw new RejectedCommandException("Carrier has to stop first at ENGAGED position");
            }
        }
    }

    @Deprecated // Francis said this condition has to be tested by autochanger not by loader.
    private void checkFilterSafetyAtHandoff() {
        /*
         * if a filter is at HANDOFF in the loader, it has to be held by loader or
         * autochanger.
         */
        if (!isAutochangerHoldingFilter() && clamp.isOpened()) {
            throw new RejectedCommandException(name + " carrier can't move because the filter in the loader is not held"
                    + " neither by loader neither by autochanger. " + "Close loader clamp or autochanger latches. ");
        }
    }

    /**
     * Check if the clamp can be opened.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the clamp can be opened.")
    public void checkConditionsForOpeningHooks() {

        FCSLOG.info(name + " checking pre-conditions for opening hooks");

        updateStateAndCheckSensors();

        if (!this.isEmpty()) {
            /*
             * if a filter is in the loader and carrier is NOT at HANDOFF, clamp can't be
             * open.
             */
            if (!carrier.isAtHandoff()) {
                String msg = name + ": carrier is loaded with a filter but not "
                        + "at handoff position - can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            }
            /* carrier is at Handoff position */
            if (!isAutochangerHoldingFilter()) {
                String msg = name + ": A filter is in the loader but not held by autochanger " + "- can't open clamp.";
                FCSLOG.error(msg);
                throw new RejectedCommandException(msg);
            }
        }
    }

    public void checkConditionsForClampingHooks() {
        if (!carrier.isAtEngaged()) {
            String msg = name + ": carrier is loaded with a filter but not " + "at ENGAGED position - can't clamp.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        if (isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(getName()
                    + " Autochanger is holding filter. Open autochanger latches before " + "clamping loader hooks.");
        }
    }

    public void checkConditionsForUnclampingHooks() {
        if (!carrier.isAtEngaged()) {
            String msg = name + ": carrier is loaded with a filter but not " + "at ENGAGED position - can't unclamp.";
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
        // TODO ask Francis : with new process at HANDOFF this condition should be
        // obsolete.
        if (isAutochangerHoldingFilter()) {
            throw new RejectedCommandException(name + " Autochanger is holding filter. Can't unclamp Loader Clamp.");
        }
    }

    /**
     * Check if the clamp can be closed. Clamp can be close if sensors are not in
     * error and a filter is in the loader.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Check if the clamp can be closed.")
    public void checkLoaderNotEmpty() {
        FCSLOG.info(name + " checking pre-conditions for closing hooks");
        updateStateAndCheckSensors();

        this.clamp.updatePosition();

        if (this.isEmpty()) {
            String msg = name + ": no filter in loader - can't execute close nor clamp command." + clamp.getName();
            FCSLOG.error(msg);
            throw new RejectedCommandException(msg);
        }
    }

    public void updateStateAndCheckSensors() {
        updateStateWithSensors();
        autochanger.updateStateWithSensors();
        checkConnectedOnCamera();
        clamp.checkInitialized();
        clamp.checkSensors(LO_SENSOR_ERROR, clamp.getName());
        carrier.checkSensors(LO_SENSOR_ERROR, carrier.getName());
    }

    /**
     * This methods updates the carrier and clamp state in reading all the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {

        FCSLOG.info(name + " bridge=" + bridge.toString());
        FCSLOG.info(name + " bridge CANbus connected=" + bridge.isCanbusConnected());

        if (this.isCANbusConnected()) {

            loaderPlutoGateway.checkBooted();
            loaderPlutoGateway.checkInitialized();

            this.loaderPlutoGateway.updateValues();
            this.clamp.updateState();

            this.publishData();
            this.clamp.publishData();
            this.carrier.publishData();
        }
    }

    /**
     * Update FCS state and FCS readyness state and publish on the status bus. Check
     * that Loader hardware is ready to be operated and moved. This means that : -
     * all CAN open devices are booted, identified and initialized, -
     * checkHardwareStateAndDoHomingIfPossible has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSStateToReady() {
        if (clamp.isHomingDone() && carrier.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /**
     * move a filter from Storage position or a position between ENGAGED and
     * STORAGE to HANDOFF position.
     *
     * This command executes sequence of actions :
     * - check initial conditions (a filter, at STORAGE)
     * - move carrier to ENGAGED
     * - unclamp filter (not open, just unclamp to release constraint on filter)
     * - move slowly filter to HANDOFF.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Move a filter from STORAGE to HANDOFF. "
            + "This command executes sequence of actions :\n" + "- check initial conditions (a filter, at STORAGE) \n"
            + "- move carrier to ENGAGED with high speed\n" + "- unclamp filter \n"
            + "- move slowly filter to HANDOFF. ", timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveFilterToHandoff() {
        if (this.isEmpty()) {
            throw new RejectedCommandException(name + " is empty. " + "Command moveFilterToHandoff is not relevant.");
        }
        if (!carrier.isAtStorage()) {
            throw new RejectedCommandException(
                    name + " is not at STORAGE. " + "Command moveFilterToHandoff is not relevant.");
        }
        /*
         * checkConditionsForCarrierMotion is called by goToEngaged and again by
         * goToHandOff
         */
        carrier.slowProfileVelocity();
        carrier.raiseProfileAcceleration();
        carrier.raiseProfileDeceleration();
        carrier.goToEngaged();

        /**
         * a little nap to let time to the protection system to update its
         * status, otherwise if we try to unclamp hooks before lpm authorizes we
         * have a UNDER_VOLTAGE error.
         */
        FcsUtils.sleep(50, name);

        /* in ENGAGED position the filter can be UNCLAMPED*/
        clamp.unclamp();

        /* move smoothly from ENGAGED to HANDOFF with filter*/
        carrier.slowProfileAcceleration();
        carrier.slowProfileDeceleration();
        carrier.goToHandOff();
    }

    /**
     * Move a filter from HANDOFF to STORAGE. - checks initial conditions : a filter
     * is at HANDOFF, autochanger holding the filter (latches CLOSED), clamp is
     * CLOSED - move filter slowly to ENGAGED - clamp - move filter to STORAGE with
     * high speed.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move a filter from HANDOFF to STORAGE : move filter slowly to ENGAGED,"
            + "clamp, move filter to STORAGE with high speed",
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveFilterToStorage() {
        if (this.isEmpty()) {
            throw new RejectedCommandException(name + " is empty. " + "Command moveFilterToStorage is not relevant.");
        }
        if (!carrier.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + " is not at HANDOFF. " + "Command moveFilterToStorage is not relevant.");
        }
        if (!clamp.isClosed()) {
            throw new RejectedCommandException(
                    name + " clamp is not CLOSED. " + "Command moveFilterToStorage is not relevant.");
        }
        carrier.slowProfileVelocity();
        carrier.slowProfileAcceleration();
        carrier.slowProfileDeceleration();
        carrier.goToEngaged();

        /**
         * a little nap to let time to the protection system to update its
         * status, otherwise if we try to clamp hooks before lpm authorizes we
         * have a UNDER_VOLTAGE error.
         */
        FcsUtils.sleep(100, name);

        /* At ENGAGED the filter can be CLAMPED */
        clamp.clamp();

        /* move filter to storage whith high acceleration and high decelaration*/
        carrier.raiseProfileAcceleration();
        carrier.raiseProfileDeceleration();
        carrier.goToStorage();
    }

    /**
     * This command can be executed when loader is at Handoff with a filter. The
     * autochanger must hold the filter.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "open clamp and move empty to Storage Position at high speed.",
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void openClampAndMoveEmptyToStorage() {
        if (this.isEmpty()) {
            throw new RejectedCommandException(
                    name + " is empty. " + "Command openClampAndMoveEmptyToStorage is not relevant. Use goToStorage");
        }
        if (!carrier.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + " is not at HANDOFF. " + "Command openClampAndMoveEmptyToStorage is not relevant.");
        }
        clamp.open();
        carrier.raiseProfileVelocity();
        carrier.raiseProfileAcceleration();
        carrier.raiseProfileDeceleration();
        carrier.goToStorage();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "move empty from Storage to Handoff at high speed, and close clamp.",
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void moveEmptyToHandoffAndClose() {
        if (!this.isEmpty()) {
            throw new RejectedCommandException(
                    name + " is not empty. " + "Command moveEmptyToHandoffAndClose is not relevant. Use goToHandoff");
        }
        if (!carrier.isAtStorage()) {
            throw new RejectedCommandException(
                    name + " is not at STORAGE. " + "Command moveEmptyToHandoffAndClose is not relevant.");
        }
        /* approach HANDOFF position with care*/
        carrier.raiseProfileVelocity();
        carrier.raiseProfileAcceleration();
        carrier.slowProfileDeceleration();
        carrier.goToHandOff();
        clamp.close();
    }

    /**
     * Creates an Object to be published on the STATUS bus.
     *
     * @return;
     */
    public StatusDataPublishedByLoader createStatusDataPublishedByLoader() {
        StatusDataPublishedByLoader status = new StatusDataPublishedByLoader();
        status.setFilterPresenceSensorValue(loaderFilterPresenceSensors.isOn());
        status.setFilterPresenceSensorsInError(loaderFilterPresenceSensors.isInError());
        status.setLoaderOnCameraSensorValue(loaderOnCameraSensors.isOn());
        status.setLoaderOnCameraSensorsInError(loaderOnCameraSensors.isInError());
        status.setAf0(acAF0.isOn());
        status.setAf1(acAF1.isOn());
        status.setAf3(acAF3.isOn());
        status.setAp2(acAP2.isOn());
        status.setAf0InError(acAF0.isInError());
        status.setAf1InError(acAF1.isInError());
        status.setAf3InError(acAF3.isInError());
        status.setAp2InError(acAP2.isInError());
        status.setKeyEng(keyEngSensors.isOn());
        status.setKeyLock(keyLockSensors.isOn());
        status.setKeyEngInError(keyEngSensors.isInError());
        status.setKeyLockInError(keyLockSensors.isInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByLoader status = this.createStatusDataPublishedByLoader();
        KeyValueData kvd = new KeyValueData("loaderGeneral", status);

        s.publishSubsystemDataOnStatusBus(kvd);
    }

    /* FilterHolder methods */
    @Override
    public boolean isAtHandoff() {
        return carrier.isAtHandoff();
    }

    @Override
    public boolean isAtStandby() {
        return false;
    }
    /* end of FilterHolder methods */

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Print list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(carrier.printHardwareState());
        sb.append("\n");
        sb.append(clamp.printHardwareState());
        return sb.toString();
    }

    @Override
    public int getFilterID() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

}
