
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * This is a model for the Inclinometer used in CPPM test bench and on whole
 * size prototype.
 *
 * Because for mechanical raisons, the filter changer system X axis has to be
 * horizontal in order to be able to rotate carousel or move autochanger trucks
 * along its rails. So we have 2 inclinometers which give redondant information.
 * We must read permanently the 2 inclinometers to check that inclinaison is
 * correct. We read the inclinaison on the plutoGateway.
 *
 * @author virieux
 */
public class Inclinometer implements SensorPluggedOnDevice {

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subs;

    @LookupName
    private String name = "unset";

    @ConfigurationParameter
    private int byteNumero;

    @ConfigurationParameter
    private String deviceName;

    private double inclinaison;

    /**
     *
     * methods overriden from interface SensorPLuggedOnDevice
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDeviceName() {
        return deviceName;
    }

    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    @Override
    public void updateValue(int newValue) {
        FCSLOG.fine(name + " uptating inclinaison");
        inclinaison = convertVoltage2Degree(newValue / 10);
        publishData();
    }

    /**
     *
     * end of methods overriden from interface SensorPLuggedOnDevice
     */

    /**
     * Returns inclinaison value in degrees. Doesn't read again plutoGateway.
     *
     * @return inclinaison
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Returns inclinaison value in degrees. Doesn't read again CANopen device.")
    public double getInclinaison() {
        return inclinaison;
    }

    /**
     * Concert a voltage in volt between 0.5 and 4.5 volts.
     *
     * @param voltage
     * @return
     */
    private static double convertVoltage2Degree(int voltage) {
        if (voltage < 0 || voltage > 5) {
            throw new IllegalArgumentException(
                    voltage + ": bad value for voltage. " + "Voltage has to be between 0 and 5");
        }
        // voltage is in volt
        double sin_angle = (voltage - 2.5) / 11.52;
        return Math.toDegrees(Math.asin(sin_angle));
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, inclinaison));
    }

}
