
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupName;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * Represents a model for a force sensor.
 * A force sensor is used in loader subsystem to check if the clamp is CLAMPED or not.
 * It's connected to the gateway.
 * The value returned by the force sensor is coded on 1 byte.
 * The analog value is used mainly to be displayed on the GUI.
 * The gateway gives also a status to say if the clamp is CLAMPED or not.
 *
 * @author virieux
 */
public class ForceSensor implements SensorPluggedOnDevice {

    @LookupName
    private String name = "unset";

    /**
     * voltage value read on the gateway.
     * format = decimal
     * unit = mV (millivolts)
     */
    private int voltage;

    /**
     * name of the gateway this sensor is plugged on.
     */
    @ConfigurationParameter
    protected String deviceName;

    /**
     * byte number on the gateway
     */
    @ConfigurationParameter
    protected int byteNumero;

    /**
     * min value for range where hooks are clamped
     * over this value and below maxRangeValue, hooks are clamped.
     * below this value, hooks are not clamped
     */
    @Deprecated
    @ConfigurationParameter(description = "Min value for range where hooks are clamped: "
            + "over this value and below maxRangeValue, hooks are clamped.  Unit = N")
    private int minRangeValue = 15; // 1,5

    /**
     * max value for range where hooks are clamped
     * under this value and over minRangeValue, hooks are clamped
     * above this value hooks are not clamped
     *
     */
    @Deprecated
    @ConfigurationParameter(description = "Max value for range where hooks are clamped:"
            + "below this value and over minRangeValue, hooks are clamped.  Unit = N")
    private int maxRangeValue = 25; // 2,5

    @Deprecated
    @ConfigurationParameter(description = "Below this limit, there is no load on the clamp  so "
            + "clamp is really unclamped."
            + "Over this limit and below minRangeValue, clamp is UNDER_LOAD, neither clamped, "
            + "neither unclamped.  Unit = N")
    private int underLoadLimit = 5; // 0,5

    /**
     * computed from force sensor value.
 - UNCLAMPED : value is lower that underLoadLimit.
 - UNDER_CLAMPED : value is between underLoadLimit and minRangeValue
      filter is not clamped enough so carrier can't be move
      but too much clamped so autochanger can't close its latches.
 - CLAMPED : value is between minRangeValue and maxRangeValue
 - ERROR : value is bigger than maxRangeValue
     */
    @Deprecated
    private LockStatus forceStatus;

    /* from interface SensorPluggedOnDevice */
    @Override
    public String getName() {
        return name;
    }

    /**
     * return byteNumero
     *
     * @return
     */
    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    @Override
    public String getDeviceName() {
        return deviceName;
    }

    @Deprecated
    public int getMinRangeValue() {
        return minRangeValue;
    }

    /**
     * return high limit for force sensor value
     *
     * @return
     */
    @Deprecated
    public int getMaxRangeValue() {
        return maxRangeValue;
    }

    @Deprecated
    public int getUnderLoadLimit() {
        return underLoadLimit;
    }

    @Override
    public void updateValue(int newValue) {
        FCSLOG.info(name + " updating voltage = " + newValue);
        /*
         * voltage sent by sensor is given in 0.1V so it has to be multiplied by 100 to
         * be expressed in mV
         */
        voltage = newValue;
        computeForceStatus();
    }
    /* end from interface SensorPluggedOnDevice */

    /**
     * Returns voltage value read on force sensor in mV.
     *
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Returns force sensor analog Value. Unit= mV")
    public int getVoltage() {
        return voltage;
    }

    /**
     * return force Status
     *
     * @return
     */
    @Deprecated
    public LockStatus getForceStatus() {
        return forceStatus;
    }

    /**
     * update forceStatus from field force
     */
    @Deprecated
    private void computeForceStatus() {
        if (voltage < underLoadLimit) {
            forceStatus = LockStatus.UNCLAMPED;

        } else if (voltage < minRangeValue) {
            forceStatus = LockStatus.UNDER_CLAMPED;

        } else if (voltage < maxRangeValue) {
            forceStatus = LockStatus.CLAMPED;

        } else {
            forceStatus = LockStatus.OVER_CLAMPED;
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/deviceName=");
        sb.append(this.deviceName);
        sb.append("/byteNumero=");
        sb.append(String.valueOf(this.byteNumero));
        sb.append("/voltage=");
        sb.append(String.valueOf(voltage));
        return sb.toString();
    }
}
