
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.TimeUnit;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX_MODE_CURRENT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.UNLOCK_ONLINECLAMPS;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * Prototype online clamps have no encoder so they are opened and closed in
 * CURRENT mode. This class gathers methods and commands which are specific to
 * prototype.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClampsProto extends AutochangerThreeOnlineClamps {

    public AutochangerThreeOnlineClampsProto(AutochangerOnlineClamp onlineClampXminus, AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        super(onlineClampXminus, onlineClampXplus, onlineClampYminus);
    }

    /**
     * Unlock and openClampInCurrentMode the online clamps. initial state =
     * LOCKED final state = OPENED This action is not
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unlock and open the online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockAndOpenClamps() {
        updateStateAndCheckSensors();
        autochanger.checkConditionsForActioningOnlineClamps();
        enableAndCheckControllersBeforeAction(UNLOCK_AND_OPEN_ONLINECLAMPS);
        // TODO test conditions for unlocking clamps
        // ce n'est pas une MobileItemAction parce que ça ne marchait pas.
        if (this.isLocked()) {
            autochanger.checkOnlineClampMotionAllowed();
            this.currentAction = FcsEnumerations.MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS;
            long dur;
            long beginTime = System.currentTimeMillis();
            doUnlock();
            updateStateAndCheckSensors();
            if (!isClosed()) {
                throw new FailedCommandException(name + "couldn't unlock clamps");
            }
            onlineClampYminus.openClampInCurrentMode();
            openClampsXInCurrentMode();
            updateStateAndCheckSensors();
            activateBrakesAndSetCurrentToZero();
            if (!isOpened()) {
                throw new FailedCommandException(name + "couldn't open clamps");
            }
            dur = System.currentTimeMillis() - beginTime;
            FCSLOG.fine(name + " unlockAndOpenClamps duration = " + dur);

        } else if (this.isOpened()) {
            FCSLOG.info(name + " clamps already UNLOCKED and OPENED - nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be LOCKED before unlockAndOpen action.");
        }
    }

    /**
     * Closes and locks the online clamps. This command first release brakes,
     * sends currentToClose to the 3 clamps controllers and then increases
     * progressively current until currentToClamp is reached for every clamps
     * controllers.
     *
     *
     * state before this action : OPENED or CLOSED (if CLOSED, nothing has to be
     * done) state after this action : LOCKED
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Closes and locks the online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeAndLockClampsInModeCurrent() {
        updateStateAndCheckSensors();
        // Test conditions for closing clamps are checked in startAction
        if (isOpened()) {
            long dur;
            long beginTime = System.currentTimeMillis();
            onlineClampYminus.closeClampInCurrentMode();
            /* See startAction */
            this.executeAction(FcsEnumerations.MobileItemAction.CLOSE_AND_LOCK_ONLINECLAMPS_MODE_CURRENT, timeoutForLockingClamps);
            dur = System.currentTimeMillis() - beginTime;
            FCSLOG.fine(name + " closeAndLockClamps duration = " + dur);

        } else if (isLocked()) {
            FCSLOG.info(name + " is already LOCKED - nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be OPENED before a closeAndLockClamps");
        }
    }

    /**
     * Closes the 3 clamps in mode current. initial state = OPENED final state =
     * CLOSED. for prototype only.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Close the 3 online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsInModeCurrent() {

        updateStateAndCheckSensors();
        // TODO test conditions for closing clamps
        if (isOpened()) {
            /* See startAction */
            onlineClampYminus.closeClampInCurrentMode();
            this.executeAction(FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPS_MODE_CURRENT,
                    timeoutForLockingClamps);

        } else if (isClosed()) {
            FCSLOG.info(name + " clamps already CLOSED nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be unlocked before.");
        }
    }

    @Command(type = Command.CommandType.ACTION, description = "close clamps X", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsXInModeCurrent() {
        checkParametersToCloseClampsX();
        this.currentAction = FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPS;
        onlineClampXminus.enableAndCheckControllerBeforeAction(CLOSE_ONLINECLAMPS);
        onlineClampXminus.getController().doReleaseBrake();
        onlineClampXplus.enableAndCheckControllerBeforeAction(CLOSE_ONLINECLAMPS);
        onlineClampXplus.getController().doReleaseBrake();
        onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getInitialCurrentToClose());
        onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getInitialCurrentToClose());
        /* tempo de waitTime et rampe de initialCurrentToClose à finalCurrentToClose */
        FcsUtils.sleep(onlineClampXminus.getWaitTimeToClose(), name);
        this.executeCurrentRampToCloseClampsX();
        this.updateStateAndCheckSensors();
        onlineClampXminus.getController().activateBrake();
        onlineClampXplus.getController().activateBrake();
        onlineClampXminus.getController().writeCurrent((short) 0);
        onlineClampXplus.getController().writeCurrent((short) 0);
    }

    /**
     * Opens the 3 clamps with a small pressure. initial state = CLOSED final
     * state = OPENED. For prototype only. For final pruducts AC1 and AC2 see
     * openClamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Opens the 3 online clamps with a small pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsInCurrentMode() {
        updateStateAndCheckSensors();
        // TODO test conditions for opening clamps
        if (isClosed()) {
            checkReadyForAction();
            checkParametersToOpenClampsX();
            this.currentAction = FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPS_MODE_CURRENT;
            onlineClampYminus.openClampInCurrentMode();
            openClampsXInCurrentMode();

        } else if (isOpened()) {
            FCSLOG.info(name + " clamps already OPENED. Nothing to do.");
            throw new RejectedCommandException(name + " clamps already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be closed before.");
        }
    }

    /**
     * opens the 2 clampsX in mode CURRENT. for prototype only.
     */
    @Command(type = Command.CommandType.ACTION, description = "open clampXs. for proto only.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsXInCurrentMode() {
        checkParametersToOpenClampsX();
        this.currentAction = OPEN_ONLINECLAMPSX_MODE_CURRENT;
        onlineClampXminus.enableAndCheckControllerBeforeAction(OPEN_ONLINECLAMPSX_MODE_CURRENT);
        onlineClampXplus.enableAndCheckControllerBeforeAction(OPEN_ONLINECLAMPSX_MODE_CURRENT);
        /* At the beginning clampsX are CLOSED */
        onlineClampXminus.sendCurrentToControllerAndSaveValue(onlineClampXminus.getFinalCurrentToClose());
        onlineClampXplus.sendCurrentToControllerAndSaveValue(onlineClampXplus.getFinalCurrentToClose());
        FcsUtils.sleep(500, name);
        onlineClampXminus.getController().doReleaseBrake();
        onlineClampXplus.getController().doReleaseBrake();
        this.executeAction(currentAction, timeoutForUnlockingClamps);
    }

    private void executeCurrentRampToCloseClampsX() {
        writeCurrentRampToCloseClampsX();
        waitForEndOfCurrentRamp(timeoutForLockingClamps);
    }

    private void executeCurrentRampToOpenClampsX() {
        FCSLOG.info(name + " executeCurrentRampToOpenClampsX");
        writeCurrentRampToOpenClampsX();
        waitForEndOfCurrentRamp(timeoutForUnlockingClamps);
    }

    private void writeCurrentRampToCloseClampsX() {
        writeCurrentRampToClampsX(
                onlineClampXminus.getInitialCurrentToClose(),
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getIncrementCurrentToClose(),
                onlineClampXplus.getInitialCurrentToClose(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getIncrementCurrentToClose(),
                maxTimeToCloseClampsX
        );
    }

    /**
     * sends current alternativaly to clamps X- and X+ with a ramp of current
     * from initialCurrent to finalCurrent. Uses a scheduler.
     */
    private void writeCurrentRampToOpenClampsX() {
        FCSLOG.info(name + " writeCurrentRampToOpenClampsX");
        writeCurrentRampToClampsX(
                onlineClampXminus.getFinalCurrentToClose(),
                onlineClampXminus.getCurrentToOpen(),
                onlineClampXminus.getIncrementCurrentToOpen(),
                onlineClampXplus.getFinalCurrentToClose(),
                onlineClampXplus.getCurrentToOpen(),
                onlineClampXplus.getIncrementCurrentToOpen(),
                maxTimeToOpenClampsX
        );
    }

    private void writeCurrentRampToClampsX(int initialValueXminus, int finalValueXminus, int incrementCurrentXminus,
            int initialValueXplus, int finalValueXplus, int incrementCurrentXplus, int maxTime) {
        int stepHeightXminus = FcsUtils.getSignedStepHeight(initialValueXminus, finalValueXminus,
                incrementCurrentXminus);
        int stepHeightXplus = FcsUtils.getSignedStepHeight(initialValueXplus, finalValueXplus, incrementCurrentXplus);
        final int nbStepXminus = Math.abs((finalValueXminus - initialValueXminus) / incrementCurrentXminus);
        final int nbStepXplus = Math.abs((finalValueXplus - initialValueXplus) / incrementCurrentXplus);
        final int period = maxTime / Math.min(nbStepXminus, nbStepXplus);
        FCSLOG.finest("############################################################");
        FCSLOG.finest("initialValue for clampXminus=" + initialValueXminus);
        FCSLOG.finest("finalValue for clampXminus=" + finalValueXminus);
        FCSLOG.finest("initialValue for clampXplus=" + initialValueXplus);
        FCSLOG.finest("finalValue for clampXplus=" + finalValueXplus);
        FCSLOG.finest("period=" + period);
        FCSLOG.finest("nbStepXminus=" + nbStepXminus);
        FCSLOG.finest("stepHeightXplus=" + stepHeightXplus);
        FCSLOG.finest("nbStepXminus=" + nbStepXminus);
        FCSLOG.finest("nbStepXplus=" + nbStepXplus);
        FCSLOG.finest("############################################################");

        final Runnable currentRamp = new Runnable() {
            private int newCurrentValueXminus = initialValueXminus;
            private int newCurrentValueXplus = initialValueXplus;
            private boolean finalValueReachedXminus = false;
            private boolean finalValueReachedXplus = false;

            @Override
            public void run() {
                newCurrentValueXminus = FcsUtils.computeNewCurrentValue(stepHeightXminus, newCurrentValueXminus,
                        finalValueXminus);
                newCurrentValueXplus = FcsUtils.computeNewCurrentValue(stepHeightXplus, newCurrentValueXplus,
                        finalValueXplus);

                try {
                    if (!finalValueReachedXminus) {
                        onlineClampXminus.sendCurrentToControllerAndSaveValue(newCurrentValueXminus);
                    }
                    if (!finalValueReachedXplus) {
                        onlineClampXplus.sendCurrentToControllerAndSaveValue(newCurrentValueXplus);
                    }

                } finally {
                    finalValueReachedXminus = newCurrentValueXminus == finalValueXminus;
                    finalValueReachedXplus = newCurrentValueXplus == finalValueXplus;
                }

                if (finalValueReachedXminus && finalValueReachedXplus) {
                    cancelCurrentRamp();
                }
            }
        };
        this.currentRampHandle = scheduler.scheduleAtFixedRate(currentRamp, 0, period, TimeUnit.MILLISECONDS);
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {
        enableAndCheckControllersBeforeAction(action);
        autochanger.checkConditionsForActioningOnlineClamps();
        autochanger.checkOnlineClampMotionAllowed();
        autochanger.increaseCurrentMonitoring();
        if (action == CLOSE_AND_LOCK_ONLINECLAMPS) {
            /*
             * for action CLOSE_AND_LOCK_ONLINECLAMPS, first we closeClampInCurrentMode and then we lock, the
             * locking is done in postAction
             */
            closeClamps();
            doLock();

        } else if (action == FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPS_MODE_CURRENT) {
            closeClampsXInModeCurrent();

        } else if (action == FcsEnumerations.MobileItemAction.UNLOCK_AND_OPEN_ONLINECLAMPS) {
            /*
             * clamps are LOCKED with a high pressure (currentToClamp previously sent to
             * controllers)
             */
            /*
             * to avoid an elastic return we send currentToClamp BEFORE the brakes are
             * released.
             */

            openClampsXInCurrentMode();

        } else if (action == UNLOCK_ONLINECLAMPS) {
            /*
             * clamps are LOCKED with a high pressure (currentToClamp previously sent to
             * controllers)
             */
            /*
             * to avoid an elastic return we send currentToClamp BEFORE the brakes are
             * released.
             */
            doUnlock();

        } else if (action == FcsEnumerations.MobileItemAction.LOCK_ONLINECLAMPS) {
            /* clamps are CLOSED */
            doLock();

        } else if (action == FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX) {
            onlineClampXminus.getController().enableAndReleaseBrake();
            onlineClampXplus.getController().enableAndReleaseBrake();
            onlineClampXminus.getController().writeTargetPosition(onlineClampXminus.getTargetPositionToOpen());
            onlineClampXplus.getController().writeTargetPosition(onlineClampXplus.getTargetPositionToOpen());
            onlineClampXminus.getController().writeControlWord(0x3F);
            onlineClampXplus.getController().writeControlWord(0x3F);

        } else if (action == FcsEnumerations.MobileItemAction.CLOSE_ONLINECLAMPSX) {
            onlineClampXminus.getController().enableAndReleaseBrake();
            onlineClampXplus.getController().enableAndReleaseBrake();
            onlineClampXminus.getController().writeTargetPosition(onlineClampXminus.getTargetPositionToClose());
            onlineClampXplus.getController().writeTargetPosition(onlineClampXplus.getTargetPositionToClose());
            onlineClampXminus.getController().writeControlWord(0x3F);
            onlineClampXplus.getController().writeControlWord(0x3F);

        } else if (action == FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX_MODE_CURRENT) {
            this.executeCurrentRampToOpenClampsX();

        } else if (action == CLOSE_AND_LOCK_ONLINECLAMPS_MODE_CURRENT) {
            /*
             * for action CLOSE_AND_LOCK_ONLINECLAMPS, first we closeClampInCurrentMode and then we lock, the
             * locking is done in postAction
             */
            closeClampsXInModeCurrent();
            doLock();

        } else {
            throw new IllegalArgumentException(name + " invalid action:" + action);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        autochanger.decreaseCurrentMonitoring();
        if (action == FcsEnumerations.MobileItemAction.OPEN_ONLINECLAMPSX_MODE_CURRENT) {
            onlineClampXminus.getController().activateBrake();
            onlineClampXplus.getController().activateBrake();
            onlineClampXminus.getController().writeCurrent(0);
            onlineClampXplus.getController().writeCurrent(0);

        } else if (action == OPEN_ONLINECLAMPSX || action == CLOSE_ONLINECLAMPSX) {
            onlineClampXminus.getController().stopAction();
            onlineClampXplus.getController().stopAction();
            onlineClampXminus.getController().activateBrake();
            onlineClampXplus.getController().activateBrake();

        } else {
            /* activate brakes on the 3 clamps controllers and set current to 0 */
            activateBrakesAndSetCurrentToZero();
        }
    }

}
