
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This class is the Main Module when autochanger is in standalone mode.
 *
 * @author virieux
 */
public class AutochangerMain extends MainModule {

    @LookupField(strategy = Strategy.CHILDREN)
    private Autochanger autochanger;

    private FakeFilterHolder carousel;
    private FakeFilterHolder loader;

    /**
     * Build a AutochangerMainModule.
     *
     * @param bridge
     */
    public AutochangerMain(BridgeToHardware bridge) {
        super(bridge);
    }

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make AutochangerGUI listen to my
         * subsystem (cf GeneralGUI.connecting())
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.autochanger", "autochanger");
    }

    /**
     * Update autochanger state in reading the sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update autochanger state in reading the sensors.")
    @Override
    public void updateStateWithSensors() {
        autochanger.updateStateWithSensors();
        carousel.updateStateWithSensors();
        loader.updateStateWithSensors();
    }

    /**
     * Check that Autochanger hardware is ready to be operated and moved. This means
     * that : - all CAN open devices are booted, identified and initialized, -
     * homing has been done on the controllers. This updates the FCS state and FCS
     * readyness state and publishes on the status bus.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    @Override
    public void updateFCSStateToReady() {
        if (autochanger.isInitialized() && isInState(AlertState.ALARM)) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY, FilterReadinessState.READY);
        }
    }

}
