
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByPlutoGateway;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This represents the gateway to the pluto PLC involved in the camera
 * protection system. The sensors are read through this gateway. We read the
 * data from the gateway byte by byte. Usually we read 2 bytes at one time. This
 * class has to be extended to fit the real hardware control software needs and
 * the simulator.
 *
 * @author virieux
 */
public abstract class PlutoGateway extends CanOpenDevice implements PlutoGatewayInterface {

    protected int[] readValues;

    /**
     * map of DigitalSensors attached to this gateway.
     */
    @LookupField(strategy = LookupField.Strategy.TREE)
    protected Map<String, SensorPluggedOnDevice> sensorsMap = new HashMap<>();

    /**
     * Build a new PlutoGatewayModule.
     * 
     * @param nodeID
     * @param serialNB
     */
    public PlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
        this.readValues = new int[16];
    }

    @Override
    public void init() {
        FCSLOG.fine(name + " =====> initialization of my sensors map");
        StringBuilder sb = new StringBuilder();
        for (Iterator<SensorPluggedOnDevice> it = sensorsMap.values().iterator(); it.hasNext();) {
            SensorPluggedOnDevice sensor = it.next();
            if (sensor.getDeviceName().equals(name)) {
                sb.append(sensor.getName());
                sb.append(" ====> IS MY SENSOR ");
                sb.append(sensor.toString());
                sb.append("#\n");
            } else {
                it.remove();
            }
        }
        FCSLOG.fine(sb.toString());
    }

    /**
     * Read new readValues on the device plutoGateway and updates field readValues.
     * 
     * @throws FcsHardwareException
     */
    @Override
    public void updateValues() {
        this.readValues = readNewValues();
        FCSLOG.info(name + " updating my sensors values.");
        sensorsMap.values().stream().forEach((SensorPluggedOnDevice sensor) -> {
            sensor.updateValue(this.readValues[sensor.getByteNumero()]);
        });
        publishData();
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        for (int i = 0; i < readValues.length; i++) {
            sb.append("/byte");
            sb.append(i);
            sb.append('=');
            sb.append(String.valueOf(this.readValues[i]));
        }
        return sb.toString();
    }

    /**
     * Create an object with status data to be published on the status bus.
     * 
     * @return
     */
    public StatusDataPublishedByPlutoGateway createStatusDataPublishedByPlutoGatewayModule() {
        return new StatusDataPublishedByPlutoGateway(isBooted(), isInitialized(), readValues);
    }

    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(
                new KeyValueData(getName(), createStatusDataPublishedByPlutoGatewayModule()));
    }
}
