
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MIN;
import org.lsst.ccs.subsystems.fcs.drivers.CanCBXAI814;
import org.lsst.ccs.subsystems.fcs.simulation.AutochangerTruckPositionListener;
import org.lsst.ccs.subsystems.fcs.simulation.SimuCanOpenADC;
import org.lsst.ccs.subsystems.fcs.simulation.SimuCanOpenSensor14bits;

/**
 * To simulate an analogic sensor which indicates if a Carousel Clamp is LOCKED
 * or not. For the carousel, we read the new values of the lock sensors by PDO.
 * This class provides methods to simulate that clamp is locked, or clamp is
 * unlocked.
 *
 * @author virieux
 */
public class SimuCarouselClampLockSensor extends SimuCanOpenSensor14bits implements AutochangerTruckPositionListener {

    int acTrucksStandbyPosition = 992000;

    /**
     * Build a new SimuCarouselClampLockSensor
     *
     * @param adc
     * @param numOfAnalogInput
     */
    public SimuCarouselClampLockSensor(CanCBXAI814 adc, int numOfAnalogInput) {
        super(adc, numOfAnalogInput);
    }

    @Override
    public void start() {
        simulateClampIsLOCKED();
    }

    /**
     * We observe SimuAutochangerPlutoGateway.
     *
     * When it changes trucks position, we are notified, we can then simulate that
     * carousel is LOCKED or not. In phase OPERATIONAL, when this sensor is at
     * STANDBY and autochanger trucks position has changed, and autochanger trucks
     * are holding a filter, then the state of this simulated clamp lock sensor
     * changes.
     *
     * @param newPos the new trucks position
     */
    @Override
    public void onTruckPositionChange(int newPos) {
        FCSLOG.finest(name + " :old Value=" + this.printValue());
        if (this.isAtStandby()) {
            FCSLOG.finest(name + " is AT STANDBY.");
        }
        if (agentStateService.isInState(PhaseState.OPERATIONAL) && this.isAtStandby()
                && autochanger.isHoldingFilter()) {

            if (newPos < acTrucksStandbyPosition - 500) {
                simulateClampIsUNLOCKED();

            } else if (newPos == acTrucksStandbyPosition) {
                simulateClampIsLOCKED();
            }
        }
        FCSLOG.finest(name + " :new Value=" + this.printValue());
    }

    /**
     * Simulate that the clamp is locked.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsLOCKED() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, clamp.getLockSensorValueB(), clamp.getLockSensorMaxValue());
    }

    /**
     * Simulate that the clamp is unlocked.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsUNLOCKED() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, SENSOR14BITS_MIN, clamp.getLockSensorValueA());
    }

}
