
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerLatch;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This is a model for a latch. A latch is what holds a filter in the trucks of
 * the autochanger. There is 2 latches : one on each side of the filter. The
 * state of the latches are monitoring by digital sensors : one sensor to know
 * if a filter is in the latch or not and 2 status sensors to know if the latch
 * is EngageBrakeed or ReleaseBrakeed. If the 2 status sensors are inconsistent,
 * the latch is in error. This class is used in single-filter-test. For the
 * final product the class AutochangerLatchModule is used.
 *
 * @author virieux
 */
public class SftFilterLatch extends MobileItem {

    @LookupField(strategy = Strategy.TREE)
    private MainModule mainModule;

    private boolean locked;
    private boolean unlocked;
    private boolean inError;

    private FilterPresenceStatus presenceStatus = FilterPresenceStatus.UNKNOWN;
    private LockStatus lockStatus = LockStatus.UNKNOWN;

    protected final LatchActuator latchActuator;
    protected DigitalSensor filterPresenceSensor;
    protected DigitalSensor lockSensor;
    protected DigitalSensor unlockSensor;
    private final CompactIO filterSensorsDIO;

    @ConfigurationParameter(description = "timeout in milliseconds : if closing the autochanger latches last more"
            + " than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForOpening = 10000;

    @ConfigurationParameter(description="timeout in milliseconds : if opening the autochanger latches last more"
            + " than this amount of time, then the subsystem goes in ERROR.")
    private long timeoutForClosing = 10000;

    // Used because we have to wait for the update from the sensors to know the
    // state
    // of the latch
    private final Condition stateUpdated = lock.newCondition();

    /**
     * This is used when we update the latch state with the values returned by the
     * sensors.
     */
    protected volatile boolean updatingState = false;



    /**
     * Build a new SftFilterLatchModule
     *
     * @param filterSensorsDIO
     * @param lockSensor
     * @param unlockSensor
     * @param filterPresenceSensor
     * @param latchActuator
     */
    public SftFilterLatch(CompactIO filterSensorsDIO, DigitalSensor lockSensor, DigitalSensor unlockSensor,
            DigitalSensor filterPresenceSensor, LatchActuator latchActuator) {
        this.filterSensorsDIO = filterSensorsDIO;
        this.lockSensor = lockSensor;
        this.unlockSensor = unlockSensor;
        this.filterPresenceSensor = filterPresenceSensor;
        this.latchActuator = latchActuator;
    }

    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    // for FCSUtils
    public DigitalSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    // for FCSUtils
    public LatchActuator getLatchActuator() {
        return latchActuator;
    }

    // for FCSUtils
    public DigitalSensor getLockSensor() {
        return lockSensor;
    }

    // for FCSUtils
    public DigitalSensor getUnlockSensor() {
        return unlockSensor;
    }

    // for FCSUtils
    public boolean isControllerInFault() {
        return false;
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS
     */
    /**
     * ***********************************************************************************************
     */

    /**
     * Returns true if hardware is connected and ready.
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return mainModule.allDevicesBooted();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.OPEN)) {
            return this.getLockStatus().equals(LockStatus.UNLOCKED);
        } else if (action.equals(MobileItemAction.CLOSE)) {
            return this.getLockStatus().equals(LockStatus.LOCKED);
        } else {
            throw new IllegalArgumentException("Action on latch must be OPEN or CLOSE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        this.updateStateWithSensors();
    }

    /**
     * Reads latch sensors and updates latch state.
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads latch sensors and updates latch state")
    public void updateStateWithSensors()  {
        filterSensorsDIO.updateValue();

        int filterSensorsValue = filterSensorsDIO.getValue();

        this.updateState(filterSensorsValue);
    }

    @Override
    public void startAction(MobileItemAction action) throws
            FcsHardwareException {
        if (action.equals(MobileItemAction.OPEN)) {
            this.latchActuator.open();
        } else if (action.equals(MobileItemAction.CLOSE)) {
            this.latchActuator.close();
        } else {
            throw new IllegalArgumentException(name + "Action on latch must be OPEN or CLOSE");
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public LockStatus getLockStatus() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": getLockStatus was interrupted while waiting for update.");
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }

    }

    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }

    /**
     * Return the filter presence status. Wait if the state is beeing updated.
     * @return
     */
    public FilterPresenceStatus getPresenceStatus() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": getPresenceStatus was interrupted while waiting for update.");
                }

            }
            return presenceStatus;

        } finally {
            lock.unlock();
        }

    }

    /**
     * Open the latch if the carousel is holding the filter.
     *
     * @return
     * @throws RejectedCommandException
     * @throws                          org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Open latch")
    public String open() {

        this.updateStateWithSensors();

        if (!this.isLocked()) {
            return getName() + "is already OPEN";
        }

        Carousel carousel = (Carousel) s.getComponentLookup().getComponentByPath("carousel");

        if (!carousel.isHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    name + "CANNOT OPEN LATCH at STANDBY : CAROUSEL is NOT HOLDING THE FILTER");
        }

        this.executeAction(MobileItemAction.OPEN, this.timeoutForOpening);

        this.latchActuator.powerOff();

        return getName() + " is OPEN";

    }

    /**
     * Close the latch : turn on the actuator, read the sensors and wait until the
     * latch is closed. If the latch is not closed during the delai :
     * timeoutForClosing throws an Exception.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, description = "Close latch")
    public String close() {

        this.updateStateWithSensors();

        if (this.isLocked()) {
            return getName() + "is already CLOSED";
        }

        this.executeAction(MobileItemAction.CLOSE, this.timeoutForClosing);

        this.latchActuator.powerOff();

        return getName() + " is CLOSED";

    }

    /**
     *
     * @return true if the latch is closed.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the latch is closed")
    public boolean isLocked() {
        return this.getLockStatus().equals(LockStatus.LOCKED);
    }

    /**
     * Update Latch State.
     * @param value
     */
    public void updateState(int value) {
        lock.lock();
        try {
            updatingState = true;
            this.filterPresenceSensor.updateValue(value);
            this.lockSensor.updateValue(value);
            this.unlockSensor.updateValue(value);

            if (this.filterPresenceSensor.isOn()) {
                this.presenceStatus = FilterPresenceStatus.ENGAGED;
            } else {
                this.presenceStatus = FilterPresenceStatus.NOFILTER;
            }
            locked = this.lockSensor.isOn();
            unlocked = this.unlockSensor.isOn();
            inError = locked && unlocked;

            if (inError) {
                lockStatus = LockStatus.ERROR;
            } else if (locked) {
                lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = LockStatus.UNLOCKED;
            } else {
                lockStatus = LockStatus.UNKNOWN;
            }

        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }

        this.publishData();

    }

    /**
     * This methods is used in the single-filter-test.
     * @return
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setLockSensorValue(getLockSensor().isOn());
        status.setUnlockSensorValue(getUnlockSensor().isOn());
        status.setFilterPresenceSensorValue(getFilterPresenceSensor().isOn());
        status.setLockStatus(getLockStatus());
        status.setControllerInFault(isControllerInFault());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = this.createStatusDataPublishedByLatch();
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/ lock sensor=");
        sb.append(this.lockSensor.getName());
        sb.append("/ unlock sensor=");
        sb.append(this.unlockSensor.getName());
        sb.append("/ filter presence sensor=");
        sb.append(this.filterPresenceSensor.getName());
        return sb.toString();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        throw new UnsupportedOperationException("Not supported yet.");
    }

}
