
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.Autochanger;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.AutochangerOnlineClamp;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.ComplementarySensors;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FilterIdentificator;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * To simulate the plutoGateway on the autochanger.
 * @author virieux
 */
public class SimuAutochangerPlutoGateway extends SimuPlutoGateway implements SimuFilterIDPlutoGatewayInterface {

    /* To read filterID hall effect sensors.*/

    @LookupField(strategy=Strategy.TREE)
    private FilterIdentificator filterIdentificator;

    @LookupField(strategy=Strategy.TREE, pathFilter = ".*carousel")
    private FilterHolder carousel;

    @LookupField(strategy=Strategy.TREE)
    private Autochanger autochanger;
    //trucks controllers
    private AutochangerTwoTrucks trucks;
    private SimuAutochangerLinearRailController linearRailMasterController; //(X-)
    private SimuAutochangerLinearRailController linearRailSlaveController; //(X+)

    //online clamps
    private AutochangerOnlineClamp onlineClampXminus;
    private AutochangerOnlineClamp onlineClampXplus;
    private AutochangerOnlineClamp onlineClampYminus;
    private SimuCanOpenStrainGauge onlineStrainGauge;

    //loader connected sensors
    private ComplementarySensors loaderConnectedSensors;

    //local protection status
    private DigitalSensor lpmLinearRailMasterStatus;
    private DigitalSensor lpmLinearRailSlaveStatus;
    private DigitalSensor lpmOnlineClampsStatus;
    private DigitalSensor lpmLatchesStatus;

    //latches sensors
    private ComplementarySensors closeSensorsLatchXminus;
    private ComplementarySensors closeSensorsLatchXplus;
    private ComplementarySensors openSensorsLatchXminus;
    private ComplementarySensors openSensorsLatchXplus;
    private ComplementarySensors filterEngagedSensorsLatchXminus;
    private ComplementarySensors filterEngagedSensorsLatchXplus;

    //trucks position sensors
    private ComplementarySensors handoffPositionSensorsXminus;
    private ComplementarySensors onlinePositionSensorsXminus;
    private ComplementarySensors standbyPositionSensorsXminus;
    private ComplementarySensors handoffPositionSensorsXplus;
    private ComplementarySensors onlinePositionSensorsXplus;
    private ComplementarySensors standbyPositionSensorsXplus;

    // Truck position listeners
    @LookupField(strategy=Strategy.TREE)
    private final Set<AutochangerTruckPositionListener> truckListeners = new HashSet<>();

    /**
     * Build a new SimuAutochangerPlutoGateway
     * @param nodeID
     * @param serialNB
     */
    public SimuAutochangerPlutoGateway(
            int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    @Override
    public List<DigitalSensor> getFilterIDSensors() {
        return filterIdentificator.getFilterIDSensors();
    }

    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        //lpm status
        this.lpmLinearRailMasterStatus = (DigitalSensor) lookup.getComponentByPath("lpmLinearRail1Status");
        this.lpmLinearRailSlaveStatus = (DigitalSensor) lookup.getComponentByPath("lpmLinearRail2Status");
        this.lpmOnlineClampsStatus = (DigitalSensor) lookup.getComponentByPath("lpmOnlineClampsStatus");
        this.lpmLatchesStatus = (DigitalSensor) lookup.getComponentByPath("lpmLatchesStatus");

        //trucks
        this.loaderConnectedSensors = (ComplementarySensors) lookup.getComponentByPath("loaderConnectedSensors");
        this.linearRailMasterController = (SimuAutochangerLinearRailController) lookup.getComponentByPath("linearRailMasterController");
        this.linearRailSlaveController = (SimuAutochangerLinearRailController) lookup.getComponentByPath("linearRailSlaveController");
        this.trucks = (AutochangerTwoTrucks) lookup.getComponentByPath("autochangerTrucks");

        //latches
        this.closeSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByPath("closeSensorsLatchXminus");
        this.openSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByPath("openSensorsLatchXminus");
        this.filterEngagedSensorsLatchXminus = (ComplementarySensors) lookup.getComponentByPath("filterEngagedSensorsLatchXminus");
        this.closeSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByPath("closeSensorsLatchXplus");
        this.openSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByPath("openSensorsLatchXplus");
        this.filterEngagedSensorsLatchXplus = (ComplementarySensors) lookup.getComponentByPath("filterEngagedSensorsLatchXplus");



        //ONLINE clamps
        this.onlineClampXminus = (AutochangerOnlineClamp)
                lookup.getComponentByPath("onlineClampXminus");
        this.onlineClampXplus = (AutochangerOnlineClamp)
                lookup.getComponentByPath("onlineClampXplus");
        this.onlineClampYminus = (AutochangerOnlineClamp)
                lookup.getComponentByPath("onlineClampYminus");

        this.onlineStrainGauge = (SimuCanOpenStrainGauge) lookup.getComponentByPath("onlineStrainGauge");

        //Trucks
        this.handoffPositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByPath("handoffPositionSensorsXminus");
        this.handoffPositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByPath("handoffPositionSensorsXplus");
        this.onlinePositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByPath("onlinePositionSensorsXminus");
        this.onlinePositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByPath("onlinePositionSensorsXplus");
        this.standbyPositionSensorsXminus = (ComplementarySensors)
                lookup.getComponentByPath("standbyPositionSensorsXminus");
        this.standbyPositionSensorsXplus = (ComplementarySensors)
                lookup.getComponentByPath("standbyPositionSensorsXplus");
    }

    protected void simuLPMStatusOn() {
        replaceSensorValue(this.lpmLinearRailMasterStatus, true);
        replaceSensorValue(this.lpmLinearRailSlaveStatus, true);
        replaceSensorValue(this.lpmOnlineClampsStatus, true);
        replaceSensorValue(this.lpmLatchesStatus, true);

    }

    protected void simuComplementarySensorsOn(ComplementarySensors sensors) {
        replaceSensorValue(sensors.getSensor(), true);
        replaceSensorValue(sensors.getSensorC(), false);
    }

    protected void simuComplementarySensorsNotOn(ComplementarySensors sensors) {
        replaceSensorValue(sensors.getSensor(), false);
        replaceSensorValue(sensors.getSensorC(), true);
    }

    protected void simuComplementarySensorsInError(ComplementarySensors sensors) {
        replaceSensorValue(sensors.getSensor(), true);
        replaceSensorValue(sensors.getSensorC(), true);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreAtHandoff() {
        FCSLOG.info("simulateAutochangerTrucksAreAtHandoff");
        simuComplementarySensorsOn(handoffPositionSensorsXminus);
        simuComplementarySensorsOn(handoffPositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {standbyPositionSensorsXminus,
            standbyPositionSensorsXplus, onlinePositionSensorsXminus, onlinePositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
        setTrucksPosition(trucks.getHandoffPosition());
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreOnline() {
        FCSLOG.info("simulateAutochangerTrucksAreOnline");
        simuComplementarySensorsOn(onlinePositionSensorsXminus);
        simuComplementarySensorsOn(onlinePositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {standbyPositionSensorsXminus,
            standbyPositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
        setTrucksPosition(trucks.getOnlinePosition());
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreAtStandby() {
        FCSLOG.info("simulateAutochangerTrucksAreAtStandby");
        simuComplementarySensorsOn(standbyPositionSensorsXminus);
        simuComplementarySensorsOn(standbyPositionSensorsXplus);
        ComplementarySensors[] sensors = new ComplementarySensors[] {onlinePositionSensorsXminus,
            onlinePositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
        setTrucksPosition(trucks.getStandbyPosition());
        simulateFilterIDAtStandby();
        FCSLOG.info("end of simulateAutochangerTrucksAreAtStandby");
    }

    private void setTrucksPosition(int pos) {
        linearRailMasterController.setPosition(pos);
        linearRailMasterController.setSsiPosition(linearRailMasterController.getEncoderRibbonMinValue() + pos);
        linearRailSlaveController.setPosition(pos);
        linearRailSlaveController.setSsiPosition(linearRailSlaveController.getEncoderRibbonMinValue() + pos);
        for(AutochangerTruckPositionListener l : truckListeners) {
            l.onTruckPositionChange(pos);
        }

    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerTrucksAreInTravel() {
        ComplementarySensors[] sensors = new ComplementarySensors[] {onlinePositionSensorsXminus,
            onlinePositionSensorsXplus, handoffPositionSensorsXminus, handoffPositionSensorsXplus,
            standbyPositionSensorsXminus, standbyPositionSensorsXplus};
        for (ComplementarySensors coupleSensors : sensors ) {
            simuComplementarySensorsNotOn(coupleSensors);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnected() {
        FCSLOG.info(name + " simulateLoaderIsConnected");
        simuComplementarySensorsOn(loaderConnectedSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsDisconnected() {
        FCSLOG.info(name + " simulateLoaderIsDisconnected");
        simuComplementarySensorsNotOn(loaderConnectedSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnAutochanger() {
        FCSLOG.info(name + " simulateFilterIsOnAutochanger");
        simuComplementarySensorsOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsOn(filterEngagedSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerIsEmpty() {
        FCSLOG.info(name + " simulateAutochangerIsEmpty");
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXminus);
        simuComplementarySensorsNotOn(filterEngagedSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreLocked() {
        FCSLOG.info(name + " simulateAutochangerLatchesAreLocked");
        simuComplementarySensorsOn(closeSensorsLatchXminus);
        simuComplementarySensorsOn(closeSensorsLatchXplus);
        simuComplementarySensorsNotOn(openSensorsLatchXminus);
        simuComplementarySensorsNotOn(openSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchesAreUnlocked() {
        FCSLOG.info(name + " simulateAutochangerLatchesAreUnlocked");
        simuComplementarySensorsNotOn(closeSensorsLatchXminus);
        simuComplementarySensorsNotOn(closeSensorsLatchXplus);
        simuComplementarySensorsOn(openSensorsLatchXminus);
        simuComplementarySensorsOn(openSensorsLatchXplus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsLocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsOn(closeSensorsLatchXminus);
            simuComplementarySensorsNotOn(openSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsOn(closeSensorsLatchXplus);
            simuComplementarySensorsNotOn(openSensorsLatchXplus);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerLatchIsUnlocked(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsNotOn(closeSensorsLatchXminus);
            simuComplementarySensorsOn(openSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsNotOn(closeSensorsLatchXplus);
            simuComplementarySensorsOn(openSensorsLatchXplus);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLatchLockSensorsInError(String latchName) {
        if (latchName.contains("Xminus")) {
            simuComplementarySensorsInError(closeSensorsLatchXminus);
        } else if (latchName.contains("Xplus")) {
            simuComplementarySensorsInError(closeSensorsLatchXplus);
        }
    }

    /********************************************************************************/
    /***                       onlineClamps                                       ***/
    /********************************************************************************/

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreLocked() {
        FCSLOG.info("simulateAutochangerOnlineClampsAreLocked");
        simulateAutochangerOnlineClampIsLocked(onlineClampXminus);
        simulateAutochangerOnlineClampIsLocked(onlineClampXplus);
        simulateAutochangerOnlineClampIsLocked(onlineClampYminus);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreClosed() {
        FCSLOG.info("simulateAutochangerOnlineClampsAreClosed");
        simulateAutochangerOnlineClampIsClosed(onlineClampXminus);
        simulateAutochangerOnlineClampIsClosed(onlineClampXplus);
        simulateAutochangerOnlineClampIsClosed(onlineClampYminus);
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampsAreOpened() {
        FCSLOG.info("simulateAutochangerOnlineClampsAreOpened");
        simulateAutochangerOnlineClampIsOpened(onlineClampXminus);
        simulateAutochangerOnlineClampIsOpened(onlineClampXplus);
        simulateAutochangerOnlineClampIsOpened(onlineClampYminus);
    }

    /**
     * simulates onlineClamp is CLOSED : CLOSED and sentCurrent is currentToClamp.
     * @param clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsClosed(AutochangerOnlineClamp clamp) {
        simuComplementarySensorsOn(clamp.getCloseSensors());
        simuComplementarySensorsNotOn(clamp.getOpenSensors());
        clamp.setSentCurrent(clamp.getFinalCurrentToClose());
        simuStrainGaugeClosed();
    }

    /**
     * simulates onlineClamp is LOCKED : CLOSED and sentCurrent is currentToClamp.
     * @param clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsLocked(AutochangerOnlineClamp clamp) {
        simuComplementarySensorsOn(clamp.getCloseSensors());
        simuComplementarySensorsNotOn(clamp.getOpenSensors());
        clamp.setSentCurrent(clamp.getCurrentToClamp());
        simuStrainGaugeLocked();
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateAutochangerOnlineClampIsOpened(AutochangerOnlineClamp clamp) {
        simuComplementarySensorsOn(clamp.getOpenSensors());
        simuComplementarySensorsNotOn(clamp.getCloseSensors());
        clamp.setSentCurrent(clamp.getCurrentToOpen());
        simuStrainGaugeOpened();
    }

    private void simuStrainGaugeLocked() {
        onlineStrainGauge.setStrain((short)1000);
    }

    private void simuStrainGaugeClosed() {
        onlineStrainGauge.setStrain((short)1500);
    }

    private void simuStrainGaugeOpened() {
        onlineStrainGauge.setStrain((short)1700);
    }

    /********************************************************************************/
    /***                 end of onlineClamps                                      ***/
    /********************************************************************************/




    /**
     * Changes the values of the gateway in order to simulate that the filter read by autochanger
     * at STANDBY is the filterID of the filter on carousel at STANDBY.
     * This has to be done only for wholeFCS.
     * In autochanger-standalone mode this is not needed.
     */
    private void simulateFilterIDAtStandby() {
        if ((carousel instanceof Carousel) && ((Carousel)carousel).isAtStandby()) {
            if (((Carousel)carousel).getSocketAtStandby() != null) {
                simulateFilterID(((Carousel)carousel).getSocketAtStandby().getFilterID());
            } else {
                throw new IllegalArgumentException(name + " error in simulated carousel : filter is null ");
            }
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIDOnAutochanger(int id) {
        this.simulateFilterIsOnAutochanger();
        this.simulateFilterID(id);
        autochanger.setFilterOnTrucksID(id);
    }

    /**
     * Here we want to simulate the autochanger sensors to be consistent with the filter location.
     * If a filter is on autochanger, we suppose that it'subs at ONLINE position and onlineClamps are LOCKED,
 and latches are LOCKED, and filter presence sensors detect a filter.
     */
    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart");
        simuLPMStatusOn();
        if (autochanger.getFilterID() != 0) {
                simulateAutochangerTrucksAreOnline();
                simulateFilterIsOnAutochanger();
                simulateAutochangerLatchesAreLocked();
                simulateAutochangerOnlineClampsAreLocked();

        } else {
                simulateAutochangerTrucksAreAtHandoff();
                simulateAutochangerIsEmpty();
                simulateAutochangerLatchesAreUnlocked();
                simulateAutochangerOnlineClampsAreOpened();
        }
        FCSLOG.info(name + " END postStart");
    }

}
