
package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;

/**
 * This interface is implemented by each class that can raises an Alert.
 * @author virieux
 */
public interface AlertRaiser {
    /**
     * to separate Alert Name or ID from device name.
     */
    char alertSeparator = '/';

    String getName();

    Subsystem getSubsystem();

    /**
     * return alertService
     * @return
     */
    AlertService getAlertService();

    default char getAlertSeparator() {
        return alertSeparator;
    }


    /**
     * Raises an ALARM with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause) {
        getAlertService().raiseAlert(new Alert(fcsAlert.name(),fcsAlert.getLongDescription()), ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and an Exception.
     * @param fcsAlert
     * @param cause
     * @param ex
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, Exception ex) {
        FCSLOG.error(ex);
        raiseAlarm(fcsAlert, cause + " due to " + ex);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(new Alert(deviceName+alertSeparator+fcsAlert.name(),fcsAlert.getLongDescription()), ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause, a deviceName and an Exception
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseAlarm(FcsAlert fcsAlert, String cause, String deviceName, Exception ex) {
        raiseAlarm(fcsAlert, cause + " due to " + ex, deviceName);
    }

    /**
     * Clear Alarm
     * @param fcsAlert
     * @param deviceName
     */
    default void clearAlarm(FcsAlert fcsAlert, String deviceName) {
        String[] alarmList = new String[] {deviceName+alertSeparator+fcsAlert.name()};
        getAlertService().clearAlerts(alarmList);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(new Alert(deviceName +alertSeparator+ fcsAlert.name(),fcsAlert.getLongDescription()), WARNING, cause);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause, a deviceName and an exception.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseWarning(FcsAlert fcsAlert, String cause, String deviceName, Exception ex) {
        FCSLOG.warning(ex);
        raiseWarning(fcsAlert, cause + " due to " + ex, deviceName);
    }


}
