package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is the Main Module for the Loader control software when the loader is in
 * standalone.
 *
 * @author virieux
 */
public class LoaderMain extends MainModule {

    @LookupField(strategy = Strategy.CHILDREN)
    private Loader loader;

    @LookupField(strategy = Strategy.CHILDREN)
    private FilterHolder autochanger;

    /**
     * Build a LoaderMainModule with a bridge.
     *
     * @param bridge
     */
    public LoaderMain(BridgeToHardware bridge) {
        super(bridge);
    }

    @Override
    public void init() {
        /*
         * define a role for my subsystem in order to make LoaderGUI listen to my
         * subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.loader", "loader");
    }

    /**
     * Update loader state in reading the sensors.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update loader state in reading the sensors.")
    @Override
    public void updateStateWithSensors() {
        loader.updateStateWithSensors();
        autochanger.updateStateWithSensors();
    }

    /**
     * Check that Loader hardware is ready to be operated and moved. This means
     * that :
     * - all CAN open devices are booted, identified and initialized,
     * - locateHardware has been done on the controllers. This updates the FCS
     * state and FCS readyness state and publishes on the status bus.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    @Override
    public void updateFCSStateToReady() {
        if (loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            subs.getAgentService(AgentStateService.class).updateAgentState(FcsEnumerations.FilterState.READY, FcsEnumerations.FilterReadinessState.READY);
        }
    }

    /**
     * Print list of hardware with initialization information.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "Print the list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(this.bridge.printHardwareState());
        sb.append("\n ");
        sb.append(loader.printHardwareState());
        return sb.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n").append(loader.toString());
        return sb.toString();
    }

}
