/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.canopenjni.CanOpenInterface;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.canopenjni.SDOException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FCSCst;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenErrorsTable;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEventListener;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

public class CanOpenProxy
implements HardwareController,
ClearAlertHandler,
BridgeToHardware,
HasLifecycle,
AlertRaiser,
CanOpenEventListener {
    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subs;
    @LookupName
    protected String name;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    private CanOpenInterface canInterface;
    private int master = 1;
    @ConfigurationParameter(description="CANbus rate.")
    private String baud = "125k";
    @ConfigurationParameter(description="CANbus name. Can be 0 for changer or 1 for loader.")
    private String busName = "0";
    @ConfigurationParameter(description="CANbus master nodeID.")
    private int masterNodeID = 8;
    @ConfigurationParameter(range="1000..100000", description="A timeout for the hardware booting process")
    public long hardwareBootTimeout = 5000L;
    protected final Map<Integer, PieceOfHardware> hardwareMapByNodeID = new HashMap<Integer, PieceOfHardware>();
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    protected final Map<String, PieceOfHardware> hardwareMapByName = new HashMap<String, PieceOfHardware>();
    protected boolean hardwareBootProcessEnded = false;
    protected int bootedDeviceNB;
    private PDOData pdoData = new PDOData();
    private ScheduledFuture<?> checkDevicesHandle;
    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    private final Lock lock = new ReentrantLock();
    private final Condition bootingCompleted = this.lock.newCondition();

    public CanOpenProxy(long hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Subsystem getSubsystem() {
        return this.subs;
    }

    public CanOpenInterface getCanInterface() {
        return this.canInterface;
    }

    @Command
    public PDOData getPdoData() {
        return this.pdoData;
    }

    @Override
    @Command(type=Command.CommandType.ACTION, level=0, description="Disconnect the hardware. Used for loader disconnection.")
    public void disconnectHardware() {
        try {
            this.canInterface.quit();
            this.initialize();
        }
        catch (DriverException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Override
    @Command(type=Command.CommandType.ACTION, level=0, description="Connect the hardware. Used for loader.")
    public void connectHardware() {
        this.start();
        this.postStart();
    }

    public void init() {
        this.initialize();
        this.hardwareMapByName.values().stream().forEach(poh -> this.hardwareMapByNodeID.put(poh.getNodeID(), (PieceOfHardware)poh));
        FCSCst.FCSLOG.info((Object)(this.name + ": init MODULE CanOpenProxy."));
        FCSCst.FCSLOG.info((Object)(this.name + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + this.hardwareMapByName.size()));
        FCSCst.FCSLOG.info((Object)this.toString());
        CanOpenErrorsTable.loadCanOpenErrorTables();
    }

    private void initialize() {
        this.bootedDeviceNB = 0;
        this.hardwareBootProcessEnded = false;
        this.hardwareMapByName.values().stream().forEach(poh -> poh.setBooted(false));
    }

    @Override
    @Command(type=Command.CommandType.ACTION, description="Check if all pieces of hardware are booted with the corect serial number.")
    public void bootProcess() {
        FCSCst.FCSLOG.info((Object)(this.name + ": BEGIN OF BOOT PROCESS"));
        this.initialize();
        this.subs.updateAgentState(new Enum[]{FcsEnumerations.FilterState.CAN_DEVICES_BOOTING, FcsEnumerations.FilterReadinessState.NOT_READY});
        this.lock.lock();
        try {
            this.readDevicesInfo();
            this.waitForEndOfBooting();
        }
        catch (Exception ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " ERROR during booting process", ex);
        }
        finally {
            FCSCst.FCSLOG.debug((Object)(this.name + ": finally in bootProcess"));
            this.hardwareBootProcessEnded = true;
            this.bootingCompleted.signalAll();
            this.lock.unlock();
        }
        if (this.bootedDeviceNB == this.hardwareMapByName.size()) {
            FCSCst.FCSLOG.info((Object)(this.name + " ALL HARDWARE is booted"));
            this.subs.updateAgentState(new Enum[]{FcsEnumerations.FilterState.HOMING_TO_BE_DONE, FcsEnumerations.FilterReadinessState.READY});
        } else if (this.bootedDeviceNB == 0) {
            FCSCst.FCSLOG.error((Object)(this.name + " NONE HARDWARE is booted"));
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " NO HARDWARE booted - Power failure ?");
        } else if (this.bootedDeviceNB < this.hardwareMapByName.size()) {
            FCSCst.FCSLOG.error((Object)(this.name + " SOME HARDWARE is MISSING"));
            this.hardwareMapByName.values().stream().forEach(poh -> poh.raiseAlarmIfMissing());
        } else {
            FCSCst.FCSLOG.error((Object)(this.name + " this.bootedDeviceNB =  hardwareMapByName.size() = " + this.hardwareMapByName.size()));
        }
        FCSCst.FCSLOG.info((Object)(this.name + " BOOTED HARDWARE=" + this.listBootedNodes()));
        FCSCst.FCSLOG.info((Object)(this.name + ": END OF BOOT PROCESS"));
    }

    @Command(type=Command.CommandType.QUERY, description="Initialize PDO for all my devices.")
    public void initializePDOs() {
        this.hardwareMapByNodeID.values().stream().forEach(poh -> {
            try {
                poh.initializePDOs();
            }
            catch (DriverException ex) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " ERROR during PDO initialization : could not addReceivedPDOs", poh.getName(), (Exception)((Object)ex));
            }
        });
    }

    @Override
    public PDOData updatePDOData() {
        long duration;
        long beginTime = System.currentTimeMillis();
        try {
            this.pdoData = this.canInterface.sync();
            duration = System.currentTimeMillis() - beginTime;
            FCSCst.FCSLOG.info((Object)(this.name + " this.canInterface.sync() duration = " + duration));
        }
        catch (DriverException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " couldn't update PDOs", (Exception)((Object)ex));
        }
        this.hardwareMapByNodeID.values().stream().forEach(poh -> {
            try {
                poh.updateFromPDO(this.pdoData);
            }
            catch (DriverException ex) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " ERROR during PDO reading : could not update PDOs", poh.getName(), (Exception)((Object)ex));
            }
        });
        if (!this.pdoData.isComplete()) {
            FCSCst.FCSLOG.error((Object)(this.name + " some PDOs are missing. Power failure or bad configuration of a device ?"));
        }
        duration = System.currentTimeMillis() - beginTime;
        FCSCst.FCSLOG.info((Object)(this.name + " updatePDOData duration = " + duration));
        return this.pdoData;
    }

    public void start() {
        this.subs.updateAgentState(new Enum[]{FcsEnumerations.FilterState.WAITING_FOR_HARWARE_BOOTING, FcsEnumerations.FilterReadinessState.NOT_READY});
        try {
            this.canInterface.init(this.master, this.baud, this.busName, this.masterNodeID);
            this.canInterface.scan();
            Thread.sleep(this.hardwareBootTimeout);
        }
        catch (InterruptedException | DriverException ex) {
            throw new RuntimeException(ex);
        }
    }

    public void postStart() {
        this.bootProcess();
        this.initializePDOs();
        this.publishData();
    }

    public void readDevicesInfo() {
        long startTime = System.currentTimeMillis();
        Runnable checkDevices = () -> {
            try {
                this.retrieveHardwareInfo();
                long duration = System.currentTimeMillis() - startTime;
                FCSCst.FCSLOG.fine((Object)("duration=" + duration + " : " + this.bootedDeviceNB + " booted devices."));
                if (duration > this.hardwareBootTimeout || this.bootedDeviceNB == this.hardwareMapByName.size()) {
                    this.hardwareBootProcessEnded = true;
                    this.cancelReadDevicesInfo();
                }
            }
            catch (Exception ex) {
                FCSCst.FCSLOG.error((Object)(this.name + " error in devices boot process"), (Throwable)ex);
            }
        };
        this.checkDevicesHandle = this.scheduler.scheduleWithFixedDelay(checkDevices, 500L, 500L, TimeUnit.MILLISECONDS);
    }

    private void cancelReadDevicesInfo() {
        this.lock.lock();
        try {
            FCSCst.FCSLOG.debug((Object)(this.name + " => stop waiting for devices info."));
            this.bootingCompleted.signalAll();
        }
        finally {
            this.lock.unlock();
        }
        this.checkDevicesHandle.cancel(true);
        FCSCst.FCSLOG.debug((Object)(this.name + " => readDevicesInfo canceled"));
    }

    private void waitForEndOfBooting() {
        while (!this.hardwareBootProcessEnded) {
            try {
                FCSCst.FCSLOG.info((Object)(this.name + " waiting until all pieces of hardware are booted."));
                this.bootingCompleted.await();
            }
            catch (InterruptedException ex) {
                FCSCst.FCSLOG.info((Object)(this.name + ": InterruptedException received=" + ex));
                break;
            }
        }
        FCSCst.FCSLOG.info((Object)(this.name + " STOP WAITING FOR END OF BOOTING PROCESS"));
    }

    public void checkStarted() {
        FCSCst.FCSLOG.info((Object)(this.name + " BEGIN checkStarted"));
        this.bootProcess();
        FCSCst.FCSLOG.info((Object)(this.name + " END checkStarted"));
    }

    public void shutdown() {
        List taskNames = this.subs.getAgentPeriodicTaskService().getAgentPeriodicTaskNames();
        if (taskNames.contains("monitor-update")) {
            this.subs.getAgentPeriodicTaskService().setPeriodicTaskPeriod("monitor-update", Duration.ofSeconds(-1L));
        }
        if (taskNames.contains("monitorCurrent")) {
            this.subs.getAgentPeriodicTaskService().setPeriodicTaskPeriod("monitorCurrent", Duration.ofSeconds(-1L));
        }
    }

    @Override
    public void doShutdown() {
        FCSCst.FCSLOG.info((Object)(this.name + " is shutting down."));
        this.scheduler.shutdown();
        try {
            this.canInterface.quit();
            this.canInterface.stop();
            this.subs.updateAgentState(new Enum[]{FcsEnumerations.FilterState.OFF_LINE, FcsEnumerations.FilterReadinessState.NOT_READY});
        }
        catch (DriverException ex) {
            FCSCst.FCSLOG.error((Object)"cannot shut down properly", (Throwable)ex);
        }
        FCSCst.FCSLOG.info((Object)(this.name + " is shutdown."));
    }

    public String getNodeName(int aNodeID) {
        if (this.hardwareMapByNodeID.get(aNodeID) == null) {
            return "UnknownDevice" + aNodeID;
        }
        return this.hardwareMapByNodeID.get(aNodeID).getName();
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Print the list of CANopen nodes which are booted on the CAN bus.")
    public String listBootedNodes() {
        if (this.bootedDeviceNB == 0) {
            return this.name + ": no booted CANopen devices.";
        }
        StringBuilder sb = new StringBuilder();
        for (PieceOfHardware poh : this.hardwareMapByName.values()) {
            if (!poh.isBooted()) continue;
            sb.append("\n==>");
            sb.append(poh);
        }
        return sb.toString();
    }

    public void checkNodeID(int nodeID) {
        if (!this.hardwareMapByNodeID.containsKey(nodeID)) {
            throw new IllegalArgumentException(nodeID + " is not in the hardware list for tcpProxy " + this.name);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="Send a CanOpen writeSDO command to the Can Bus. \n size represents the number of bytes on which the value is encoded. See device documentation.")
    public synchronized void writeSDO(int nodeID, int index, int subindex, int size, int value) {
        this.checkNodeID(nodeID);
        if (size < 0 || size > 4) {
            throw new IllegalArgumentException("size must be > 0 and < 4");
        }
        try {
            this.tryWsdo(nodeID, index, subindex, size, value);
        }
        catch (DriverException ex) {
            String msg = " error for request=  writeSDO,0x" + Integer.toHexString(nodeID) + ",0x" + Integer.toHexString(index) + ",0x" + Integer.toHexString(subindex) + "," + size + "," + value;
            if (ex instanceof SDOException) {
                String errorName = CanOpenErrorsTable.getCommErrorNameByCode(((SDOException)ex).getErrCode());
                String deviceName = this.getNodeName(nodeID);
                this.raiseAlarm(FcsEnumerations.FcsAlert.SDO_ERROR, msg + " errorCode = " + ((SDOException)ex).getErrCode() + " " + errorName, deviceName, (Exception)((Object)ex));
            }
            throw new SDORequestException(this.name + msg, (Throwable)ex);
        }
    }

    private void tryWsdo(int nodeID, int index, int subindex, int size, int value) throws DriverException {
        try {
            this.canInterface.wsdo(nodeID, index, subindex, size, (long)value);
        }
        catch (SDOException ex) {
            String msg = " error in tryWsdo for nodeID 0x" + Integer.toHexString(nodeID) + " (in decimal=" + nodeID + ") ";
            String deviceName = this.getNodeName(nodeID);
            this.raiseWarning(FcsEnumerations.FcsAlert.SDO_ERROR, msg + " errorName = " + ex.getErrCode(), deviceName, (Exception)((Object)ex));
            FCSCst.FCSLOG.warning((Object)("Retry wsdo for nodeID 0x" + Integer.toHexString(nodeID) + " Index=0x" + Integer.toHexString(index) + " Subindex=0x" + Integer.toHexString(subindex)));
        }
    }

    public synchronized long readSDO(int nodeID, int index, int subindex) {
        try {
            return this.canInterface.rsdo(nodeID, index, subindex);
        }
        catch (SDOException ex) {
            String msg = this.name + " error for request= readSDO,0x" + Integer.toHexString(nodeID) + ",0x" + Integer.toHexString(index) + ",0x" + Integer.toHexString(subindex);
            String errorName = CanOpenErrorsTable.getCommErrorNameByCode(ex.getErrCode());
            String deviceName = this.getNodeName(nodeID);
            this.raiseWarning(FcsEnumerations.FcsAlert.SDO_ERROR, msg + " errorName=" + errorName, deviceName, (Exception)((Object)ex));
            throw new SDORequestException(msg, (Throwable)ex);
        }
        catch (DriverException ex) {
            String msg = this.name + " error for request= readSDO," + Integer.toHexString(nodeID) + "," + Integer.toHexString(index) + "," + Integer.toHexString(subindex);
            String deviceName = this.getNodeName(nodeID);
            this.raiseAlarm(FcsEnumerations.FcsAlert.SDO_ERROR, msg, deviceName, (Exception)((Object)ex));
            throw new SDORequestException(ex.getMessage(), (Throwable)ex);
        }
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command sync to CAN bus to read sensors by PDO.")
    public PDOData sync() throws DriverException {
        PDOData pdoD = this.canInterface.sync();
        FCSCst.FCSLOG.info((Object)("PDOData=" + pdoD.toString()));
        return pdoD;
    }

    @Command(type=Command.CommandType.QUERY, level=3, description=" init CAN interface.")
    public void init(int master, String baud, String busName, int nodeID) throws DriverException {
        this.canInterface.init(master, baud, busName, nodeID);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="add a receive PDO to canInterface.")
    public void addReceivedPDO(int cobId) throws DriverException {
        this.canInterface.addReceivedPDO(cobId);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Scan CAN bus.")
    public int scan() throws DriverException {
        return this.canInterface.scan();
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command info to CAN bus for nodeID to retrieve device information.")
    public String info(int nodeID) throws DriverException {
        return this.canInterface.info(nodeID);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command wsdo to device.")
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException {
        this.canInterface.wsdo(nodeId, index, subindex, size, data);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command rsdo to device.")
    public long rsdo(int nodeId, int index, int subindex) throws DriverException {
        return this.canInterface.rsdo(nodeId, index, subindex);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command ssta (slave start) to device.")
    public void ssta(int nodeId) throws DriverException {
        this.canInterface.ssta(nodeId);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command ssto (slave stop) to device.")
    public void ssto(int nodeId) throws DriverException {
        this.canInterface.ssto(nodeId);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command reset to device.")
    public void reset(int nodeId) throws DriverException {
        this.canInterface.reset(nodeId);
    }

    @Command(type=Command.CommandType.QUERY, level=3, description="Send a command quit to canInterface to device.")
    public void quit() throws DriverException {
        this.canInterface.quit();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Identification of the hardware : we want to retrieve the information stored in the hardware of the CANopen devices")
    public void retrieveHardwareInfo() {
        FCSCst.FCSLOG.info((Object)(this.name + ":Identification of the hardware"));
        this.hardwareMapByName.values().stream().filter(poh -> !poh.isBooted()).forEach(poh -> this.updateDeviceInfo((PieceOfHardware)poh));
        this.publishData();
    }

    private void updateDeviceInfo(PieceOfHardware poh) {
        int nodeID = poh.getNodeID();
        String commandInfo = "info," + Integer.toHexString(nodeID);
        try {
            FCSCst.FCSLOG.debug((Object)(this.name + ":Sending to can open command :" + commandInfo));
            String result = this.canInterface.info(nodeID);
            FCSCst.FCSLOG.info((Object)("result=" + result));
            this.processInfoMessage(poh, result);
        }
        catch (DriverException ex) {
            this.raiseWarning(FcsEnumerations.FcsAlert.HARDWARE_MISSING, " no response to command : " + commandInfo + "\f", poh.getName(), (Exception)((Object)ex));
        }
        catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, " Error in boot process for device.", poh.getName(), (Exception)((Object)ex));
        }
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if all CANopen devices are booted and identified.")
    public boolean allDevicesBooted() {
        return this.hardwareBootProcessEnded && this.bootedDeviceNB == this.hardwareMapByNodeID.size();
    }

    @Override
    public void onBootMessage(int nodeID) {
        if (this.hardwareMapByNodeID.containsKey(nodeID)) {
            PieceOfHardware poh = this.hardwareMapByNodeID.get(nodeID);
            if (poh.isBooted() && this.hardwareBootProcessEnded) {
                String cause = " received an unsynchronous boot message for node ID = " + Integer.toHexString(nodeID) + " Has the device been reset or powered on ?";
                this.raiseWarning(FcsEnumerations.FcsAlert.UNSYNC_BOOT, cause, poh.getName());
            }
        } else {
            this.raiseAlarm(FcsEnumerations.FcsAlert.UNSYNC_BOOT, Integer.toHexString(nodeID) + ":UNKNOWN device. This device is not in the harware list.");
        }
    }

    @Override
    public void onEmergencyMessage(int nodeID, int errCode, int errReg) {
        String deviceErrorName = CanOpenErrorsTable.getDeviceErrorNameByCode(errCode);
        String errorRegisterName = CanOpenErrorsTable.getErrorRegisterNameByCode(errReg);
        String deviceName = this.getNodeName(nodeID);
        EmergencyMessage emcyMsg = new EmergencyMessage(nodeID, deviceName, errCode, deviceErrorName, errReg, errorRegisterName);
        PieceOfHardware poh = this.hardwareMapByNodeID.get(nodeID);
        if (poh != null) {
            this.subs.getScheduler().schedule(() -> poh.onEmergencyMessage(emcyMsg), 0L, TimeUnit.SECONDS);
        }
        FCSCst.FCSLOG.warning((Object)(this.name + " received EMERGENCY message=" + emcyMsg.toString() + " for nodeID=" + Integer.toHexString(nodeID)));
    }

    void processInfoMessage(PieceOfHardware poh, String message) {
        String[] words = message.split(",", -1);
        String type = words[2];
        String vendor = words[3];
        String productCode = words[4];
        String revision = words[5];
        String serialNB = words[6];
        FCSCst.FCSLOG.info((Object)(this.name + "/" + poh.getName() + " informations read on device: type=" + type + " vendor=" + vendor + " product code=" + productCode + " revision=" + revision + " serialNB" + serialNB));
        if (poh.getSerialNB().equals(serialNB)) {
            poh.setBooted(true);
            ++this.bootedDeviceNB;
        } else {
            this.raiseAlarm(FcsEnumerations.FcsAlert.BAD_SERIAL_NB, " serial number read on device=" + serialNB + " ==>serial number expected=" + poh.getSerialNB() + " ==>device configuration=" + poh, poh.getName());
        }
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=1, description="Return a printed list of hardware expected in this subsystem.")
    public String printHardwareList() {
        return this.hardwareMapByName.toString();
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=1, description="Return a printed list of hardware with the initialization state.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(this.name + " CANopen devices : {");
        for (PieceOfHardware poh : this.hardwareMapByNodeID.values()) {
            sb.append(poh.printState());
            sb.append(";\n");
        }
        sb.append('}');
        return sb.toString();
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=1, description="Return an Array containing hardware names handled by this component.")
    public List<String> listHardwareNames() {
        ArrayList<String> res = new ArrayList<String>();
        for (PieceOfHardware pieceOfHardware : this.hardwareMapByName.values()) {
            res.add(pieceOfHardware.getName());
        }
        return res;
    }

    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            case "FCS001": {
                try {
                    if (this.canInterface.isReady()) {
                        return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                    }
                }
                catch (DriverException ex) {
                    FCSCst.FCSLOG.error((Object)ex);
                }
                return ClearAlertHandler.ClearAlertCode.DONT_CLEAR_ALERT;
            }
            case "FCS002": {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        }
        return ClearAlertHandler.ClearAlertCode.UNKWNOWN_ALERT;
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=1, description="Publish booting information for all CANopen devices.")
    public void publishData() {
        FCSCst.FCSLOG.info((Object)(this.name + " is publishing:" + this.bootedDeviceNB));
        this.subs.publishSubsystemDataOnStatusBus(new KeyValueData(this.name, (Serializable)Integer.valueOf(this.bootedDeviceNB)));
        this.hardwareMapByName.values().stream().forEach(device -> device.publishData());
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("name=");
        sb.append(this.name);
        sb.append("/hardwareBootTimeout=");
        sb.append(this.hardwareBootTimeout);
        return sb.toString();
    }

    @Override
    public boolean isReady() {
        try {
            return this.canInterface.isReady() && this.allDevicesBooted();
        }
        catch (DriverException ex) {
            FCSCst.FCSLOG.error((Object)ex);
            return false;
        }
    }

    @Override
    public AlertService getAlertService() {
        return this.subs.getAlertService();
    }
}

