
package org.lsst.ccs.subsystems.fcs;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;

/**
 * This is the Main Module for every software related to the Filters Exchanger :
 * - single-filter-test control-command - scale 1 prototype control-command -
 * real Filters Exchanger This class is designed to gather the commons methods
 * to all these softwares.
 *
 * @author virieux
 */
public class MainModule implements AlertRaiser, SignalHandler, HasLifecycle {

    @LookupName
    protected String name;
    
    @LookupField(strategy=Strategy.TOP)
    protected Subsystem subs;

    @LookupField(strategy=Strategy.TREE)
    private AlertService alertService;
    
    /**
     * This field can be null
     */
    @LookupField(strategy=Strategy.TREE)
    private FilterManager filterManager;
    
    protected BridgeToHardware bridge;
    protected AtomicBoolean haltRequired = new AtomicBoolean(false);
    protected AtomicBoolean stopRequired = new AtomicBoolean(false);

    public MainModule(BridgeToHardware bridge) {
        this.bridge = bridge;
    }
    
    @Override
    public String getName() {
        return name;
    }
    
    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    public boolean isHaltRequired() {
        return haltRequired.get();
    }

    public boolean isStopRequired() {
        return stopRequired.get();
    }

    /**
     * This methods updates the boolean hardwareReady from the hardware bridge.
     *
     * @return true if hardware is booted, identified and initialized
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if all hardware is booted with the correct serial number.")
    public boolean allDevicesBooted() {
        return this.bridge.allDevicesBooted();
    }
    
    /**
     * Returns list of hardware names.
     * Used by GUI.
     * @return list of hardware this MainModule manages.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, 
            description = "Returns the list of CANopen hardware managed by this subsystem.")
    public List<String> listHardwareNames() {
        return this.bridge.listHardwareNames();
    }
    
    /**
     * For Whole FCS GUI. 
     * Has to be overriden in FcsMainModule.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, 
            description = "Return the list of LOADER CANopen hardware managed by this subsystem.")
    public List<String> listLoaderHardwareNames() {
        return Collections.emptyList();
    }    
    
    
    /**
     * Return a list of filter names that the subsystem manages.
     * For the GUI or the Console.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Return the list of names of filters  "
            + "that this subsystem manages.",             
            level = Command.NORMAL,
            alias="listFilterNames")
    public List<String> getFilterNames() {
        return (filterManager==null) ? Collections.emptyList() : filterManager.getFilterNamesSortedByFilterID();
    }

    /**
     * Update state in reading sensors.
     * This method has to be overridden by subclasses.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors()  {
    }
    
    /**
     * Updates the FCS state and FCS readyness state and publishes on the status bus.
     * Checks that hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readyness state and publishes on the status bus.
     * This has to be overriden if there is other initializations to do on the hardware like homing.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update FCS state and FCS readiness state and publishes on the status bus.")    
    public void updateFCSStateToReady() {
        if (this.allDevicesBooted() && !subs.isInState(ALARM)) {
            /* The initialization has been done, so now the hardware is ready */
            subs.updateAgentState(FcsEnumerations.FilterState.READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }    


    @Override
    public TreeWalkerDiag signal(Signal signal) {
        switch (signal.getLevel()) {
            case HALT:
                FCSLOG.debug("HALT required");
                this.haltRequired.set(true);
                break;
            case STOP:
                FCSLOG.debug("STOP required");
                this.stopRequired.set(true);
                break;
            default:
                assert false;
        }
        //we want the signal to go to all my children.
        return TreeWalkerDiag.GO;
    }

    /**
     * for GUI
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Publish Data to populate GUI.")    
    public void publishData() {
        this.bridge.publishData();
    }
    
    @Override
    public void postShutdown() {
        FCSLOG.info(name + " is shutting down.");
        bridge.doShutdown();
        FCSLOG.info(name + " is shutdown.");
    }

}
