
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileAcceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileDeceleration;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter.ProfileVelocity;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * This is a model for the carrier in the loader.
 *
 * @author virieux
 */
public class LoaderCarrier extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    
    private static final int TIMEOUT_FOR_MOVING_CARRIER = 120000;

    @LookupField(strategy=Strategy.BYNAME)
    private BinarySensor handoffSensors;  
    
    @LookupField(strategy=Strategy.BYNAME)
    private BinarySensor storageSensors;
    
    @LookupField(strategy=Strategy.BYNAME)
    private BinarySensor engagedSensors;

    private int position = 0;
    private int ssiPosition = 0;
    private int absoluteTargetPosition = 0;
    
    @ConfigurationParameter(description="delta position in micron : used to know if carrier position "
            + "is in a range of 2*deltaPosition around a given position")
    private int deltaPosition = 50;
    
    @ConfigurationParameter(description="Loader Handoff position in micron")
    private int handoffPosition = 1037500;
    
    @ConfigurationParameter(description="Loader Engaged position in micron")
    private int engagedPosition = 1035500;    
    
    @ConfigurationParameter(description="Loader Storage position in micron")
    private int storagePosition = 0; 
    
    @ConfigurationParameter(description=
            "For the Loader GUI : Maximum current to be sent to the Loader Carrier controller (in mA).")
    private int maxCurrent = 1000;
        
    @ConfigurationParameter(description="For the Loader GUI : Loader Carrier Maximum speed (in rpm/mn).")
    private int maxSpeed = 227;

    @ConfigurationParameter(description="timeout in milliseconds to go from storage to handoff on loader")
    private long timeoutForGoingToHandOff = 120000;
    
    @ConfigurationParameter(description="timeout in milliseconds to go from storage to engaged on loader")
    private long timeoutForGoingToEngaged = 120000;    
    
    @ConfigurationParameter(description="timeout in milliseconds to go from handoff to storage on loader")
    private long timeoutForGoingToStorage = 120000;
    
    @ConfigurationParameter(description="low speed to go with a filter from ENGAGED to HANDOFF position")
    private int lowSpeed = 50;
    
    @ConfigurationParameter(description="high speed to move carrier when it's empty or with a filter from ENGAGED to STORAGE position")
    private int highSpeed = 200; 
    
    @ConfigurationParameter(description="low acceleration to move carrier with a filter from ENGAGED to STORAGE position")
    private int lowAcceleration = 30;
    
    @ConfigurationParameter(description="high acceleration to move carrier empty from ENGAGED to STORAGE position")
    private int highAcceleration = 100;
    
    @ConfigurationParameter(description="low deceleration to move carrier with a filter from ENGAGED to STORAGE position")
    private int lowDeceleration = 30;
    
    @ConfigurationParameter(description="high deceleration to move carrier empty from ENGAGED to STORAGE position")
    private int highDeceleration = 50;    

    @LookupField(strategy=Strategy.BYNAME)
    private EPOSController carrierController;
    
    @LookupField(strategy=Strategy.TREE)
    private Loader loader;
    
    @LookupField(strategy=Strategy.BYNAME)
    private BridgeToHardware loaderTcpProxy;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private volatile boolean initialized = false;
    
    private int speed;
    private int current;
    private long profileVelocity;

    /**
     * 
     * @return carrier position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns carrier position.")    
    public int getPosition() {
        return position;
    }

    /**
     * returns current for tests
     * @return 
     */
    protected int getCurrent() {
        return current;
    }
    
    /**
     * @return controllerName
     */
    @Override
    public String getControllerName() {
        return this.carrierController.getName();
    }

    /**
     * 
     * @return true if carrier is initialized and ready to receive commands.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carrier is initialized and ready to receive commands.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * Returns max speed - Used by GUI
     * @return max speed in rpm/mn (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max speed in rpm/mn (format decimal).")
    public int getMaxSpeed() {
        return maxSpeed;
    }

    /**
     * Returns max current - Used by GUI
     * @return max current in mA (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the max current in mA (format decimal).")
    public int getMaxCurrent() {
        return maxCurrent;
    }

    /**
     *
     * @return handoff position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the handoff position in microns (format decimal).")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    /**
     * 
     * @return 
     */
    public int getEngagedPosition() {
        return engagedPosition;
    }

    /**
     *
     * @return storage position in microns (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the storage position in microns (format decimal).")
    public int getStoragePosition() {
        return storagePosition;
    }

    /**
     * Used by GUI
     * @return timeout for going to Handoff action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Handoff in millis (format decimal).")
    public long getTimeoutForGoingToHandOff() {
        return timeoutForGoingToHandOff;
    }

    /**
     * Used by GUI
     * @return timeout for going to Storage action in millis (format decimal).
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return the timeout for going to Storage in millis (format decimal).")
    public long getTimeoutForGoingToStorage() {
        return timeoutForGoingToStorage;
    }

    /**
     * 
     * @return false if the 2 redondant position sensors at Storage are equal
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Storage are equal.")
    public boolean isStorageSensorsInError() {
        return this.storageSensors.isInError();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at HandOff are equal.")
    public boolean isHandoffSensorsInError() {
        return handoffSensors.isInError();
    }

    /**
     * returns true if handoff sensors or storage sensors are in error.
     * @return 
     */
    @Override
    public boolean isInError() {
        return this.handoffSensors.isInError() || this.storageSensors.isInError()
                || this.engagedSensors.isInError();
    }
    
    /**
     * This command is to be used by end user in case the carrier controller has lost
     * its position.
     * This must not be launched automatically.
     * Tested with success in may 2018 on prototype.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
        description = "Homing of carrier controller. Should be done only if necessary.")
    public void homing() {
        carrierController.definePositionFromNegativeLimitSwitch();
        FCSLOG.info(name + " homing done.");
    }

    /**
     * initialize controller
     */
    public void initializeController() {
        try {
            carrierController.initializeAndCheckHardware();
            this.profileVelocity = readProfileVelocity();
            this.initialized = true;
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, " could not initialize controller", name, ex);
        }
    }
    
    @Override
    public void postStart() {
        FCSLOG.info(name + " postStart");
        if (carrierController.isBooted()) {
            initializeController();
        } 
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }
    
    /**
     * Returns the boolean field atStorage. If the atStorage boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * atStorage is updated. If the field atStorage is not being updated, it
     * returns immediatly the field atStorage.
     *
     * @return atStorage
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at storage position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStorage() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return this.storageSensors.isOn();

        } finally {
            lock.unlock();
        }
    }

    /**
     * Return true if the carrier is at HANDOFF position.
     * This command doesn't read again the sensors but waits if sensors are beeing updated.
     * At HANDOFF engagedSensors and handoffSensors are ON.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at HANDOFF position. "
            + "This command doesn't read again the sensors."
                    + "At HANDOFF engagedSensors and handoffSensors are ON.")
    public boolean isAtHandoff() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return this.handoffSensors.isOn() && this.engagedSensors.isOn();

        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Return true if carrier is at ENGAGED position. 
     * Wait for response if sensors are being updated otherwise returns immediatly.
     * At ENGAGED position, engagedSensors are ON and handoffSensors are OFF.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at HANDOFF position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtEngaged() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.",ex);
                }

            }
            return this.engagedSensors.isOn() && !this.handoffSensors.isOn();

        } finally {
            lock.unlock();
        }
    }    



    /**
     * Move the carrier to Handoff position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            alias = "goToHandoff",
            description = "Move the carrier to Handoff position.",
            timeout = TIMEOUT_FOR_MOVING_CARRIER)
    public void goToHandOff()  {
        if (this.isAtHandoff()) {
            FCSLOG.info(name + " is already at Handoff position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(handoffPosition);
            this.absoluteTargetPosition = this.handoffPosition;
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_HANDOFF,
                    timeoutForGoingToHandOff);
            afterMotion();
        }
    }
    
    /**
     * Move the carrier to Handoff position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to Engaged position.",
            timeout=TIMEOUT_FOR_MOVING_CARRIER)
    public void goToEngaged()  {
        if (this.isAtEngaged()) {
            FCSLOG.info(name + " is already at Engaged position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(this.engagedPosition);
            this.absoluteTargetPosition = this.engagedPosition;
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ENGAGED,
                    timeoutForGoingToEngaged);
            afterMotion();
        }
    }    

    /**
     * Moves loader carrier to storage position.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to STORAGE position.",
            timeout=TIMEOUT_FOR_MOVING_CARRIER)
    public void goToStorage()  {
        if (this.isAtStorage()) {
            FCSLOG.info(name + " is already at Storage position. Nothing to do.");
        } else {
            loader.checkConditionsForCarrierMotion(this.storagePosition);
            this.absoluteTargetPosition = this.storagePosition;
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_STORAGE,
                    timeoutForGoingToStorage);
            afterMotion();
        }
    }
    
    /**
     * For end user.
     * Move the carrier to absolute position given as argument (in decimal format).
     * @param absolutePosition 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move the carrier to absolute position given as argument (in decimal format).",
            timeout=TIMEOUT_FOR_MOVING_CARRIER)    
    public void goToAbsolutePosition(int absolutePosition) {
        this.updatePosition();
        if (position == absolutePosition) {
            FCSLOG.info(name + " is already at position "+absolutePosition);
        } else {
            loader.checkConditionsForCarrierMotion(absolutePosition);
            this.absoluteTargetPosition = absolutePosition;
            this.executeAction(FcsEnumerations.MobileItemAction.MOVE_LOADERCARRIER_TO_ABSOLUTEPOSITION,
                    timeoutForGoingToStorage);
            afterMotion();
        }
    }
    
    private void checkProfileVelocity() {
        if (loader.isHoldingFilter() 
                && position >= engagedPosition && absoluteTargetPosition >= engagedPosition) {
            long velocity = this.readProfileVelocity();
            if ( velocity > this.lowSpeed) {
                throw new RejectedCommandException(name + " ProfileVelocity too high to move a filter "
                        + "between ENGAGED and HANDOFF positions.");
            }
        }
    }
    
    /**
     * change ProfileVelocity parameter to lowSpeed
     *
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileVelocity parameter to lowSpeed")
    public void slowProfileVelocity() {
        this.profileVelocity = lowSpeed;
        carrierController.changeProfileVelocity(lowSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + lowSpeed);
        slowProfileAcceleration();
        slowProfileDeceleration();
    }
    
    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileVelocity parameter to highSpeed")
    public void raiseProfileVelocity() {
        this.profileVelocity = highSpeed;
        carrierController.changeProfileVelocity(highSpeed);
        FCSLOG.info(name + " new ProfileVelocity=" + highSpeed);
        raiseProfileAcceleration();
        raiseProfileDeceleration();
    }
    
    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileAcceleration "
            + "parameter to highAcceleration")
    public void raiseProfileAcceleration() {
        carrierController.writeParameter(ProfileAcceleration, highAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + highAcceleration);
    }    
    
    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileAcceleration "
            + "parameter to lowAcceleration")
    public void slowProfileAcceleration() {
        carrierController.writeParameter(ProfileAcceleration, lowAcceleration);
        FCSLOG.info(name + " new ProfileAcceleration=" + lowAcceleration);
    } 
    
    /**
     * change ProfileVelocity parameter to highSpeed
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileAcceleration "
            + "parameter to highAcceleration")
    public void raiseProfileDeceleration() {
        carrierController.writeParameter(ProfileDeceleration, highDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + highDeceleration);
    }    
    
    /**
     * change ProfileAcceleration parameter to lowAcceleration
     */
    @Command(type = Command.CommandType.ACTION, description="change ProfileAcceleration "
            + "parameter to lowAcceleration")
    public void slowProfileDeceleration() {
        carrierController.writeParameter(ProfileDeceleration, lowDeceleration);
        FCSLOG.info(name + " new ProfileDeceleration=" + lowDeceleration);
    }      
    
    /**
     * read ProfileVelocity parameter in carrierController CPU.
     * just for end user facility
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, 
            description="read ProfileVelocity parameter in carrierController CPU")
    public long readProfileVelocity() {
        return carrierController.readParameter(ProfileVelocity);
    }

    /**
     * Tests if action is completed. 
     * During motion of the carrier, this methods tests if the desired position has been reached.
     * If yes, it returns true, else false.
     * @param action to be tested
     * @return true if required position has been reached
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        return this.position == this.absoluteTargetPosition;
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carrier position in reading controller.")
    public void updatePosition()  {
        this.position = carrierController.readPosition();
        this.publishData();
    }
    
    @Override
    public void updateCurrent() {
        this.current = this.carrierController.readCurrent();
    }

    /**
     * To display position for end user.
     * Updates carrier position in reading controller and returns it.
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "To update and display position for end user."
                    + "Updates carrier position in reading controller and returns it.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }


    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carrierController.checkFault();
            this.position = this.carrierController.readPosition();
            updateCurrent();
            //for the GUI : is it the right place ?
            loader.updateStateWithSensors();
            this.speed = this.carrierController.readVelocity();
        } catch (SDORequestException ex) {
            this.raiseWarning(SDO_ERROR,
                    "error in updateStateWithSensorsToCheckIfActionIsCompleted: " , name, ex);
        } 
    }

    /**
     * Starts an action on the carrier.
     * On the loader carrier, the action in normal mode are motion of the carrier to go to STORAGE position 
     * or HANDOFF position.
     * In engineering mode, the end user can move the carrier to a given absolute position.
     * @param action to be started
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        carrierController.checkFault();
        checkProfileVelocity();
        carrierController.enableAndWriteAbsolutePosition(this.absoluteTargetPosition);
    }
    
    /**
     * Aborts an action.
     * @param action
     * @param delay 
     */
    @Override
    public void abortAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " STOPPING action " + action.toString()
                + " within delay " + delay);
        /* if we disable controller it will be offline on the CANbus at the next startup of FCS */
        this.carrierController.disableVoltage();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " is STOPPING action " + action.toString()
                + " within delay " + delay);
        //doesn't work : we can't do a quickStop when in PROFILE_POSITION
        //this.carrierController.quickStop();
        this.carrierController.stopPosition();
    }

    
    /**
     * to be executed after carrier motion.
     * abortAction has been executed prior this method.
     */
    private void afterMotion() {
        checkPositionSensors();
        loader.updateFCSStateToReady();
        this.absoluteTargetPosition = 0;
    }
    
    /**
     * check that position sensors confirm position at the end of a motion to absoluteTargetPosition.
     */
    public void checkPositionSensors() {
        if (this.absoluteTargetPosition == this.storagePosition && !this.isAtStorage()) {
            throw new FailedCommandException(name + ": storage sensors don't confirm carrier position.");
            
        } else if (this.absoluteTargetPosition == this.engagedPosition && !this.isAtEngaged()) {
            throw new FailedCommandException(name + ": engaged sensors don't confirm carrier position.");
            
        } else if (this.absoluteTargetPosition == this.handoffPosition && !this.isAtHandoff()) {
            throw new FailedCommandException(name + ": handoff sensors don't confirm carrier position.");
        } 
    }

    /**
     * Command for ENGINEERING mode. Not used during INITIALIZATION.
     * Check that carrier controller is initialized.
     * Read sensors and update state.
     * Read carrier position on controller.
     * If position sensors and position read on carrier controller give inconsistent information
     * throw a FcsHardwareException.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Check if hardware is ready to be started.")
    public void initializeHardware()  {
        carrierController.checkInitialized();

        loader.updateStateWithSensors();
        try {
            updatePosition();
        } catch (SDORequestException ex) {
            String msg = name + ": couldn't update position";
            FCSLOG.error(msg);
            throw new FcsHardwareException(name, ex);
        }
        if (isCarrierAlmostAtPosition(this.handoffPosition) && !this.isAtHandoff()) {
            throw new FcsHardwareException(name
                    + ": handoff sensors don't confirm position read on carrierController.");
        }
        if (isCarrierAlmostAtPosition(this.engagedPosition) && !this.isAtEngaged()) {
            throw new FcsHardwareException(name
                    + ": engaged sensors don't confirm position read on carrierController.");
        }        
        if (isCarrierAlmostAtPosition(this.storagePosition) && !this.isAtStorage()) {
            throw new FcsHardwareException(name
                    + ": storage sensors don't confirm position read on carrierController.");
        }
        this.initialized = true;
        loader.updateFCSStateToReady();
    }
    
    /**
     * Returns true if carrier position is in a range of 2*deltaPosition around a given position.
     * @param position
     * @return 
     */
    private boolean isCarrierAlmostAtPosition(int position) {
        return this.position > position - deltaPosition && this.position < position + deltaPosition;
    }
    
    /**
     * Return a printed list of hardware with initialization information.
     * For debug purpose.
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Return a printed list of hardware with initialization information.")    
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (this.isInitialized()) {
            sb.append(" is INITIALIZED.");
        } else {
            sb.append(" is NOT INITIALIZED.");
        }
        return sb.toString();
    }
    
    /**
     * Creates an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByLoaderCarrier createStatusDataPublishedByLoaderCarrier() {
        StatusDataPublishedByLoaderCarrier status = new StatusDataPublishedByLoaderCarrier();
        status.setPosition(position);
        status.setSsiPosition(ssiPosition);
        status.setSpeed(speed);
        status.setProfileVelocity(profileVelocity);
        status.setCurrent(current);
        status.setAtHandoff(handoffSensors.isOn());
        status.setAtStorage(storageSensors.isOn());
        status.setAtEngaged(engagedSensors.isOn());
        status.setHandoffInError(handoffSensors.isInError());
        status.setStorageInError(storageSensors.isInError());
        status.setEngagedInError(engagedSensors.isInError());
        status.setControllerInError(carrierController.isInError());
        status.setHomingDone(initialized);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderCarrier status = this.createStatusDataPublishedByLoaderCarrier();
        KeyValueData kvd = new KeyValueData("loaderCarrier", status);
       s.publishSubsystemDataOnStatusBus(kvd);
    }
    
    @Override
    public void shutdown() {
        super.shutdown();
        if (carrierController.isBooted()) {
            carrierController.disableVoltage();
        }
        FCSLOG.info(name + " is shutdown.");
    }


}
