package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.PROFILE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console.
 *
 *
 * @author virieux
 *
 */
public class Carousel extends MobileItem implements HardwareController, FilterHolder, AlertRaiser {

    private static final long serialVersionUID = -2376279469784152348L;
    
    @LookupField(strategy = Strategy.TOP)
    private Subsystem subs;

    @ConfigurationParameter(range = "0..500000")
    private int rotationTimeout = 20000;

    private int position = 0;
    private int relativeTargetPosition;

    private final int speed = 0;
    private int current = 0;

    /**
     * 0.1 tour = 436224 pas (2Pi = 4260 X 1024)
     * 1 socket = 0.2 tour = 1/5 tour = 872448
     */
    private final int stepNB = 4362240;
    private final int halfTurn = stepNB / 2;

    /* si encoderSign = -1, l'encodeur va de 0 a -stepNB*/
    /* si encoderSign = 1, l'encodeur va de 0 a stepNB*/
    @ConfigurationParameter(description = "if encoderSign = 1, encoder goes from 0 to stepNB,"
            + "if encoderSign = -1, encoder goes from 0 to stepNB")
    private int encoderSign = -1;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = Strategy.BYNAME) 
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    @LookupField(strategy = Strategy.BYNAME)
    private EPOSControllerForCarousel carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = Strategy.BYNAME)
    private TTC580Interface hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    @LookupField(strategy = Strategy.BYNAME)
    protected PieceOfHardware ai814;

    /*To be able to know if the autochanger holds a filter. */
    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    /**
     * A map to store the sockets by their names. The key of this map is the
     * socket name.
     * This map is built by Toolkit during INITIALISATION.
     *
     */
    @LookupField(strategy = Strategy.CHILDREN)
    protected final Map<String, CarouselSocket> socketsMap = new TreeMap<>();

    @LookupField(strategy = Strategy.BYNAME)
    protected BridgeToHardware tcpProxy;

    private boolean initialized = false;
    private boolean clampsStateInitialized = false;
    private boolean homingDone = false;
    
    private int socketAtStandbyID;
    private CarouselSocket socketAtStandby;
    

    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void build() {
//        /**
//         * What has to be done for each tick of the timer. We have to publish on the
//         * status bus to refresh the GUI.
//         */
//        periodicTaskService.scheduleAgentPeriodicTask(
//                new AgentPeriodicTask(name+"-publishData",this::publishData)
//                        .withIsFixedRate(true)
//                        .withLogLevel(Level.WARNING)
//                        .withPeriod(Duration.ofSeconds(5)));
    }

    /**
     * ***********************************************************************
     */
    /**
     *
     * @return true if CANopen devices are booted and initialized and homing has
     * been done.
     */
    @Command(description = "Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        //TODO test also if homing has been done.
        return this.myDevicesReady();
    }

    /**
     * 
     * @return true if clamp state is initialized for all clamps
     */
    @Command(description = "Return true if clamp state is initialized for all clamps.")
    public boolean isClampsStateInitialized() {
        return clampsStateInitialized;
    }
    
    

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if carouselController homing is done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     * used by tests.
     * @return 
     */
    public int getStepNB() {
        return stepNB;
    }

    /**
     * used by tests.
     * @return 
     */
    public int getEncoderSign() {
        return encoderSign;
    }

    public Map<String, CarouselSocket> getSocketsMap() {
        return socketsMap;
    }
    
    

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return carousel position.", alias = "printPosition")
    public int getPosition() {
        return position;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    /**
     * This method returns the clampX- which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXminus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXplus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     * otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")
    public boolean isAtStandby() {
        //TODO this is not consistent with method CarouselSocket.isAtStandby();
        return socketAtStandbyID >= 1 && socketAtStandbyID <= 5;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one.
     * Otherwise return null.
     *
     * @return
     */
    public CarouselSocket getSocketAtStandby() {
        return socketAtStandby;
    }

    public int getSocketAtStandbyID() {
        return socketAtStandbyID;
    }
    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Returns name of filter at STANDBY position  or"
            + " NO_FILTER if there is no filter at STANDBY.")
    public String getFilterAtStandbyName() {
        if (socketAtStandby == null) {
            return NO_FILTER;
        } else {
            return socketAtStandby.getFilterName();
        }
    }
    
    /**
     * return true if filterID in on carousel.
     * @param filterID
     * @return 
     */
    public boolean isFilterOnCarousel(int filterID) {
        return socketsMap.values().stream().anyMatch((socket) -> (socket.getFilterID() == filterID));
    }
    
    @Command(description = "To change filterID on socket which ID is given as argument.")
    public void changeFilterID(int filterID, int socketID) {
        if (socketID < 1 || socketID > 5) {
            throw new IllegalArgumentException(socketID + ": bad value - enter a digit between 1 and 5");
        }
        if (this.isFilterOnCarousel(filterID)) {
            int sockID = this.getFilterSocket(filterID).getId();
            throw new IllegalArgumentException(filterID + " filter already on carousel on socket" + sockID);
        }
        String socketName = SOCKET_NAME + socketID;
        socketsMap.get(socketName).setFilterID(filterID);
    }

    /**
     * This methods returns true if the autochanger is holding a filter at
     * STANDBY position.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if Autochanger is holding filter at STANDBY.")
    public boolean isAutochangerHoldingFilter() {
        /**
         * To be done when a switch is connected : read the switch sensor and
         * update a boolean of the class like : autochangerHoldingFilter.
         */
        autochanger.updateStateWithSensors();
        return autochanger.isHoldingFilter();
    }
    
    /**
     * Executed when all components HasLifeCycle of subsystem have been checked. 
     */
    @Override
    public void postStart()  {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (carouselController.isBooted()) {
            initializeRotationController();
        }
        if (clampXminusController.isBooted()) {
            initializeClampController(clampXminusController);
        }
        if (clampXplusController.isBooted()) {
            initializeClampController(clampXplusController);           
        }  
        try {
            this.initializeClampsState();
        } catch (DriverException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " could not initiliaze clamps state.", name, ex);
        }
        FCSLOG.fine(name + " END postStart.");
    }
    
    private void initializeRotationController() {
        try {
            carouselController.initializeAndCheckHardware();
            this.initialized = true;  
            //During tests Guillaume said that it could be dangerous to do the homing during
            //startup.
//            homing();

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", 
                    carouselController.getName(), ex);
        }
//        if (!this.isHomingDone()) {
//            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't do homing", 
//                    carouselController.getName());
//        }
    }
    
    
    private void initializeClampController(EPOSController controller) {
        try {
            controller.initializeAndCheckHardware();
            if (controller.readMode() != EPOSEnumerations.EposMode.CURRENT) {
                this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR,
                        controller.getName() + "  is not in CURRENT mode.",
                        controller.getName());
            }
        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR,
                    name + " couldn't initialize controller",
                    controller.getName(), ex);
        }
    }    



    /**
     * homing of rotation controller
     * to be done before each rotation
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "CarouselHoming")
    public void homing() {
        homingDone = false;
        carouselController.disableVoltage();
        carouselController.setPositionSensorTypeEncoderSSI();
        int ssiPosition = carouselController.readSSIPosition();
        carouselController.setPositionSensorTypeSinusIncrementalEncoder();
        //TODO not useful in homing
        int newSSIPosition = computeNewPosition(ssiPosition);
        carouselController.enable();
        carouselController.defineAbsolutePosition(newSSIPosition);
        homingDone = true;
        carouselController.changeMode(PROFILE_POSITION);
        carouselController.disableVoltage();
        updatePosition();
        publishData();
    }

    private void checkHomingDone() {
        if (!isHomingDone()) {
            throw new FcsHardwareException(name + " homing has to be done before motion.");
        }
    }

    public void checkSensorTypeIncremental() {
        int sensorType = carouselController.readPositionSensorType();
        if (sensorType != 8) {
            throw new FcsHardwareException(name + " PositionSensorType has to be set to Incremental Encoder before motion.");
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData("carousel",
                createStatusDataPublishedByCarousel()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the
     * STATUS bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setPosition(position);
        status.setCurrent(current);
        status.setSpeed(speed);
        status.setHomingDone(homingDone);
        status.setAtStandby(isAtStandby());
        status.setSocketAtStandbyID(socketAtStandbyID);
        status.setFilterAtStandbyName(getFilterAtStandbyName());
        if (isAtStandby()) {
            status.setSocketAtStandbyName(socketAtStandby.getName());
        } else if (socketAtStandbyID == 0) {
            status.setSocketAtStandbyName("NO_SOCKET_AT_STANDBY");
        } else if (socketAtStandbyID == 7) {
            status.setSocketAtStandbyName("ERROR_READING_ID");
        }
        
        return status;
    }


    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the
     * filterPosition when one of its sockets is at standby filterPosition and
     * this socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() {

        if (this.isRotating()) {
            return false;
        }
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and carousel is
     * stopped
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() {
        if (this.isRotating()) {
            return false;
        }
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Returns true if carousel is rotating
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if carousel is rotating.")
    public boolean isRotating() {
        return isMoving();
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update carousel position in reading controller.")
    public void updatePosition() {
        if (this.homingDone) {
            try {
                this.position = carouselController.readPosition();
            } catch (SDORequestException ex) {
                FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
            }
            this.publishData();
        } else {
            throw new RejectedCommandException("Carousel homing not done : "
                    + "do carousel homing before updatePosition.");
        }
    }

    /**
     * Updates the field current in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update field current in reading controller.")
    public void updateCurrent() {
        try {
            this.current = carouselController.readCurrent();
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }


    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() {

        PDOData pdoStore = this.tcpProxy.updatePDOData();
        
        FCSLOG.finest(name + ":pdoStore=" + pdoStore.toString());
        socketsMap.values().stream().forEach(socket -> {
            socket.updateState();
        });
    }
    
    /**
     * In carousel, sensors are updated from PDOs.
     * PDOs are received 2 by 2 :
 0x180 + ttc580 nodeID for clamp at standby. (ttc580'sock pdo1)
 0x280 + ttc580 nodeID for a clamp not at standby. (ttc580'sock pdo2)
 PDO2 contains socketID not at STANDBY in turns.
 Exemple : if socket1 is at STANDBY 
 - first sync returns pdo1 with socket1 values and pdo2 with socket2 values
 - second sync returns pdo1 with socket1 values and pdo2 with socket3 values
 - third sync returns pdo1 with socket1 values and pdo2 with socket4 values
 - fourth sync returns pdo1 with socket1 values and pdo2 with socket5 values
 
 socketID is coded in each PDO.
 After FCS start we need to send 4 sync to know all clamps state.
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update clamps state in sending 5 sync.")
    public void initializeClampsState() throws DriverException {
        FCSLOG.info(name + " Initializing clamps state....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        byte count = 1;
        while (count <= 5) {
            FCSLOG.info(name + " sync no " + count);
            hyttc580.updateFromPDO(tcpProxy.updatePDOData());
            updateState();
            count++;
        }
        clampsStateInitialized = true;
        publishData();
    }



    /**
     * This method has no action on hardware. This method updates the
     * socketAtStandby at standby filterPosition when a filter is put on the
     * carousel at standby filterPosition. A filter must be detected in the
     * socketAtStandby at standby filterPosition (socketAtStandby not empty).
     *
     * @param filter
     * @throws RejectedCommandException
     */
    private synchronized void putFilterOnCarousel(Filter filter) {

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ": there is no socket at standby "
                    + "position to put a filter in.");
        }

        socketAtStandby.setFilterID(filter.getFilterID());
        this.publishData();
    }
    
    /**
     * 
     * @param filterID
     * @return null if filter is not on carousel, else return socket where filter is stored.
     */
    public CarouselSocket getFilterSocket(int filterID) {
        CarouselSocket socket = null;
        for (CarouselSocket sock: socketsMap.values()) {
            if (sock.getFilterID() == filterID) {
                return sock;
            }
        } 
        return socket;
    }

    private int getStandbyPositionForFilter(Filter filter) {
        CarouselSocket socket = getFilterSocket(filter.getFilterID());
        if (socket == null) {
            throw new IllegalArgumentException(name + " filter is not on carousel " + filter.getName());
        } else {
            return socket.getStandbyPosition();
        }
    }


    /**
     * This method has to be executed when a filter has just been moved on the
     * carousel at standby filterPosition by the autochanger. It updates the
     * carousel socket at standby or throw an exception.
     *
     * This method tests if the clamps are Locked on a filter at standby
     * filterPosition The clamps Locks automaticaly, and the filter presence
     * sensor tells us if the filter is here or not. If the clamps are Locked on
     * a filter at standby filterPosition, this method updates the
     * socketAtStandby at standby filterPosition : it puts the filter on the
     * socketAtStandby.
     *
     * @param filter
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public synchronized String grabFilterAtStandby(Filter filter) {
        if (filter == null) {
            throw new IllegalArgumentException(name + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateState();

        if (socketAtStandby.isEmpty()) {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }

        if (socketAtStandby.isClampedOnFilter()) {
            FCSLOG.info(name + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);

            FCSLOG.info(name + ": " + filter.getName() + " is grabbed on carousel.");
            return name + ": " + filter.getName() + " is grabbed at standby position";
        } else {
            throw new RejectedCommandException(name + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }

    }

    /**
     * This unlock clamps at STANDBY.
     *
     * @param filterName
     * @throws FcsHardwareException
     */
    public void ungrabFilterAtStandby(String filterName) {

        if (filterName == null) {
            throw new IllegalArgumentException("Filter to ungrabb should not be null");
        }

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name + ":A socket has to be halted at standby position for this operation.");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new RejectedCommandException(name + ": can't ungrabb filter which is already unclamped.");
        }

        FCSLOG.info(name + ": ungrabbing " + filterName + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.
        // The clamps can be opened.
        this.unlockClamps();
        FCSLOG.info(name + ": socket at standby state : " + socketAtStandby.toString());

        FCSLOG.info(name + ": " + filterName + " is ungrabbed from carousel.");
    }

    /**
     * Release clamps at standby position to get ready to clamp again.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release clamps at standby position to get ready to clamp again")
    public void releaseClamps() {

        if (socketAtStandby == null) {
            throw new RejectedCommandException(name
                    + ":Can't release clamps when no socket is halted at standby position.");
        } else {
            socketAtStandby.releaseClamps();
        }
    }

    /**
     * Unlocks the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Unlock the clamps at STANDBY.")
    public void unlockClamps() {

        if (socketAtStandby == null) {
            throw new RejectedCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        } else {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();
        }
    }

    /**
     * Return true if the filter given as argument is at STANDBY position.
     *
     * @param filter
     * @return
     */
    public boolean isAtStandby(Filter filter) {
        return this.getPosition() == this.getStandbyPositionForFilter(filter);
    }

    public void releaseClampsContact() {
        //TODO;
    }

    @Override
    public boolean myDevicesReady() {
        return this.tcpProxy.allDevicesBooted() && this.initialized;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ ROTATION COMMANDS *********************************
     */
    /**
     * ****************************************************************
     */
    /**
     * Check if carousel rotation is permitted.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException
     * (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation() {
        String message;
        if (!initialized) {
            throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        }

        //commented out in July 2017 because no TTC-580
//        updateState();
        if (this.isAtStandby() && socketAtStandby.isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.error(message);
            throw new RejectedCommandException(message);
        }

        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochanger is not at HANDOFF position.");
        }
    }

    /**
     * A command to rotate carousel.
     *
     * @param relativePosition
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel to a relative position.",
            timeout = 60000)
    public void rotateToRelativePosition(int relativePosition) {

//        checkConditionsForRotation();
        if (Math.abs(relativePosition) > halfTurn) {
            throw new IllegalArgumentException(relativePosition + ":  |relativePosition| should be <= " + halfTurn);
        }

        FCSLOG.info(name + " is at position: " + position + "; about to rotate to relative position: " + relativePosition);
        relativeTargetPosition = relativePosition;

        //rotate the carousel
        this.executeAction(MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION, rotationTimeout);
        
        if (!s.isInState(AlertState.ALARM)) {
            s.updateAgentState(FcsEnumerations.FilterState.CAROUSEL_STOPPED);
        }
        this.publishData();
    }

    /**
     * Rotate carousel to move a socket which name is given as argument to
     * STANDBY position. This methods computes the shortest way to go to STANDBY
     * position.
     *
     * @param socketName
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move a socket which name is given as argument to STANDBY position.",
            alias = "moveSocketToStandby",
            timeout = 50000)
    public void rotateSocketToStandby(String socketName) {

        FcsUtils.checkSocketName(socketName);

        CarouselSocket socketToMove = this.socketsMap.get(socketName);

        int requiredPosition = socketToMove.getStandbyPosition();

        String msgWhenCompleted = name + ":" + socketName + " is at STANDBY position on carousel.";

        if (position != requiredPosition) {
            FCSLOG.info(name + " is at position: " + position + "; about to rotate to position: "
                    + requiredPosition);
            int deltaPosition = requiredPosition - position;
            FCSLOG.info(name + " delta position = " + deltaPosition);
            /* compute shortest way*/
            relativeTargetPosition = computeShortestRelativePosition(deltaPosition);
            rotateToRelativePosition(relativeTargetPosition);
        }
        FCSLOG.info(msgWhenCompleted);
    }

    /**
     * Move an empty socket to STANDBY position. The socket moved is the first
     * empty one in the socketsMap.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void rotateEmptySocketAtStandby() {
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.isEmpty()) {
                rotateSocketToStandby(entry.getKey());
                return;
            }
        }
        FCSLOG.error("no empty socket on carousel - nothing to do.");
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            return carouselController.isTargetReached();
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            carouselController.checkFault();
            updatePosition();
            updateCurrent();
            FCSLOG.debug(name + " position=" + this.position);
        } catch (SDORequestException ex) {
            FCSLOG.warning(name + "=> SDO ERROR IN READING CONTROLLER:", ex);
        }
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or
     * ROTATE_CAROUSEL_TO_RELATIVE_POSITION. This : - enable controller, - make
     * controller go to required position previously written in field
     * absoluteTargetPosition or relativeTargetPosition, - writeControlWord "3F"
     * or "7F" on controller depending if we want to go to a relative position
     * or to an absolute position.
     *
     * @param action
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {

        /* carouselController should be initialized*/
        carouselController.checkInitialized();

        /* carouselController should not be in fault*/
        carouselController.checkFault();

        /* a homing must be done before every motion*/
        homing();
        checkHomingDone();

        s.updateAgentState(FcsEnumerations.FilterState.valueOf("ROTATING"),
                FcsEnumerations.FilterReadinessState.NOT_READY);

        if (MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            carouselController.enableAndWriteRelativePosition(this.relativeTargetPosition);
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.debug(name + " is ABORTING action " + action.toString() + " within delay " + delay);
        /* a tempo to let time to the motion command to complete.*/
        try {
            Thread.sleep(3000);
        } catch (InterruptedException ex) {
            FCSLOG.warning(ex);
        }
        this.carouselController.stopPosition();

        /* At the end of an action a homing has to be done.*/
        homing();
        this.carouselController.disableVoltage();

    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.carouselController.stopPosition();
        this.carouselController.disableVoltage();
    }


    /**
     *
     * @param pos
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "compute new position to take in account tours which can have been runned")
    public int computeNewPosition(int pos) {
        int newPos;
        int modulo = pos % stepNB;
        if (pos * encoderSign > 0 || modulo == 0) {
            newPos = modulo;
        } else {
            newPos = encoderSign * (stepNB - Math.abs(modulo));
        }
        FCSLOG.finest("stepNB=" + stepNB + " pos=" + pos + " modulo=" + modulo
                + " ==> new Position = " + newPos);
        return newPos;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Compute relative position to find the shortest way to run carousel.")
    public int computeShortestRelativePosition(int d) {
        int relativePosition;
        if (Math.abs(d) <= halfTurn) {
            relativePosition = d;
        } else {
            int s;
            if (d > 0) {
                s = -1;
            } else {
                s = 1;
            }
            relativePosition = s * Math.min(Math.abs(d), stepNB - Math.abs(d));
        }
        return relativePosition;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of ROTATION COMMANDS **************************
     */
    /**
     * ****************************************************************
     */

    /**
     * ****************************************************************
     */
    /**
     * ************ methods which override FilterHolder *** ***********
     */
    /**
     * ****************************************************************
     */
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Override
    public boolean isHoldingFilter() {
        return this.isHoldingFilterAtStandby();
    }
    
    //TODO find something more cleaver to do here.
    @Override
    public boolean isNotHoldingFilter() {
        return !this.isHoldingFilterAtStandby();
    }    

    @Override
    public boolean isAtHandoff() {
        return false;
    }
    
    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read sensors and update state", timeout=2000)
    @Override
    public void updateStateWithSensors() {
        long beginTime = System.currentTimeMillis();
        if (!clampsStateInitialized) {
            throw new FcsHardwareException(name + ": clamps state not initialized. "
                    + " Please launch command initializeClampsState.");
        }
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        try {
            hyttc580.updateFromPDO(tcpProxy.updatePDOData());
            updateState();
            
        } catch (DriverException ex) {
            throw new FcsHardwareException(name + "Can't updatePDO data.", ex);
        }
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateStateWithSensors duration = " + duration);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update socket at standby state from hyttc580 data.", timeout=2000)    
    public void updateStateWithSensorsAtStandby() {
        long beginTime = System.currentTimeMillis();
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        try {
            hyttc580.updateFromPDO(tcpProxy.updatePDOData());
            long duration = System.currentTimeMillis() - beginTime;
            FCSLOG.info(name + " hyttc580.updateFromPDO(tcpProxy.updatePDOData()) duration = " + duration);
        } catch (DriverException ex) {
            throw new FcsHardwareException(name + "Can't updatePDO data.", ex);
        }
        updateSocketAtStandbyState();
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateStateWithSensorsAtStandby duration = " + duration);
    }
    
    public void updateSocketAtStandbyState() {
        long beginTime = System.currentTimeMillis();
        this.socketAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo1());
        if (socketAtStandbyID >= 1 && socketAtStandbyID <= 5) {
            socketAtStandby = socketsMap.get("socket" + socketAtStandbyID);
            socketAtStandby.updateState();
            socketAtStandby.getClampXminus().setAtStandby(true);
            socketAtStandby.getClampXplus().setAtStandby(true);
        } else {
            socketAtStandby = null;
        }
        publishData();
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateSocketAtStandbyState duration = " + duration);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update state from hyttc580 data.", timeout=2000)
    public void updateState() throws DriverException {
        long beginTime = System.currentTimeMillis();
        updateSocketAtStandbyState();
        // The use of the scheduler is required in order to leave current thread as soon as possible.
        subs.getScheduler().schedule(() -> {
            updateSocketNotAtStandby();
        }, 0, TimeUnit.SECONDS);
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateState duration = " + duration);
    }
    
    public void updateSocketNotAtStandby() {
        long beginTime = System.currentTimeMillis();
        int socketNotAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo2());
        if (socketNotAtStandbyID >= 1 && socketNotAtStandbyID <=5) {
            CarouselSocket socketToUpdate = socketsMap.get("socket" + socketNotAtStandbyID);
            socketToUpdate.updateState();
            socketToUpdate.getClampXminus().setAtStandby(false);
            socketToUpdate.getClampXplus().setAtStandby(false);
        }
        long duration = System.currentTimeMillis() - beginTime;
        FCSLOG.info(name + " updateSocketNotAtStandby duration = " + duration);
    }


    /**
     * ****************************************************************
     */
    /**
     * ************ END of methods which override FilterHolder ********
     */
    /**
     * ****************************************************************
     */
}
