
package org.lsst.ccs.subsystems.fcs.testbench;

import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenThermometer;

/**
 * This is the main module for the control software of the APC test bench.
 * This testbench is made of 2 CAN open devices :
 * - CAN-CBX-ai814
 * - CAN-CBX-ao412
 * - a thermometer connected on the ai814
 * - a led connected on the ao412
 *
 *
 * @author virieux
 */
public class APCTestBenchMain {

    private final BridgeToHardware bridge;

    /**
     * * The thermometer mesures the temperature inside the clamp.
     */
    private final CanOpenThermometer thermometer;
    private final CanOpenLED led;
    private final CanOpenLED fakeLed;
    private final CanOpenThermometer fakeSensor;

    public double temperature;

    public APCTestBenchMain(
            BridgeToHardware bridge, 
            CanOpenThermometer thermometer, 
            CanOpenLED led, CanOpenLED fakeLed, 
            CanOpenThermometer fakeSensor) {
        this.bridge = bridge;
        this.thermometer = thermometer;
        this.led = led;
        this.fakeLed = fakeLed;
        this.fakeSensor = fakeSensor;
    }

    /**
     * Returns the value read on the socket for the input of the ADC given as an
     * argument.
     *
     * @param responseLine A string read on the socket.
     * @param inputNB The numero if the input on the ADC. (1<=inputNB<=8)
     * @return the value (int) given
     */
    public int getReturnValue(String responseLine, int inputNB) 
    {
        if (inputNB < 1 || inputNB > 8) 
        {
            throw new IllegalArgumentException();
        }
        String[] words = responseLine.split(",");
        int value = Integer.parseInt(words[inputNB], 16);
        FCSLOG.debug("Value in decimal:" + words[inputNB]);
        return value;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Read temperature with readSDO commands.")     
    public double readTemperature()  {
        return this.thermometer.readTemperature();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Read temperature from PDO.")    
    public double readTemperatureFromPDO()   {
        bridge.updatePDOData();
        return this.thermometer.readTemperatureFromPDO();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Turn ON the Led.") 
    public String turnOnLED()  {
        return led.turnON();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Turn OFF the Led.")     
    public String turnOffLED()  {
        return led.turnOFF();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Turn ON the fake Led.")    
    public String turnOnFakeLED()  {
        return fakeLed.turnON();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Turn OFF the fake Led.")     
    public String turnOffFakeLED()  {
        return fakeLed.turnOFF();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Read fake Sensor.")    
    public double readFakeSensor()  {
        return this.fakeSensor.readTemperatureFromPDO();
    }


}
