
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * To simulate the controller which rotates carousel.
 * @author virieux
 */
public class SimuCarouselController extends SimuEPOSController implements EPOSControllerForCarousel {
    
    @LookupField(strategy=Strategy.TREE)
    private Carousel carousel;
    
    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = Strategy.TREE)
    private SimuTTC580 hyttc580;
    
    /**
     * Build a new SimuCarouselController
     * @param nodeID
     * @param serialNB
     */
    public SimuCarouselController(
            int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    /**
     * To initialize a simulated ssiPosition.
     * @param ssiPosition 
     */
    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
        this.position = ssiPosition;
    }    
    
    @Override
    public void init() {
        /* when simulation starts carousel position is set to socket1 standby position.*/
        setSsiPosition(carousel.getSocketByName("socket1").getStandbyPosition());
    }
    
    @Override
    public void enableAndWriteRelativePosition(int pos) {
        enable();
        changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
        writeTargetPosition(pos);
        writeControlWord(0x7F);
    }     
        
    
    /**
     * This methods does nothing but save the relativePosition given as argument.
     * @param absolutePosition 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    @Override
    public void writeTargetPosition(int absolutePosition) throws FcsHardwareException {
        FCSLOG.info(name + " => carousel is rotating.");
        FCSLOG.info(name + " => initial position=" + this.position);
        FCSLOG.info(name + " => target position=" + absolutePosition);
        int iterNB = 10;
        int initialPosition = carousel.getPosition();
        int step = (absolutePosition - initialPosition) / iterNB;
        FCSLOG.info("step=" + step);
        try {
            for (int i = 1; i < iterNB; i++) {
                int newPos = initialPosition + (i * step);
                setSsiPosition(newPos);
                FCSLOG.info(name + " i=" + i + ", position=" + position);
                try {
                    Thread.sleep(250);
                    if (carousel.getHaltRequired().get()) {
                        FCSLOG.fine(name + " STOP simulated carousel motion.");
                        return;
                    }
                } catch (InterruptedException ex) {
                    throw new FcsHardwareException(name + " sleep was interrupted.",ex);
                }
            }
            int finalPos = absolutePosition;
            setSsiPosition(finalPos);
            carousel.updatePosition();
            simulateSocketSensors(finalPos);

        } catch (FcsHardwareException  ex) {
            FCSLOG.error(name+" should not raise an Exception:",ex);
        }
    }
    
    private void simulateSocketSensors(int pos) {
        System.out.println(name + " simulateSocketSensors with position = " + pos);
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (socket.getStandbyPosition() == pos) {
                    hyttc580.simulateSocketAtStandby((short)socket.getId());
                    hyttc580.simulateSlaveStatus((short)1);
            } else {
                    hyttc580.simulateSocketNotAtStandby((short)socket.getId());
                    hyttc580.simulateSlaveStatusNotAtS((short)2);
            }
            try {
                tcpProxy.updatePDOData();
                carousel.updateState();
            } catch (DriverException ex) {
                FCSLOG.error(name + " error in simulation ", ex);
            }
        });
    }
    
    
}
