
package org.lsst.ccs.subsystems.fcs.simulation;

import static java.lang.Math.abs;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.Autochanger;

import org.lsst.ccs.subsystems.fcs.AutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.AutochangerTwoLatches;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.MASTER_ENCODER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * To simulate a controller which moves autochanger trucks along its linear
 * rails.
 *
 * @author virieux
 */
public class SimuAutochangerLinearRailController extends SimuEPOSController implements EPOSControllerForLinearRail {

    @ConfigurationParameter(description = "minimal position of the encoder ribbon.")
    private int encoderRibbonMinValue;

    @LookupField(strategy=Strategy.TREE)
    private Autochanger autochanger;
    private AutochangerTwoTrucks trucks;
    private final SimuAutochangerPlutoGateway plutoGateway;
    
    @LookupField(strategy=Strategy.BYNAME)
    private FilterHolder carousel;
    private AutochangerTwoLatches latches;

    /* this field is null for Slave controller, it'subs initialized to Slave controller for the Master controller.*/
    private SimuAutochangerLinearRailController slaveController;
    
    //in the simulation, homing is always done.
    private boolean homingDone = true;
    
    private boolean brakeActivated;

    /**
     * Build a new SimuAutochangerLinearRailController
     *
     * @param nodeID
     * @param serialNB
     * @param plutoGateway
     */
    public SimuAutochangerLinearRailController(
            int nodeID, String serialNB,
            SimuAutochangerPlutoGateway plutoGateway) {
        super(nodeID, serialNB);
        this.plutoGateway = plutoGateway;
    }

    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
    }

    public int getEncoderRibbonMinValue() {
        return encoderRibbonMinValue;
    }
    
    
    private void setTrucksPosition(int pos) {
        if (this.getName().contains("Master")) {
            this.position = pos;
            this.slaveController.setPosition(pos);
        } else {
            this.position = pos;
        }
        this.ssiPosition = this.encoderRibbonMinValue + pos;
    }

    @Override
    public void init() {
        ComponentLookup lookup = subs.getComponentLookup();
        this.trucks = (AutochangerTwoTrucks) lookup.getComponentByName("autochangerTrucks");
        this.latches = (AutochangerTwoLatches) lookup.getComponentByName("latches");
        this.setSsiPosition(encoderRibbonMinValue);
        /* if this controller is Master controller, field slaveController is initialized to linearRailSlaveController*/
        if (this.getName().contains("Master")) {
            this.slaveController = (SimuAutochangerLinearRailController) lookup.getComponentByName("linearRailSlaveController");
        } 
        if (this.getName().contains("Slave")) {
            this.mode = MASTER_ENCODER;
        }
    }
    

    /**
     * Simulates a command writeTargetPosition : the position given as argument
     * is the position to be reached. This method updates position incremently
     * to simulate a real trucks with a motor controller.
     *
     * @param positionToReached
     */
    @Override
    public void writeTargetPosition(int positionToReached) {
        this.targetPosition = positionToReached;
        int positionInitiale = this.position;
        FCSLOG.debug("loader trucks is moving.");
        FCSLOG.debug(name + "=>position intitiale=" + position);
        int stepsNB;
        if (abs(targetPosition - this.position) > 10000) {
            stepsNB = 20;
        } else {
            stepsNB = 5;
        }

        int stepHeight = (targetPosition - this.position) / stepsNB;

        simulateTrucksMotion(positionInitiale, stepsNB, stepHeight);

        setTrucksPosition(targetPosition);
        trucks.updatePosition();
        trucks.publishData();
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());

        if (position == trucks.getHandoffPosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsAtHandoff");
            this.plutoGateway.simulateAutochangerTrucksAreAtHandoff();

        } else if (position == trucks.getStandbyPosition()) {
            this.plutoGateway.simulateAutochangerTrucksAreAtStandby();
            if (carousel.isHoldingFilter()) {
                FCSLOG.info("simulateFilterIsOnAutochanger");
                this.plutoGateway.simulateFilterIsOnAutochanger();
            }

        } else if (position == trucks.getOnlinePosition()) {
            FCSLOG.info("simulateAutochangerTrucksIsOnline");
            this.plutoGateway.simulateAutochangerTrucksAreOnline();

        } else {
            FCSLOG.info("simulateAutochangerTrucksIsInTravel");
            this.plutoGateway.simulateAutochangerTrucksAreInTravel();
        }
        FCSLOG.info("plutoGateway=" + plutoGateway.toString());
    }

    /**
     * simulates FilterPresenceSensors according to the position of the trucks
     * and publishes trucks data.
     *
     * @param positionInitiale
     * @param stepHeight
     */
    private void simulateFilterPresenceSensors(int positionInitiale, int stepHeight) {
        /* on se rapproche de STANDBY subs'il y a un filtre sur le carousel il faut que l'autochanger en tienne compte*/
        if (positionInitiale < trucks.getStandbyPosition() 
                && position > trucks.getStandbyPosition() - (2 * stepHeight) 
                && carousel.isHoldingFilter()) {
            FCSLOG.info("simulateFilterIsOnAutochanger");
            this.plutoGateway.simulateFilterIsOnAutochanger();

        /* on subs'eloigne de STANDBY il faut mettre a jour l'autochanger*/
        } else if (positionInitiale == trucks.getStandbyPosition() 
                && position < trucks.getStandbyPosition() - (2 * stepHeight)
                && latches.isOpened()) {
            FCSLOG.info("simulateAutochangerIsEmpty");
            this.plutoGateway.simulateAutochangerIsEmpty();
        }
        autochanger.updateStateWithSensors();
        trucks.publishData();
    }

    /**
     * simulates trucks motion : increment or decrement position value from an
     * initialPosition, with a number of steps and a step height.
     *
     * @param positionInitiale
     * @param stepsNB number of steps
     * @param stepHeight
     */
    private void simulateTrucksMotion(int positionInitiale, int stepsNB, int stepHeight) {
        for (int i = 1; i < stepsNB; i++) {
//            this.position = positionInitiale + (i * stepHeight);
//            this.slaveController.setPosition(this.position);
            setTrucksPosition(positionInitiale + (i * stepHeight));
            trucks.updatePosition();
            FCSLOG.info("trucks Position=" + trucks.readPosition());
            simulateFilterPresenceSensors(positionInitiale, stepHeight);
            FCSLOG.debug(name + "i=" + i + ",position=" + position);
            try {
                Thread.sleep(100);
                if (trucks.getHaltRequired().get()) {
                    FCSLOG.debug(name + " STOP simulated trucks motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                FCSLOG.error(ex);
                throw new RejectedCommandException(name + " sleep was interrupted.");
            }
        }
    }

    @Override
    public void configureDigitalInputOfLinearRails() {
    }

    @Override
    public void configureDigitalOutputOfLinearRails() {
    }

    @Override
    public void homing() {
        homingDone = true;
    }

    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    @Override
    public void activateBrakeAndDisable() {
        activateBrake();
        /* mandatory otherwise issues !!!*/
        shutdownController();
        
    }

    @Override
    public void doReleaseBrake() {
        brakeActivated = false;
    }

    @Override
    public boolean isBrakeActivated() {
        return brakeActivated;
    }
    
    @Override
    public void activateBrake() {
        brakeActivated = true;
    }
}
