
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class deviceErrorFile a utility class to load the property files we need
 * to monitor a Can Open device : 
 * - device error codes 
 * - error register codes 
 * - communication errors.
 *
 * @author virieux
 */
public final class CanOpenErrorsTable {

    //the file maxonMotorDeviceErrors.properties deviceErrorFile in the directory : org-lsst-ccs-subsystem-fcs/trunk/src/main/resources
    private static final String DEVICE_ERROR_CODES_FILENAME = "/maxonMotorDeviceErrors.properties";

    //the file errorRegister.properties deviceErrorFile in the directory : org-lsst-ccs-subsystem-fcs/trunk/src/main/resources
    private static final String ERROR_REGISTER_CODES_FILENAME = "/errorRegister.properties";
    
    //the file maxonMotorDeviceErrors.properties deviceErrorFile in the directory : org-lsst-ccs-subsystem-fcs/trunk/src/main/resources
    private static final String COMM_ERROR_CODES_FILENAME = "/commErrors.properties";
    
    private static final Map<String, Map<Integer, String>> tables = new HashMap<>();
    
    static {
        /*Device Error Code (index 1003)*/
        tables.put(DEVICE_ERROR_CODES_FILENAME, new HashMap<>());
        /*Error Register Code (index 1001)*/
        tables.put(ERROR_REGISTER_CODES_FILENAME, new HashMap<>());
        /*Communication Error Code (index 1003)*/
        tables.put(COMM_ERROR_CODES_FILENAME, new HashMap<>());
    }

    /**
     * A private constructor to hide the implicit public one.
     */
    private CanOpenErrorsTable() {
        
    }
    
    public static void loadCanOpenErrorTables() {
        // Cannot use BootstrapResourceUtils because we don't want to chain properties.
        for(Map.Entry<String, Map<Integer,String>> entry : tables.entrySet()) {
            String tableName = entry.getKey();
            InputStream is = CanOpenErrorsTable.class.getResourceAsStream(tableName);
            if (is == null) {
                throw new RuntimeException("Could not find " + tableName + " properties file in resources.");
            }
            try {
                Properties props = new Properties();
                props.load(is);
                for (String name : props.stringPropertyNames()) {
                    entry.getValue().put(Integer.parseInt(name, 16), props.getProperty(name));
                }
                FCSLOG.debug(tableName + " size : " + entry.getValue().size());
            } catch (IOException ex) {
                throw new RuntimeException("could not load " + tableName + " from local resources", ex);
            }
        }
    }

    /**
     * Return the name of an error code given as a parameter in reading in the
     * error codes table.
     *
     * @param aCode device error code
     * @return the name
     */
    public static String getDeviceErrorNameByCode(int aCode) {
        return getErrorInternal(aCode, DEVICE_ERROR_CODES_FILENAME);
    }
    
    /**
     * Return the name of an error register given as a parameter in reading in the
     * error register table.
     *
     * @param aCode error register
     * @return the name
     */
    public static String getErrorRegisterNameByCode(int aCode) {
        return getErrorInternal(aCode, ERROR_REGISTER_CODES_FILENAME);
    }
    
    /**
     * Return the name of a communication error code given as a parameter in reading in the
     * communication error table.
     *
     * @param aCode error register
     * @return the name
     */
    public static String getCommErrorNameByCode(int aCode) {
        return getErrorInternal(aCode, COMM_ERROR_CODES_FILENAME);
    }
    
    private static String getErrorInternal(int aCode, String fileName) {
        String commErrorName = tables.get(fileName).get(aCode);
        if (commErrorName == null) {
            commErrorName = Integer.toHexString(aCode) + "= unknown in " + fileName;
        }
        return commErrorName;        
        
    }

}
