package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public interface EPOSControllerWithBrake extends EPOSController {


    void activateBrakeAndDisable();
    
    /**
     * Release holding brake to be able to move linear rail trucks or open/close ONLINE clamps. 
     *
     * doReleaseBrake for controller consists in
     * forcing to ONE bit 15 of parameter DigitalOutputFonctionnalityState.      
     *
     * It can't be a command because if this controller is a slave we have to check
     * if master controller is enabled. See AutoChangerTwoTrucksModule.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    default void doReleaseBrake() {
        int val = (int) readParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState);
        writeParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2one(val, 15));
        publishData();
        FCSLOG.info(getName() + ": brake released.");
    }
    
    /**
     * enableAndReleaseBrake 
     * if this controller is a slave command is rejected because we have to check 
     * if master controller is enabled. See AutoChangerTwoTrucksModule.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Enable and release brake to permit motion. Not allowed in MASTER_ENCODER mode.")
    default void enableAndReleaseBrake() {
        //OK if not master_encoder
        if (this.isInMode(EPOSEnumerations.EposMode.MASTER_ENCODER)) {
            throw new RejectedCommandException(getName() 
                    + " brake can't be released when in mode MASTER_ENCODER");
        }
        enable();
        doReleaseBrake();
    }

    /**
     * Activate brake to prevent motion.
     * A holding brake can be associated to a Controller. 
     * This brake is activated when the controller is powered on. 
     * A brake is associated with Autochanger Linear Rail trucks controllers and
     * Autochanger Online Clamps controllers.      
     *
     * activateBrake consists in forcing to ZERO bit 15 of
     * parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Activate brake to prevent motion.")
    default void activateBrake() {
        int val = (int) readParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState);
        writeParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState,
                FcsUtils.force2zero(val, 15));
        publishData();
        FCSLOG.info(getName() + ": brake activated.");
    }



    /**
     * return true if brake if activated.
     * This is when bit15 of Parameter.DigitalOutputFonctionnalityState is ZERO.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return true if brake if activated.")
    default boolean isBrakeActivated() {
        int digitalOutput = (int) readParameter(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityState);
        //TODO check the 2 other outputs
        return ((digitalOutput >> 15) & 1) == 0;
    }
    
    @Override
    default StatusDataPublishedByEPOSController createStatusDataPublishedByEPOSController() {
        StatusDataPublishedByEPOSController status 
                = new StatusDataPublishedByEPOSController(isBooted(),isInitialized());
        status.setInError(isInError());
        status.setErrorRegister(getErrorRegister());
        status.setErrorHistory(getErrorHistory());
        status.setEnabled(isEnabledToPublish());
        status.setMode(getMode());
        status.setControllerWithBrake(true);
        status.setBrakeActivated(isBrakeActivated());
        return status;
    }
    
}
