
package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerWithBrake;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * A model for an autochanger truck. 
 * The goal of this class is to factorise the code which reads AC truck sensors.
 * On the Autochanger there are 2 trucks : one on each side : 
 * truckXminus and truckXplus.
 * @author virieux
 */
public class AutochangerTruck implements MovedByEPOSController, HasLifecycle, AlertRaiser {
    
    @LookupField(strategy = Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;    
    
    @LookupField(strategy = Strategy.TOP)
    private Subsystem s;
    
    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;
    
    @LookupName
    private String name;
    
    public static final String CURRENT_MONITOR_TASK_NAME = "-monitorCurrent";
    
    private final ComplementarySensors handoffPositionSensors;
    private final ComplementarySensors onlinePositionSensors;
    private final ComplementarySensors standbyPositionSensors;
    
    private final EPOSControllerWithBrake controller;
        
    private int position;
    private int current;
    private int averageCurrent;
    private int followingError;
    private int averageVelocity;
    
    /**
     * Builds a AutochangerTruckModule with 3 couple of complementary sensors 
     * to detect position of the truck at ONLINE, HANDOFF and STANDBY position.
     * @param controller
     * @param handoffPositionSensors
     * @param onlinePositionSensors
     * @param standbyPositionSensors 
     */
    public AutochangerTruck(
            EPOSControllerWithBrake controller,
            ComplementarySensors handoffPositionSensors, 
            ComplementarySensors onlinePositionSensors, 
            ComplementarySensors standbyPositionSensors) {
        this.controller = controller;
        this.handoffPositionSensors = handoffPositionSensors;
        this.onlinePositionSensors = onlinePositionSensors;
        this.standbyPositionSensors = standbyPositionSensors;        
    }
    
    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return s;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * *** lifecycle methods *************************************************
     */
    /**
     * 
     */
    @Override
    public void build() {
        /**
         * Monitor current read on controller.
         */
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask(name+CURRENT_MONITOR_TASK_NAME,this::monitorCurrent)
                        .withIsFixedRate(true)
                        .withLogLevel(Level.WARNING)
                        .withPeriod(Duration.ofSeconds(60)));
    }
    
    /**
     * @return true if truck is at HANDOFF position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at HANDOFF position.")    
    public boolean isAtHandoff() {
        return handoffPositionSensors.isOn();
    }
    
    /**
     * @return true if truck is at ONLINE position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at ONLINE position.")
    public boolean isAtOnline() {
        return onlinePositionSensors.isOn();
    }

    /**
     * @return true if truck is at STANDBY position.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if truck is at STANDBY position.")    
    public boolean isAtStandby() {
        return standbyPositionSensors.isOn();
    }
    
    

    /**
     * Return false if the 2 redondant position sensors at Standby are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Standby are equal. "
                    + "Doesn't read again sensors.")
    public boolean isStandbySensorsInError() {
        return standbyPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Handoff are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Handoff are equal."
                    + "Doesn't read again sensors.")
    public boolean isHandoffSensorsInError() {
        return handoffPositionSensors.isInError();
    }

    /**
     * Return false if the 2 redondant position sensors at Online are equal.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Online are equal."
                    + "Doesn't read again sensors.")
    public boolean isOnlineSensorsInError() {
        return onlinePositionSensors.isInError();
    }
    
    /**
     * Return true if position sensors are in error.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
        description = "Return true if position sensors are in error."
                + "Doesn't read again sensors.")
    public boolean isPositionSensorsInError() {
        return isOnlineSensorsInError() || isHandoffSensorsInError() || isStandbySensorsInError();
    }   

    /**
     * Return truck position enregistred in field position.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return truck position.")    
    public int getPosition() {
        return position;
    }
    
    /**
     * @return controller name.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return controller name.")    
    @Override
    public String getControllerName() {
        return this.controller.getName();
    }

    /**
     * Updates the field position of the truck in reading the CPU of the
     * controller.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update truck position in reading controller.")
    public void updatePosition()  {
        try {
            this.position = controller.readPosition();
            this.publishData();

        } catch (SDORequestException ex) {
            raiseWarning(SDO_ERROR, "=> ERROR IN READING CONTROLLER:", controller.getName(), ex);
        }        
    }   

            
    /**
     * read actual current on controller, update field current and publish it on STATUS bus.
     */
    @Override
    public void updateCurrent() {
        if (controller.isBooted()) {
            try {
                this.current = controller.readCurrent();
                this.averageCurrent = controller.readCurrentAverageValue();
                this.followingError = controller.readFollowingError();
                this.averageVelocity = controller.readVelocity();
                this.publishData();

            /* we don't want to have an ALARM when monitor-current could not read current.*/
            } catch (SDORequestException ex) {
                raiseWarning(SDO_ERROR, " could not updateCurrent", controller.getName(), ex);
            }
        }
    }
    
        
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerTruck createStatusDataPublishedByAutoChangerTruck() {
        StatusDataPublishedByAutochangerTruck s = new StatusDataPublishedByAutochangerTruck();
        s.setControllerName(this.getControllerName());
        s.setHandoffInError(handoffPositionSensors.isInError());
        s.setOnlineInError(onlinePositionSensors.isInError());
        s.setStandbyInError(standbyPositionSensors.isInError());
        s.setHandoffSensorValue(handoffPositionSensors.isOn());
        s.setOnlineSensorValue(onlinePositionSensors.isOn());
        s.setStandbySensorValue(standbyPositionSensors.isOn());
        s.setPosition(position);
        s.setCurrent(current);
        s.setAverageCurrent(averageCurrent);
        s.setAverageVelocity(averageVelocity);
        s.setFollowingError(followingError);
        s.setControllerInFault(controller.isInError());
        return s;
    }

    @Override
    public void publishData() {
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, 
                createStatusDataPublishedByAutoChangerTruck()));
    }    
    
}
