
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. 
 * The ACTION methods : open and close of this class is used in ENGINEERING mode only 
 * when an operator may need to open or close only one Latch. 
 * In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. 
 * On the autochanger there are 2 latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatch extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    
    @LookupField(strategy=Strategy.TREE)
    private Autochanger autochanger;
    
    private final EPOSController latchController;
    private final ComplementarySensors filterEngagedSensors;
    private final ComplementarySensors closeSensors;
    private final ComplementarySensors openSensors;

    
    @ConfigurationParameter(description="in mA current to be sent to the controller to open the latch.")
    private int currentToOpen = 300;
    
    @ConfigurationParameter(description = "current to close Autochanger latch in mA")
    private int currentToClose = -currentToOpen;
 
    @ConfigurationParameter(description="timeout for opening a latch in milliseconds.")
    private long timeoutForOpening = 3000;
    
    @ConfigurationParameter(description="timeout for closing a latch in milliseconds.")
    private long timeoutForClosing = 3000;
    
    private int readCurrent;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    private volatile boolean initialized;

    /**
     * Buils a AutochangerLatchModule with a controller and 3 sensors.
     * @param latchController 
     * @param closeSensors 
     * @param openSensors 
     * @param filterEngagedSensors 
     */
    public AutochangerLatch(
            EPOSController latchController,
            ComplementarySensors closeSensors,
            ComplementarySensors openSensors,
            ComplementarySensors filterEngagedSensors) {
        this.latchController = latchController;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
        this.filterEngagedSensors = filterEngagedSensors;
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this latch controller name.")
    @Override
    public String getControllerName() {
        return latchController.getName();
    }

    /**
     * Return latchController
     * @return 
     */
    public EPOSController getLatchController() {
        return latchController;
    }


    /**
     * Returns true if latch is initialized. 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    /**
     * For simulation, return currentToClose
     * @return 
     */
    public int getCurrentToClose() {
        return currentToClose;
    }

    /**
     * Return timeoutForOpening
     * @return 
     */
    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    /**
     * Return timeoutForClosing
     * @return 
     */
    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    /**
     * Return lockStatus.
     * Used for simulation and tests.
     * 
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=LOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED. Doesn't read again sensors.",
            alias="isClosed")
    public boolean isClosed() {
        return this.closeSensors.isOn();
    }

    /**
     * Returns true if LockStatus=UNLOCKED
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED. Doesn't read again sensors.",
            alias="isOpen")
    public boolean isOpened() {
        return this.openSensors.isOn();
    }

    /**
     * Returns true if LockStatus=ERROR, this means that openSensor and closeSensor return 
     * non consistant values.
     * Doesn't read again sensors.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    /**
     * Returns true if autochanger is empty.
     * 
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger is empty. Doesn't read again sensors.")
    public boolean isEmpty() {
        return !this.filterEngagedSensors.isOn();
    }

    /**
     * Initialize field autochanger and latchController
     */
    @Override
    public void init() {
        if (latchController == null) {
            FCSLOG.error(name + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null latchController - fix groovy description file.");
        }
    }

    
    /**
     * Executed during INITIALIZATION phase.
     * Initialize and check latchController. 
     */    
    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (latchController.isBooted()) {
            initializeController();
        }
        FCSLOG.fine(name + " END postStart.");
    }
    
    private void initializeController() {
        try {
            latchController.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, name + " could not initialize controller", 
                    latchController.getName(), ex);
        }
        try {
            latchController.changeMode(EPOSEnumerations.EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            this.raiseWarning(FcsEnumerations.FcsAlert.HARDWARE_ERROR, name + " could not change mode of controller", 
                    latchController.getName(), ex);
        }
    }
    

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    protected void updateState() {
        boolean closed = this.closeSensors.isOn();            
        boolean opened = this.openSensors.isOn();

        boolean someSensorsInError = this.filterEngagedSensors.isInError()
                || this.closeSensors.isInError() || this.openSensors.isInError();

        boolean inError = (closed && opened) || someSensorsInError;

        if (inError) {
            lockStatus = FcsEnumerations.LockStatus.ERROR;
        } else if (closed) {
            lockStatus = FcsEnumerations.LockStatus.CLOSED;
        } else if (opened) {
            lockStatus = FcsEnumerations.LockStatus.OPENED;
        } else {
            lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
        }

        this.publishData();

    }
    

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update latch current in reading controller.")
    @Override
    public void updateCurrent()  {
        try {
            this.readCurrent = this.latchController.readCurrent();
            this.publishData();

        /* we don't want to have an ALARM when monitor-current could not read current.*/
        } catch (Exception ex) {
            raiseWarning(SDO_ERROR, " could not updateCurrent", latchController.getName(), ex);
        }
    }

    /**
     * Returns true if autochanger CANopen hardware is connected and ready.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return autochanger.isCANDevicesReady();
    }

    /**
     * Close this latch.
     * @throws FcsHardwareException 
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latch.")
    public void close()  {
        updateStateAndCheckSensors();
        if (isClosed()) {
            throw new RejectedCommandException(name + " is already CLOSED.");
        }
        autochanger.checkConditionsForClosingLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.CLOSE, timeoutForClosing);
    }

    /**
     * Open this latch.
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open latch.")
    public void open()  {
        updateStateAndCheckSensors();
        if (isOpened()) {
            throw new RejectedCommandException(name + " is already OPENED.");
        }
        autochanger.checkConditionsForOpeningLatches();
        this.executeAction(FcsEnumerations.MobileItemAction.OPEN,timeoutForOpening);
    }

    /**
     * Return true if action is completed with success.
     * @param action
     * @return 
     */
    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                return lockStatus == LockStatus.OPENED;

            case CLOSE:
                return lockStatus == LockStatus.CLOSED;
             
            default:
                assert false : action;
        }
        return false;
    }
    
    /**
     * Reads sensors, updates state and checks if sensors are in error.
     * If sensors are in error, raises an ALERT.
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * Read sensors and update State. Used during an action.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
    }

    /**
     * Start Action OPEN or CLOSE
     * @param action
     * @throws FcsHardwareException 
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action)  {
        autochanger.checkLatchMotionAllowed();
        switch (action) {
            case OPEN:
                latchController.enableAndWriteCurrent((short)this.currentToOpen);
                break;

            case CLOSE:
                latchController.enableAndWriteCurrent((short)this.currentToClose);
                break;
                
            default:
                assert false : action;
        }
    }

    /**
     * Stop action OPEN or CLOSE
     * @param action
     * @param delay
     * @throws FcsHardwareException 
     */
    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)  {
        FCSLOG.debug(name + " STOPPING action " + action.toString()
                + " within delay " + delay);
        this.latchController.stopAction();
    }

    /**
     * @throws FcsHardwareException
     * @param action
     * @param delay 
     */
    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        //TODO : is there something different to do ? TBD
        abortAction(action, delay);
    }

    
    /**
     * Creates an return an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setLockSensorValue(closeSensors.isOn());
        status.setUnlockSensorValue(openSensors.isOn());
        status.setFilterPresenceSensorValue(filterEngagedSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setControllerInFault(latchController.isInError());
        status.setLockSensorsInError(closeSensors.isInError());
        status.setUnlockSensorsInError(openSensors.isInError());
        status.setFilterEngagedSensorsInError(filterEngagedSensors.isInError());
        status.setInError(lockStatus == LockStatus.ERROR);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = createStatusDataPublishedByLatch();
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }
}
