
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.ForceSensor;
import org.lsst.ccs.subsystems.fcs.LoaderCarrier;
import org.lsst.ccs.subsystems.fcs.LoaderClamp;
import org.lsst.ccs.subsystems.fcs.RedondantSensors;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;

/**
 * This class is for the simulation of a pluto gateway for the loader.
 * @author virieux
 */
public class SimuLoaderPlutoGateway extends SimuPlutoGateway {

    @LookupField(strategy = Strategy.BYNAME)
    private LoaderCarrier carrier;
    @LookupField(strategy = Strategy.BYNAME)
    private LoaderClamp clamp;

    @LookupField(strategy = Strategy.BYNAME)
    private SimuLoaderCarrierController carrierController;
    
    @LookupField(strategy = Strategy.BYNAME)
    private SimuLoaderClampController hooksController;
    
    private DigitalSensor closeSensor1;
    private DigitalSensor openSensor1;
    private DigitalSensor closeSensor2;
    private DigitalSensor openSensor2;
    private DigitalSensor closeSensor3;
    private DigitalSensor openSensor3;
    private DigitalSensor closeSensor4;
    private DigitalSensor openSensor4;
    
    private DigitalSensor unclampedStatusSensor;
    private DigitalSensor underClampedStatusSensor;
    private DigitalSensor clampedStatusSensor;
    private DigitalSensor overClampedStatusSensor;
    private ForceSensor forceSensor0;
    private ForceSensor forceSensor1;
    
    
    
    private RedondantSensors filterPresenceSensors;
    private RedondantSensors loaderOnCameraSensors;
    private BinarySensor storagePositionSensors;
    private BinarySensor handoffPositionSensors;
    private BinarySensor engagedPositionSensors;
    
    /**
     * Build a new SimuLoaderPlutoGateway
     * @param nodeID
     * @param serialNB 
     */
    public SimuLoaderPlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        this.closeSensor1 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor1");
        this.closeSensor2 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor2");
        this.closeSensor3 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor3");
        this.closeSensor4 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor4");
        this.openSensor1 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor1");
        this.openSensor2 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor2");
        this.openSensor3 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor3");
        this.openSensor4 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor4");
        this.clampedStatusSensor = (DigitalSensor) lookup.getComponentByName("clampedStatusSensor");
        this.unclampedStatusSensor = (DigitalSensor) lookup.getComponentByName("unclampedStatusSensor");
        this.underClampedStatusSensor = (DigitalSensor) lookup.getComponentByName("underClampedStatusSensor");
        this.overClampedStatusSensor = (DigitalSensor) lookup.getComponentByName("overClampedStatusSensor");
        this.forceSensor0 = (ForceSensor) lookup.getComponentByName("forceSensor0");
        this.forceSensor1 = (ForceSensor) lookup.getComponentByName("forceSensor1");
                
        this.loaderOnCameraSensors = (RedondantSensors) lookup.getComponentByName("loaderOnCameraSensors");
        this.filterPresenceSensors = (RedondantSensors) lookup.getComponentByName("loaderFilterPresenceSensors");
        this.storagePositionSensors = (RedondantSensors) lookup.getComponentByName("loaderStorageSensors");
        this.handoffPositionSensors = (BinarySensor) lookup.getComponentByName("loaderHandoffSensors");
        this.engagedPositionSensors = (BinarySensor) lookup.getComponentByName("loaderEngagedSensors");
       
        //Filter is closed && clamped and carrier is at storage position
        simulateLoaderClampIsClosed();
        simulateLoaderIsConnectedOnCamera();
        simulateClampedStatusSensorIsOn();
        simulateFilterIsOnLoader();
        simulateLoaderCarrierIsAtStorage();
    }
    
    protected void simuBinarySensorOn(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceSensorValue(((RedondantSensors) sensor).getSensor(), true);
            replaceSensorValue(((RedondantSensors) sensor).getSensorR(), true);
        } else {
            replaceSensorValue((DigitalSensor) sensor, true);
        }
    }
    
    protected void simuBinarySensorOff(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceSensorValue(((RedondantSensors) sensor).getSensor(), false);
            replaceSensorValue(((RedondantSensors) sensor).getSensorR(), false);
        } else {
            replaceSensorValue((DigitalSensor) sensor, false);
        }
    }
    
    protected void simuRedondantSensorsInError(RedondantSensors sensors) {
        replaceSensorValue(sensors.getSensor(), true);
        replaceSensorValue(sensors.getSensorR(), false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsClosed() {
        simulateHooksAreClosed();
        simulateUnclampedStatusSensorIsOn();
        hooksController.setPosition(clamp.getRelativePositionToClose());
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsClamped() {
        simulateHooksAreClosed();
        simulateClampedStatusSensorIsOn();
        hooksController.setPosition(clamp.getTargetPositionToClamp());
    }    
    
    private void simulateHooksAreClosed() {
        simulateHookIsClosed(closeSensor1, openSensor1);
        simulateHookIsClosed(closeSensor2, openSensor2);
        simulateHookIsClosed(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsOpened() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsOpened(closeSensor4, openSensor4);
        simulateUnclampedStatusSensorIsOn();
        hooksController.setPosition(clamp.getTargetPositionToOpen());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsInTravel() {
        simulateHookIsInTravel(closeSensor1, openSensor1);
        simulateHookIsInTravel(closeSensor2, openSensor2);
        simulateHookIsInTravel(closeSensor3, openSensor3);
        simulateHookIsInTravel(closeSensor4, openSensor4);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsUnknown() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }    
 
    /**
     * simulate that clamp is in error : if one hook is in error, clamp is in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsInError() {
        simulateHookIsInError(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsClosed(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, true);
        replaceSensorValue(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsOpened(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, false);
        replaceSensorValue(openSensor, true);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsInError(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, true);
        replaceSensorValue(openSensor, true);
    }    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsInTravel(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, false);
        replaceSensorValue(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnectedOnCamera() {
        simuBinarySensorOn(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsNotConnectedOnCamera() {
        simuBinarySensorOff(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnLoader() {
        simuBinarySensorOn(filterPresenceSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsNotOnLoader() {
        simuBinarySensorOff(filterPresenceSensors);
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtStorage() {
        simuBinarySensorOn(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOff(engagedPositionSensors);
        carrierController.setPosition(carrier.getStoragePosition());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtHandoff() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOn(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        carrierController.setPosition(carrier.getHandoffPosition());
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtEngaged() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        carrierController.setPosition(carrier.getEngagedPosition());
    }    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsInTravel() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHandoffSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) handoffPositionSensors);
    }    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateStorageSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) storagePositionSensors);
    }    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampedStatusSensorIsOn() {
        replaceSensorValue(clampedStatusSensor, true);
        replaceSensorValue(unclampedStatusSensor, false);
        replaceSensorValue(underClampedStatusSensor,false);
        replaceSensorValue(overClampedStatusSensor,false);        
        simulateForceSensorsOn();
    }
    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateUnclampedStatusSensorIsOn() {
        replaceSensorValue(unclampedStatusSensor, true);
        replaceSensorValue(clampedStatusSensor, false);
        replaceSensorValue(underClampedStatusSensor,false);
        replaceSensorValue(overClampedStatusSensor,false);
    }    
    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)   
    public void simulateForceSensorsOn() {
        simulateForceSensorClamped(forceSensor0);
        simulateForceSensorClamped(forceSensor1);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)   
    public void simulateForceSensorsOff() {
        simulateForceSensorUnclamped(forceSensor0);
        simulateForceSensorUnclamped(forceSensor1);
    }    
    
    public void simulateForceSensorClamped(ForceSensor forceSensor) {
        this.updateForceSensor(forceSensor, 30);
    }
    
    public void simulateForceSensorUnclamped(ForceSensor forceSensor) {
        this.updateForceSensor(forceSensor, 10);
    }
    
    private void updateForceSensor(ForceSensor forceSensor, int voltage) {
        this.readValues[forceSensor.getByteNumero()] = voltage;
    }
    
    
    
}
