
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.framework.HasLifecycle;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;

/**
 * This class is a model for the fakeAutochanger that we use on loader and carousel test bench 
 * in standalone mode.
 * 
 * As there is no autochanger, a numeric sensor is setup to simulate that the autochanger holds the filter
 at STANDBY or HANDOFF position.
 It'subs also a model for the fakeCarousel and the fakeLoader in autochanger-standalone subsystem.
 
 FakeFilterHolder implements interface FilterHolder.
 In the whole FCS, carousel, autochanger and loader implements FilterHolder too.
 * 
 * @author virieux
 */
public class FakeFilterHolder implements FilterHolder, HasLifecycle {
    
    @LookupField(strategy=Strategy.TOP)
    private Subsystem subs;
    
    private BinarySensor holdingFilterSensor;
    private BinarySensor notHoldingFilterSensor;
    
    @LookupField(strategy=Strategy.CHILDREN)
    private String holdingFilterSensorName;
    
    @LookupField(strategy=Strategy.CHILDREN)
    private String notHoldingFilterSensorName;    
    
    @LookupName
    private String name;

    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the fake autochanger is holding the filter. "
                    + "This command doesn't read again the sensor.")
    @Override
    public boolean isHoldingFilter() {
        return this.holdingFilterSensor.isOn();
    }
    
    /**
     * Return true if the fake filter holder is holding the filter.
     * This command doesn't read again the sensor.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the fake autochanger is holding the filter. "
                    + "This command doesn't read again the sensor.")
    @Override
    public boolean isNotHoldingFilter() {
        return this.notHoldingFilterSensor.isOn();
    }   
    
    /**
     * This methods updates the field holdingFilter in reading the holding filter sensor.
     * Then it published status data on the STATUS bus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, 
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors()  {
        this.publishData();
    }
    
    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        KeyValueData kvd;
            kvd = new KeyValueData(name, holdingFilterSensor.isOn());
        subs.publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at HANDOFF.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtHandoff() {
        return true;
    }

    /**
     * For the fake autochanger there is no autochanger trucks, so the position has not a real meaning.
     * The autochanger is supposed to be at STANDBY.
     * Idem for fakeCarousel and fakeLoader.
     * @return 
     */
    @Override
    public boolean isAtStandby() {
        return true;
    }
    
    @Override
    public void start() {
        FCSLOG.info(name + " FakeFilterHolder BEGIN start");
        ComponentLookup lookup = subs.getComponentLookup();
        this.holdingFilterSensor = (BinarySensor) lookup.getComponentByName(holdingFilterSensorName);
        this.notHoldingFilterSensor = (BinarySensor) lookup.getComponentByName(notHoldingFilterSensorName);
        FCSLOG.info(name + " FakeFilterHolder END start");
    }
    
    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart");
        FCSLOG.info(name + " END postStart");
    }
    
    

}
