
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenDevice;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * This represents a device we use in FCS for to connect our numeric sensors(CAN-CBX-DIO8). 
 * It has 8 channels. Each channel can be configured to be inputs or output channel. 
 * So we can plug up to 8 sensors on one CompactIOModule. 
 * We read all the inputs channels in the same commands. 
 * At the initialization, it has to be configured to say which are the inputs and which are the ouputs.
 * The outputs for the standby or standback motors of the autochanger are on this device. 
 * The sensors to know if a latch on the autochanger is closed or open are also connected to a CAN-CBX-DIO8.
 *
 * @author virieux
 */
public class CompactIO extends CanOpenDevice {


    public int value = 0;

   //updatingValue is true while we are reading the value sent by the Sensor
    //this could take some time
    private volatile boolean updatingValue = false;
    private final Lock lock = new ReentrantLock();
    private final Condition valueUpdated = lock.newCondition();

    /**
     * Build a new CompactIOModule
     * @param nodeID
     * @param serialNB 
     */
    public CompactIO(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    /**
     * returns the value of all the sensors connected on the inputs
     * channels of the device. cf CAN-CBX-DIO8 examples : if returns "80" means
     * that the input 8 is set to 1, the others to 0 if returns "A0" means that
     * the inputs 7 and 5 are set to 1, the others to 0 if returns "FF" all the
     * inputs are set to 1 if returns "00" all the inputs are set to 0
     *
     * @return value read on the device
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public int readNewValue()  {
        try {
            value = (int) tcpProxy.readSDO(this.nodeID, 0x6000, 1);
            
        } catch (SDORequestException ex) {
            String message = name + "=> ERROR IN READING SENSOR:";
            this.raiseWarning(SDO_ERROR, message, name, ex);

        } 
        return value;
    }

    /**
     * @return the value.
     */
    public int getValue() {
        lock.lock();
        try {
            while (updatingValue) {
                try {
                    //TODO put a timeout
                    valueUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ex.getMessage());

                }
            }
            return this.value;
        } finally {
            lock.unlock();
        }
    }


    /**
     * 
     * @throws FcsHardwareException 
     */
    public void updateValue()  {
        lock.lock();

        try {
            updatingValue = true;
            this.value = readNewValue();
            FCSLOG.debug(name + " VALUE READ=" + this.value);

        } finally {

            updatingValue = false;
            valueUpdated.signal();
            lock.unlock();

        }
    }
    
    /**
     * Write a value to the output of the DAC
     *
     * @param outputNB in 1..8
     * @param value 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     */
    public void writeDigitalOutput(int outputNB, int value)  {

        if (!this.tcpProxy.allDevicesBooted()) {
            String message = String.format("DAC %s / nodeID=%s is NOT BOOTED", getName(), this.nodeID);
            FCSLOG.error(message);
            throw new HardwareNotDetectedException("a CanOpenDIO is missing", getName(), this.nodeID, this.serialNB);
        }

        if ((outputNB < 1) || (outputNB > 8)) {
            throw new IllegalArgumentException("CAN-CBX-DIO8 : input has to be 1..8");
        }
        tcpProxy.writeSDO(nodeID, 0x6200, outputNB, 2, value);
    }
    
    /**
     * This methods initialize the CAN-CBX-DIO8 with the mask (value 3) to
     * configure the 2 first channels of the device to be output channel. (3 =
     * 00000011)
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void writeMask()  {
        tcpProxy.writeSDO(nodeID, 0x2250, 1, 1, 0x3);
    }

    /**
     * This methods checks if the mask of the device is the good one. The good
     * one is when the first 2 channels are output channels, that means that the
     * value of the mask is 3 (3= 00000011)
     *
     * @return true if the mask is OK, false otherwise
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public boolean checkMask()  {
        return tcpProxy.readSDO(nodeID, 0x6000, 1) == 3;
    }

}
