
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.CarouselClamp;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenTTC580;

/**
 *
 * @author virieux
 */
public class SimuTTC580 extends CanOpenTTC580 {
    
    CarouselSocket socket1;
    CarouselSocket socket2;
    CarouselSocket socket3;
    CarouselSocket socket4;
    CarouselSocket socket5;
    
    protected final Map<String, CarouselSocket> socketMapByID = new HashMap<>();
    
    public SimuTTC580(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }
    
    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        
        socket1 = (CarouselSocket) lookup.getComponentByName("socket1");
        socket2 = (CarouselSocket) lookup.getComponentByName("socket2");
        socket3 = (CarouselSocket) lookup.getComponentByName("socket3");
        socket4 = (CarouselSocket) lookup.getComponentByName("socket4");
        socket5 = (CarouselSocket) lookup.getComponentByName("socket5");
        socketMapByID.put("1", socket1);
        socketMapByID.put("2", socket2);
        socketMapByID.put("3", socket3);
        socketMapByID.put("4", socket4);
        socketMapByID.put("5", socket5);
    }
    
    @Override
    public void initializeAndCheckHardware() {
        super.initializeAndCheckHardware();
        updateFakePDOData(cobid1,0);
        updateFakePDOData(cobid2,0);
    }    
    
    public void updateFakePDOData(int cobid, long newPdo) {
        if (cobid == cobid1) {
            this.pdo1 = newPdo;
        } else if (cobid == cobid2) {
            this.pdo2 = newPdo;
        }
        ((SimuCanOpenInterface) tcpProxy.getCanInterface()).simulatePDOData(cobid, newPdo);
    }
    
    private void checkID(short id) {
        if (id < 1 || id > 5) {
            throw new IllegalArgumentException(id + " not a valid socket number");
        }
    }
    
    private void checkSensorValue(int value) {
        if (value > 0xFFF) {
            throw new IllegalArgumentException(value + " value can't be > " + 0xFFF);
        }
    }
    
    /**
     * socket ID is coded on the first 3 bits.
     * @param id 
     */
    @Command
    public void simulateSocketAtStandby(short id) {
        checkID(id);
        long sock = (long) id << 61;
        long mask = 0x1FFFFFFFFFFFFFFFL;
        long newpdo = sock | (pdo1 & mask);
        System.out.println("newpdo1=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    }
    
    @Command
    public void simulateSocketNotAtStandby(short id) {
        checkID(id);
        long sock = (long) id << 61;
        long mask = 0x1FFFFFFFFFFFFFFFL;
        long newpdo = sock | (pdo2 & mask);
        System.out.println("newpdo2=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid2, newpdo);
    }  
    
    @Command    
    public void simuSocketAtSLocked(short id) {
        simulateSocketAtStandby(id);
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampIsLocked(socket.getClampXminus());
        simulateClampIsLocked(socket.getClampXplus());
    }
    
    @Command
    public void simulateLockSensorXminusValue(long lockValue) {
        System.out.println("lockValue=" + Long.toBinaryString(lockValue));
        long pdo = (long) (lockValue << 36);
        long mask = 0xFFFF000FFFFFFFFFL;
        long newpdo = pdo | (pdo1 & mask);
        System.out.println("newpdo1=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    }
    
    @Command
    public void simulateLockSensorXminusValueNotAtS(long lockValue) {
        System.out.println("lockValue=" + Long.toBinaryString(lockValue));
        long pdo = (long) (lockValue << 36);
        long mask = 0xFFFF000FFFFFFFFFL;
        long newpdo = pdo | (pdo2 & mask);
        System.out.println("newpdo2=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid2, newpdo);
    }    
    
    @Command
    public void simulateFilterPresenceXminusValue(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue << 24);
        long mask = 0xFFFFFFF000FFFFFFL;
        long newpdo = pdo | (pdo1 & mask);
        System.out.println("newpdo1=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    }
    
    @Command
    public void simulateFilterPresenceXminusValueNotAtS(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue << 24);
        long mask = 0xFFFFFFF000FFFFFFL;
        long newpdo = pdo | (pdo2 & mask);
        System.out.println("newpdo2=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid2, newpdo);
    }    
    
    @Command
    public void simulateLockSensorXplusValue(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue << 12);
        long mask = 0xFFFFFFFFFF000FFFL;
        long newpdo = pdo | (pdo1 & mask);
        System.out.println("newpdo1=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    } 
   
   @Command
   public void simulateLockSensorXplusValueNotAtS(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue << 12);
        long mask = 0xFFFFFFFFFF000FFFL;
        long newpdo = pdo | (pdo2 & mask);
        System.out.println("newpdo2=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid2, newpdo);
    }    
    
    @Command
    public void simulateFilterPresenceXplusValue(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue & 0xFFF);
        long mask = 0xFFFFFFFFFFFFF000L;
        long newpdo = pdo | (pdo1 & mask);
        System.out.println("newpdo1=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid1, newpdo);
    } 
    
    @Command
    public void simulateFilterPresenceXplusValueNotAtS(long newValue) {
        System.out.println("lockValue=" + Long.toBinaryString(newValue));
        long pdo = (long) (newValue & 0xFFF);
        long mask = 0xFFFFFFFFFFFFF000L;
        long newpdo = pdo | (pdo2 & mask);
        System.out.println("newpdo2=" + Long.toBinaryString(newpdo));
        updateFakePDOData(cobid2, newpdo);
    }     
    
    public void simulateClampIsLocked(CarouselClamp clamp) {
        int lockVal = (clamp.getLockSensorMaxValue()) / 4;
        simulateClampLockValue(clamp, lockVal);
    }
    
    public void simulateClampIsLockedNotAtS(CarouselClamp clamp) {
        int lockVal = (clamp.getLockSensorValueB() + clamp.getLockSensorMaxValue()) / 8;
        simulateClampLockValueNotAtS(clamp, lockVal);
    }    
    
    public void simulateClampIsUnlocked(CarouselClamp clamp) {
        int lockVal = (clamp.getLockSensorValueA() / 4) - 10;
        simulateClampLockValue(clamp, lockVal);
    }
    
    private void simulateClampLockValue(CarouselClamp clamp, int lockVal) {
        checkSensorValue(lockVal);
        if (clamp.getName().contains("Xminus")) {
            simulateLockSensorXminusValue(lockVal);
        } else if (clamp.getName().contains("Xplus")) {
            simulateLockSensorXplusValue(lockVal);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");
    }    
    
    private void simulateClampLockValueNotAtS(CarouselClamp clamp, int lockVal) {
        checkSensorValue(lockVal);
        if (clamp.getName().contains("Xminus")) {
            simulateLockSensorXminusValueNotAtS(lockVal);
        } else if (clamp.getName().contains("Xplus")) {
            simulateLockSensorXplusValueNotAtS(lockVal);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");
    }     
    
    /**
     * filter is engaged and lockable if filterPresence sensor returns a value between 
     * FilterPresenceValueA and FilterPresenceValueB
     * @param clamp 
     */
    public void simulateClampFilterIsLockable(CarouselClamp clamp) {
        int val = (clamp.getFilterPresenceMinValue() + clamp.getFilterPresenceValueA())/8;
        simulateClampFilterPresenceValue(clamp, val);
    }
    
    /**
     * filter is engaged and lockable if filterPresence sensor returns a value between 
     * FilterPresenceValueA and FilterPresenceValueB
     * @param clamp 
     */
    public void simulateClampFilterIsLockableNotAtS(CarouselClamp clamp) {
        int val = (clamp.getFilterPresenceMinValue() + clamp.getFilterPresenceValueA())/8;
        simulateClampFilterPresenceValueNotAtS(clamp, val);
    }    
    
    /**
     * filter is not loackable if filterPresence sensor returns a value between
     * FilterPresenceValueB and FilterPresenceValueC
     * @param clamp 
     */
    public void simulateClampFilterIsNotLockable(CarouselClamp clamp) {
        int val = (clamp.getFilterPresenceValueA() + clamp.getFilterPresenceValueB())/8;
        simulateClampFilterPresenceValue(clamp, val);
    }
    
    /**
     * simulate there is no filter in socket socketID clampX-
     * @param clamp
     */
    public void simulateClampNoFilter(CarouselClamp clamp) {
        int val = clamp.getFilterPresenceValueB()/4 + 1;
        simulateClampFilterPresenceValue(clamp, val);
    }
    
    
    public void simulateClampFilterIsInError(CarouselClamp clamp) {
        int val = clamp.getFilterPresenceMinValue()/4 - 1;
        simulateClampFilterPresenceValue(clamp, val);        
    }    
    
    
    private void simulateClampFilterPresenceValue(CarouselClamp clamp, int val) {
        checkSensorValue(val);
        if (clamp.getName().contains("Xminus")) {
            simulateFilterPresenceXminusValue(val);
        } else if (clamp.getName().contains("Xplus")) {
            simulateFilterPresenceXplusValue(val);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");        
    }
    
    private void simulateClampFilterPresenceValueNotAtS(CarouselClamp clamp, int val) {
        checkSensorValue(val);
        if (clamp.getName().contains("Xminus")) {
            simulateFilterPresenceXminusValueNotAtS(val);
        } else if (clamp.getName().contains("Xplus")) {
            simulateFilterPresenceXplusValueNotAtS(val);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");        
    }    

    
    
    

    
    
    
    
    
    

    

    
    
    

}
