
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.SDO_ERROR;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A class to represent the CANopen pluto gateway which is a device where we can read sensors.
 * Its raises an alert :
 - FCS004 in readNewValues
 * 
 * @author virieux
 */
public class CanOpenPlutoGateway extends PlutoGateway {
    
   /**
    * A Logger for CommandDispenser.
    */
    public static final Logger LOCAL_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.CanOpenPlutoGateway");
    


    /**
     * build a new CanOpenPlutoGateway
     * @param nodeID
     * @param serialNB
     */
    public CanOpenPlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }



    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Initialize the Gateway.")
    public void initializeAndCheckHardware()  {
        checkBooted();
        try {
            FCSLOG.info(name + " BEGIN initializeHardware");
            this.configurePlutoGateway();
            this.initialized = true;
            FCSLOG.info(name + " is INITIALIZED.");
            //publishData for the LoaderHardwareGUI
            this.publishData();

        } catch (SDORequestException | FailedCommandException ex) {
            throw new FcsHardwareException(name,ex);
        }
    }

    /**
     * Configures the gateway.
     * Configuration depends on specific needs for each subsystems.
     * This methods switches to the right configuration in reading the name of the gateway.
     * 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Configures this gateway for subsystem needs. "
                + "If gateway's name contains loader it executes configureLoaderPlutoGateway"
                + "else it executes configureAutochangerPlutoGateway.")
    public void configurePlutoGateway() {
        if (name.contains("loader")) {
            configureLoaderPlutoGateway();
        } else if (getName().startsWith("acSensors")){
            configureAutochangerPlutoGateway();
        }
    }

    /**
     * Configures this gateway to receive 3 data blocs from Pluto.
     * Used for autochanger pluto gateway.
     * Assuming Pluto has 3 fonctions Gateway_to_User_C First one =>
     * Fonction_ID=01 Second one => Fonction_ID=02 Third one => Fonction_ID=03
     *
     * @throws SDORequestException
     */
    private void configureAutochangerPlutoGateway()  {
        
        FCSLOG.debug(name+ " configureAutochangerGateway");

        //Clear config 
        this.tcpProxy.writeSDO(getNodeID(), 0x2010, 1, 0x1, 0x0);

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 1, 0x2, 0x17);

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 2, 0x2, 0x4);

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 0x1, 0x2, 0x117);
        
        //Set Additionnal data TPDO2 => define Aera_3  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 0x2, 0x02, 0x143);
    }
    
    /**
     * Configures plutoGateway for loader needs.
     */
    private void configureLoaderPlutoGateway() {
        FCSLOG.debug(name+ " configureLoaderPlutoGateway");
        
        //Clear additional data configuration.
        this.tcpProxy.writeSDO(getNodeID(), 0x2010, 1, 1, 0x1);

        //Set Additionnal data TPDO1 => define Area_0 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 1, 2, 0x1);

        //Set Additionnal data TPDO1 => define Area_1 (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2011, 2, 2, 0x2);

        //Set Additionnal data TPDO2 => define Aera_2  (doc p:96)
        this.tcpProxy.writeSDO(getNodeID(), 0x2012, 1, 2, 0x3);
        
        //Activate additional data for Area_0, Area_1 and Area_2 (Enable Data To Pluto Areas 0-3)
        this.tcpProxy.writeSDO(getNodeID(), 0x2002, 1, 1, 0x7);
    }

    /**
     * Write a transmission type in the CPU of the pluto gateway.
     * @param transmissionType
     * @throws FcsHardwareException
     * @throws FailedCommandException
     */
    public void writeTransmissionType(int transmissionType)  {
        this.tcpProxy.writeSDO(getNodeID(), 0x2005, 1, 1, transmissionType);
    }

    /**
     * return an array of values read on the device.
     * @return readValues
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return values read on the device.")
    @Override
    public int[] readNewValues()  {

        checkBooted();
        try {            
            /* Values returned by Object 0x6000..0x6003 subindex 1..4 are UINT8. Could be a represented by a byte. */
            /*Index 6000*/
            readValues[0] = (int)tcpProxy.readSDO(getNodeID(), 0x6000, 1);
            readValues[1] = (int)tcpProxy.readSDO(getNodeID(), 0x6000, 2);
            readValues[2] = (int)tcpProxy.readSDO(getNodeID(), 0x6000, 3);
            readValues[3] = (int)tcpProxy.readSDO(getNodeID(), 0x6000, 4);
            
            /*Index 6001*/
            readValues[4] = (int)tcpProxy.readSDO(getNodeID(), 0x6001, 1);
            readValues[5] = (int)tcpProxy.readSDO(getNodeID(), 0x6001, 2);
            readValues[6] = (int)tcpProxy.readSDO(getNodeID(), 0x6001, 3);
            readValues[7] = (int)tcpProxy.readSDO(getNodeID(), 0x6001, 4);
            
            /*Index 6002*/
            readValues[8] = (int)tcpProxy.readSDO(getNodeID(), 0x6002, 1);
            readValues[9] = (int)tcpProxy.readSDO(getNodeID(), 0x6002, 2);
            readValues[10] = (int)tcpProxy.readSDO(getNodeID(), 0x6002, 3);
            readValues[11] = (int)tcpProxy.readSDO(getNodeID(), 0x6002, 4);
            
            /*Index 6003*/
            readValues[12] = (int)tcpProxy.readSDO(getNodeID(), 0x6003, 1);
            readValues[13] = (int)tcpProxy.readSDO(getNodeID(), 0x6003, 2);
            readValues[14] = (int)tcpProxy.readSDO(getNodeID(), 0x6003, 3);
            readValues[15] = (int)tcpProxy.readSDO(getNodeID(), 0x6003, 4);
            
        } catch (SDORequestException ex) {
            String message = name + "=> ERROR IN READING SENSOR:";
            this.raiseWarning(SDO_ERROR, message, name, ex);
        }
        return readValues;
    }
    
    /**
     * Read new readValues for sensors connected to this gateway from a PDO request. 
     * @return an Array of 16 bytes
     */
    @Override
    public int[] readValuesFromPDO() {
        int[] values = new int[16];
        PDOData pdos = tcpProxy.updatePDOData();
        for (int i = 0; i < 16; i++) {
            //TODO find what to be done here 
            values[i] = (int)pdos.getPDOs().get(this.nodeID).intValue();
        }
        return values;
    }


}
