package org.lsst.ccs.subsystems.fcs;

import java.util.List;
import org.lsst.ccs.PersistencyService;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the
 CCS Master Control Module (MCM) in normal operation mode or from a Console
 (org.lsst.ccs.subsystem.console.TestConsole).

 <P/>
 Example of commands that can be sent to the FcsMain :
 <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>
 The main goals of the FcsMain is :
 <UL>
 * <LI>Receive commands from MCM for the whole subsystem.</LI>
 * <LI>Execute commands received from MCM in invoking methods on the Carousel
 * Module on the Autochanger Module and on Loader Module.
 * </LI>
 * <LI>Handle the logic of the whole subsystem.</LI>
 * <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 *
 * @author virieux
 *
 */
public class FcsMain extends MainModule implements FilterLocator {

    /**
     *
     */
    private static final long serialVersionUID = 7669526660659959402L;
    
    /**
     * this bridge is to communicate with the canbus1 on the PC104
     */
    private final BridgeToHardware bridgeToLoader;

    @LookupField(strategy=Strategy.CHILDREN)
    private Carousel carousel;
    
    @LookupField(strategy=Strategy.CHILDREN)
    private Autochanger autochanger;
    
    @LookupField(strategy=Strategy.CHILDREN)
    private Loader loader;

    @LookupField(strategy=Strategy.CHILDREN)
    private FilterManager filterManager;
    
    @LookupField(strategy=Strategy.TREE)
    private PersistencyService persistenceService; 
    
    private final boolean loadFilterLocationAtStartup = true;
    private final boolean loadFilterLocationAtShutdown = true;

    /**
     * Build a FcsMainModule
     * @param bridge
     * @param bridgeToLoader
     */
    public FcsMain(BridgeToHardware bridge,
            BridgeToHardware bridgeToLoader) {
        super(bridge);
        this.bridgeToLoader = bridgeToLoader;
    }
    
    @Override
    public Carousel getCarousel() {
        return carousel;
    }

    @Override
    public FilterManager getFilterManager() {
        return filterManager;
    }

    @Override
    public PersistencyService getPersistenceService() {
        return persistenceService;
    }
    
    @Override
    public void init() {
        /* define a role for my subsystem in order to make FcsGUI listen to my subsystem*/
        subs.setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs", "fcs");
        persistenceService.setAutomatic(loadFilterLocationAtStartup, loadFilterLocationAtShutdown);
    }       

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()){
            /* The initialization has been done, so now the hardware is ready */
            subs.updateAgentState(FcsEnumerations.FilterState.READY, 
                FcsEnumerations.FilterReadinessState.READY);
        }
    }
    
    /**
     * Return the name of the filter which is at ONLINE.
     * Return null if no filter is at ONLINE.
     * @return 
     */    
    public String getOnlineFilterName() {
        if (autochanger.isHoldingFilter()) {
            return autochanger.getFilterOnTrucksName();
             
        } else {
            return null;
        }
    }
    
    /**
     * For end user.
     * Return the name of the filter which is at ONLINE.
     * Return null if no filter is at ONLINE.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns the name of the filter which is at ONLINE.") 
    public String printFilterONLINEName() {
        if (getOnlineFilterName() != null) {
            return getOnlineFilterName();
        } else {
            return "NONE";
        }            
    }



    /**
     * This methods returns the filter which has for name the String given as argument.
     * @param filterName
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "returns the filter which has for name the String given as argument.")
    public Filter getFilterByName(String filterName) {
        return filterManager.getFilterByName(filterName);
    }
    
    /**
     * Return true if the changer is connected.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the changer is connected.")    
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }
    
    /**
     * Return true if the loader is connected.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return this.bridgeToLoader.isReady();
    }

    /**
     * Return true if the hardware of the changer is ready.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the hardware of the changer is ready.")     
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Return true if the hardware of the loader is ready.")    
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
        description = "Disconnect the loader hardware.")    
    public void disconnectLoader()  {
        this.bridgeToLoader.disconnectHardware();
    }
    
    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
            description = "Connect the loader hardware.")
    public void connectLoader() {
        this.bridgeToLoader.connectHardware();
    }

    /**
     * Check if a filter name given by an end user at the console is a valid Filter Name.
     * Delegate to filterManager.
     * @param aName 
     */
    public void checkFilterName(String aName) {
        filterManager.checkFilterName(aName);
    }
    
    /**
     * Check where are the autochanger trucks.
     * If a filter is at ONLINE position update state and publishes.
     */
    public void checkAutochangerTrucksLocation() {
        autochanger.getAutochangerTrucks().updatePosition();
        if (autochanger.getAutochangerTrucks().isAtOnline() 
                && autochanger.isHoldingFilter() && autochanger.getOnlineClamps().isLocked()) {
            String filterName = filterManager.getFilterOnAutochangerName();
            subs.updateAgentState(FilterState.valueOf("ONLINE_"
                +filterName.substring(6,filterName.length())), FilterReadinessState.READY);
                
        } else if (autochanger.isHoldingFilter() && autochanger.getOnlineClamps().isLocked()) {
            subs.updateAgentState(FilterState.valueOf("ONLINE_NONE"), 
                    FilterReadinessState.READY);
                
        } else {
            subs.updateAgentState(FilterState.valueOf("ONLINE_NONE"), 
                    FilterReadinessState.NOT_READY);
        }
    }
    
    /**
     * For Whole FCS GUI. 
     * Has to be overriden in FcsMain.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }    


    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append(carousel.toString());
        sb.append(autochanger.toString());
        return sb.toString();
    }

    /**
     * Move a filter to ONLINE position. 
     * The filter must be in the camera. It can be on the carousel or on the autochanger. 
     * It envolves the carousel Module and the autochanger Module. 
     * If the filter we want to move is on the autochanger, the autochanger has to move it to ONLINE position. 
     * If the filter is on the carousel, and the autochanger truck is loaded with another filter, the autochanger 
     * has to move back the other filter on the carousel at standby position. 
     * If the filter is on the carousel and the autochanger is empty, we have to rotate the carousel in order to put 
     * the required filter at standby position, then the autochanger grabbes the filter on the carousel and moves it to 
     * ONLINE position.
     *
     * @param letter name of the filter to move
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
        description = "Move filter to ONLINE position.", alias="setFilterONLINE", timeout=60)
    public void setFilter(String letter)  {
        String filterName = "filter" + letter;
        FCSLOG.info(name + ": filter to move is " + filterName);
        
        /*The filter we want to move is already at ONLINE.*/ 
        if (filterName.equals(this.getOnlineFilterName())) { 
            FCSLOG.info(filterName +" is already at ONLINE.");
            return;
        }
        
        Filter filterToSet = getFilterByName(filterName);
        if (filterToSet == null) {
            throw new IllegalArgumentException(name + ": Unknown filter name : " + filterName);
            
        /*The filter we want to move is out of Camera.*/    
        } else if (filterToSet.isUnknown() || filterToSet.isOut()) {
            throw new RejectedCommandException(name + ": can't setFilter "+filterName+" ONLINE because filter location is "
                    +filterToSet.getFilterLocation());

        /*The filter we want to move is on carousel.*/    
        } else if (filterToSet.isOnCarousel()) {
            FCSLOG.info(name + ":" + filterName + " is on carousel.");
            carouselMoveFilterToStandby(filterToSet);
        }    
        /*The filter we want to move is on autochanger or on carousel at STANDBY position.*/
        FCSLOG.info(name + ":" + filterName + " is on autochanger.");

        autochangerMoveFilterToOnline(filterToSet);
        this.saveFilterLocation();
        subs.updateAgentState(FilterState.valueOf("ONLINE_"+letter), 
                FilterReadinessState.READY);
        FCSLOG.info(filterName + " is ONLINE.");
        
    }
    
    /**
     * just for mcm testing
     * @param state 
     */
     @Command(type = Command.CommandType.ACTION, level = Command.NORMAL,
        description = "change subsystem state.")   
    public void changeState(String state) {
        subs.updateAgentState(FilterState.valueOf(state), FilterReadinessState.READY);
        FCSLOG.fine("SUBSYSTEM STATE="+subs.isInState(FilterState.valueOf(state)));
    }
    
    /**
     * Move a filter to standby position.
     * This private method is called by method setFilter when filter is on Carousel.
     * 
     * Filter can be already at STANDBY. In this case, no need to rotate carousel.
     * If another filter is on autochanger, it must be stored on carousel before rotation of carousel.
     * 
     * @param filter 
     */
    private void carouselMoveFilterToStandby(Filter filter) {
        assert filter.isOnCarousel() : "carouselMoveFilterToStandby has to be called when filter is on Carousel";
        
        if (carousel.isAtStandby(filter)) {
        /* filter is already at STANDBY - nothing more to do*/
            FCSLOG.info(name + " filter:"+filter.getName()+ " is at STANDBY position. No need to rotate carousel.");
            
        } else {
        /*  filter is on Carousel but not at STANDBY - carousel must be rotated
            but if another filter is on autochanger, it must be stored on carousel
            before rotation of carousel.
        */  
            FCSLOG.info(name + ":"+filter.getName()+ " is on carousel but not at STANDBY position. Need to rotate carousel.");
            if (!autochanger.isEmpty() && (autochanger.getFilterOnTrucksID() != filter.getFilterID())) {
                
                FCSLOG.info(autochanger.getFilterOnTrucksName()
                        +" is on autochanger. It has to be stored on carousel.");
            /*autochanger is loaded with another filter than filter to be set to STANDBY*/
            /*other filter on autochanger has to be stored on carousel in an empty socket before rotation. */
                if (!autochanger.getAutochangerTrucks().isAtOnline() 
                        && !autochanger.getAutochangerTrucks().isAtHandoff()) {
                    /*autochanger have to move to HANDOFF to let carousel rotate safely.*/
                    autochanger.getAutochangerTrucks().goToHandOff();
                }
                carousel.rotateEmptySocketAtStandby();
                autochangerStoreFilterOnCarousel();
            }
            /*put autochanger trucks in a safe place in order to rotate Carousel*/
            autochanger.getAutochangerTrucks().goToHandOff();
            FCSLOG.info(name + ": can be rotated to STANDBY to set "+filter.getName());
            carousel.rotateSocketToStandby(filter.getSocketName());
        } 
    }
    
    /**
     * Move a filter to ONLINE position and closes ONLINE clamps.
     * This private method is called when filter is on autochanger or on carousel at STANDBY.
     * @param filterName 
     */
    private void autochangerMoveFilterToOnline(Filter filter) {
        if (!autochanger.isEmpty() && autochanger.getFilterOnTrucksID() != filter.getFilterID()) {
            String msg = name+": autochanger is loaded with another filter than "+ filter.getName();
            FCSLOG.error(msg);
            throw new FailedCommandException(msg);
            
        } else if (autochanger.isEmpty() && carousel.isAtStandby(filter)) {
            FCSLOG.info(name + ":" +filter.getName()+" is at STANDBY");
            autochanger.getAutochangerTrucks().goToStandby();
            autochanger.grabFilterAtStandby();
            if (autochanger.getFilterOnTrucksID() != filter.getFilterID()) {
                throw new FailedCommandException("FilterID at STANDBY is not consistant with filterID in FCS memory.");
            }
            carousel.ungrabFilterAtStandby(filter.getName());
            
        } else if (autochanger.getFilterOnTrucksID() == filter.getFilterID()) {
            FCSLOG.info(name + ":" +filter.getName()+" is already on autochanger Trucks.");
        }
        /*
        Filter can be now moved to ONLINE.
        TODO make method moveAndClampFilterOnline() 
        */
        autochanger.getAutochangerTrucks().goToOnline();
        autochanger.getOnlineClamps().closeAndLockClamps();
//        autochanger.getAutochangerTrucks().moveAndClampFilterOnline();
        carousel.releaseClamps();
    }
    
    /**
     * This command is used when a filter is held by autochanger trucks and we want to move back this 
     * filter on Carousel.
     * Socket at STANDBY is empty.
     */
    private void autochangerStoreFilterOnCarousel() {
        if (carousel.isReadyToGrabAFilterAtStandby()){
            //this.getSubsystem().updateAgentState("FILTER", FcsEnumerations.FilterState.valueOf("UNSETTING_FILTER"));

            if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.getOnlineClamps().isLocked()) {
                autochanger.getOnlineClamps().unlockAndOpenClamps();
            }
            autochanger.getAutochangerTrucks().goToStandby();
            carousel.grabFilterAtStandby(filterManager.getFilterByID(autochanger.getFilterOnTrucksID()));
            autochanger.ungrabFilterAtStandby();
        } else {
            throw new FcsHardwareException(name + " autochanger can't store filter at STANDBY because"
                    + " carousel clamps are not ready to receive a filter.");
        }
    }

    /**
     * Load a filter from the loader to the camera. At the end of this operation
     * the filter is grabbed on the autochanger at Handoff position.
     *
     * @param aFilter
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void loadFilter(Filter aFilter){
            //TODO check the conditions for this action
        //the camera is in the position "horizontal"
        //the auto changer trucks are empty at "Hand-Off" position
        //the carousel socket at "Stand-by" position is empty
        //The Loader carrier is retracted at "Storage" position and the filter is clamped

        if (!carousel.isReadyToGrabAFilterAtStandby()) {
            throw new RejectedCommandException(name 
                    + " can't load a filter when socket at STANDBY position is not empty and ready.");
        }

        if (!(autochanger.getAutochangerTrucks().isAtHandoff() && autochanger.isEmpty())) {
            throw new RejectedCommandException(name 
                    + " autochanger is not empty at STANDOFF position; can't load a filter.");
        }

        connectLoader();

        if (!(loader.isHoldingFilter() && loader.getCarrier().isAtStorage())) {
            throw new RejectedCommandException(name 
                    + " can't load filter because loader is not holding a filter at storage position.");
        }

        loader.getCarrier().goToHandOff();
        autochanger.getAutochangerTrucks().goToHandOff();

        autochanger.grabFilterAtHandoff(aFilter);
        aFilter.setFilterLocation(FilterLocation.AUTOCHANGER);
        loader.getClamp().open();
        loader.getCarrier().goToStorage();

        disconnectLoader();

    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Carousel at STANDBY. The
     * trucks of the autochanger have to be empty at STANDOFF position.
     *
     * @param aFilter
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    public void unloadFilter(Filter aFilter) {
            //TODO check the conditions for this action
        //Conditions in which this operation is authorized :
        //the camera is in the position "horizontal"
        //the auto changer trucks are empty at the "Hand-Off" position
        //the carousel socket at standby position has the filter to extract
        //The Loader carrier is empty, in storage position, the loader clamps are unclamped
        if (!aFilter.isOnCarousel()) {
            throw new RejectedCommandException(name + " " + aFilter.getName() + " is not on the carousel, can't unload it.");
        }
        if (!carousel.isAtStandby(aFilter)) {
            throw new RejectedCommandException(name + " " + aFilter.getName() + " is not on the carousel at standby position, can't unload it.");
        }
        if (!(autochanger.getAutochangerTrucks().isAtHandoff() && autochanger.isEmpty())) {
            throw new RejectedCommandException(name + " autochanger is not empty at STANDOFF position; can't unload a filter.");
        }

        this.connectLoader();
        FCSLOG.info("=======> Autochanger trucks about to move empty to standby position.");
        autochanger.getAutochangerTrucks().goToStandby();
        autochanger.grabFilterAtStandby();
            // carousel clamps are opened and go in the READYTOCLAMP state
        // when autochanger moves up the filter
        FCSLOG.info("=======> Carousel is ungrabbing filter at standby position.");
        carousel.ungrabFilterAtStandby(aFilter.getName());
        //autochanger trucks hold the filter and move it to Online Position
        FCSLOG.info("=======> Autochanger trucks about to move loaded with the filter to handoff position.");
        autochanger.moveFilterToHandoff(aFilter);
        FCSLOG.info("=======> Carousel about to release Clamps.");
        carousel.releaseClamps();

        loader.getCarrier().goToHandOff();

        loader.getClamp().close();

        autochanger.ungrabFilterAtHandoff(aFilter);

        loader.getClamp().clamp();

        loader.getCarrier().goToStorage();
        aFilter.setFilterLocation(FilterLocation.OUT);

        this.disconnectLoader();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer : carousel, autochanger and loader.
     * @throws FcsHardwareException 
     */
    @Override
    public void updateStateWithSensors()  {
        super.updateStateWithSensors();
        carousel.updateStateWithSensors();
        autochanger.updateStateWithSensors();
        if (loader.isConnectedOnCamera()) {
            loader.updateStateWithSensors();
        }
    }

    @Override
    public void postStart() {
        checkFiltersLocation();
        checkAutochangerTrucksLocation();
    }
    
    @Override
    public void shutdown() {
        super.shutdown();
        this.saveFilterLocation();
    }



}
