package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_SENSOR_ERROR;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * A class to model a clamp that holds a filter on the carousel. 
 * Each clamp on the carousel is coupled with 2 sensors which detect
 * the presence of a filter, or if the clamp is locked or unlocked. This class
 * provides a method to lock, unlock or release the clamps, and to compute the
 * clampState of the clamp. The clampState of the clamp is computed, updated and
 * published on the status bus each tick of the timer.
 *
 *
 * @author virieux
 */
public class CarouselClamp extends MobileItem {

    @LookupField(strategy = Strategy.TREE)
    private MainModule mainModule;

    /* to be able to known if autochanger hold a filter or not at STANDBY*/
    @LookupField(strategy = Strategy.TREE)
    private FilterHolder autochanger;
    
    /* to be able to known if autochanger hold a filter or not at STANDBY*/
    @LookupField(strategy = Strategy.TREE)
    private Carousel carousel;    

    /**
     * The motor controller which controls this clamp. This is initialized in
     * the initModule method of CarouselSocket.
     */
    protected EPOSController controller;
    
    /**
     * The filterPresenceSensor detects where if the filter is in the clamp.
     *
     * @See enum PresenceFilterStateOnClamp
     */
    private final SensorPluggedOnTTC580 filterPresenceSensor;

    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final SensorPluggedOnTTC580 lockSensor;

    protected FilterClampState clampState = FilterClampState.UNDEFINED;
    protected FilterPresenceStatus filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
    private LockStatus lockStatus = LockStatus.UNKNOWN;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    @ConfigurationParameter
    protected int currentToUnlock = 1500;

    @ConfigurationParameter
    protected int currentToMaintainUnlocked = 200;
    
    /**
     * if lockSensor returns a value < lockSensorMinValue, it's an ERROR.
     */
    @ConfigurationParameter(range = "0..32768")
    private Integer lockSensorMinValue = 200;

    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA: the
     * sensor is unlocked. If the lock sensor returns a value between
     * lockSensorValueA and lockSensorValueB: we don't know. If the sensor
     * returns a value between lockSensorValueB and ValueC: the clamp is locked.
     * If the sensor returns a value greater than valueC the sensor is in ERROR.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorValueA = 5000; /*6000 for clampXplus*/

    /**
     * If sensor returns a value between lockSensorValueB and lockSensorMaxValue, the clamp is locked.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorValueB = 9500;

    /**
     * If sensor returns a value above lockSensorMaxValue, the sensor is in error.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorMaxValue = 11000;

    @ConfigurationParameter(range = "0..32768")
    protected Integer lockSensorOffset = 0;
    

    /**
     * if filter presence sensor returns a value between 0 and filterPresenceMinValue : sensor is in error.
     *
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer filterPresenceMinValue = 50; /*1533 for clampXplus*/

    /**
     * if filter presence sensor returns a value between filterPresenceMinValue
 and filterPresenceValueA : the filter is engaged and lockable.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer filterPresenceValueA = 1200; /*10500 for clampXplus in January 2016 (hardware)*/

    /**
     * if filter presence sensor returns a value between filterPresenceValueA
 and filterPresenceValueB filter is not lockable. And if the value is greater
     * than valueC, there's no filter.
     */
    @ConfigurationParameter(range = "0..32768")
    protected Integer filterPresenceValueB = 10000;
    
    @ConfigurationParameter(range = "0..32768")
    private Integer filterPresenceMaxValue = 12000;
    

    @ConfigurationParameter(range = "0..32768")
    protected Integer filterPresenceOffset;

    @ConfigurationParameter(range = "0..10000")
    protected long timeoutForUnlocking = 1500;

    @ConfigurationParameter(range = "0..10000")
    protected long timeoutForReleasing = 1500;

    /**
     * Build a new CarouselClampModule with a tickMillis value:5000;
     *
     * @param filterPresenceSensor
     * @param lockSensor
     */
    public CarouselClamp(
            SensorPluggedOnTTC580 filterPresenceSensor,
            SensorPluggedOnTTC580 lockSensor
    //            Thermometer thermometer
    ) {
        this.filterPresenceOffset = 0;
        this.filterPresenceSensor = filterPresenceSensor;
        this.lockSensor = lockSensor;
//        this.thermometer = thermometer;
    }

    /**
     * Returns the amount of current needed to unlock this clamp. This is
     * defined in the ConfigurationSystem.
     *
     * @return
     */
    public short getCurrentToUnlock() {
        return (short)currentToUnlock;
    }

    /**
     * In the initialization phase, this method is used to initialize the
     * controller. cf initModule in CarouselSocket
     *
     * @param actuator
     */
    protected void setController(EPOSController actuator) {
        this.controller = actuator;
    }

    /**
     * @return the lockSensorValueA
     * for simulation
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }

    public int getLockSensorMaxValue() {
        return lockSensorMaxValue;
    }

    /**
     * @return the filterPresenceMinValue
     */
    public int getFilterPresenceMinValue() {
        return filterPresenceMinValue;
    }

    /**
     * @return the filterPresenceValueA
     */
    public int getFilterPresenceValueA() {
        return filterPresenceValueA;
    }

    /**
     * @return the filterPresenceValueB
     */
    public int getFilterPresenceValueB() {
        return filterPresenceValueB;
    }

    /**
     * For simulation.
     *
     * @return the filterPresenceSensor
     */
    public SensorPluggedOnTTC580 getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * For simulation.
     * @return the lockSensor
     */
    public SensorPluggedOnTTC580 getLockSensor() {
        return lockSensor;
    }

    /**
     * @return the lockStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return the lockStatus")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     *
     * @return filterPresenceStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    @ConfigurationParameterChanger
    public void setLockSensorValueA(Integer lockSensorValueA) {
//        if (lockSensorValueA > lockSensorValueB) throw new IllegalArgumentException(lockSensorValueA 
//                + " bad value. Can't be > lockSensorValueB=" + lockSensorValueB);
        this.lockSensorValueA = lockSensorValueA;
    }

    @ConfigurationParameterChanger
    public void setLockSensorValueB(Integer lockSensorValueB) {
//        if (lockSensorValueB > lockSensorMaxValue) throw new IllegalArgumentException(lockSensorValueB 
//                + " bad value. Can't be > lockSensorMaxValue=" + lockSensorMaxValue);
        this.lockSensorValueB = lockSensorValueB;
    }
    
    @ConfigurationParameterChanger
    public void setLockSensorMaxValue(Integer lockSensorValueC) {
        this.lockSensorMaxValue = lockSensorValueC;
    }

    @ConfigurationParameterChanger
    public void setFilterPresenceMinValue(Integer value) {
        this.filterPresenceMinValue = value;
    }

    @ConfigurationParameterChanger
    public void setFilterPresenceValueA(Integer value) {
        this.filterPresenceValueA = value;
    }

    @ConfigurationParameterChanger
    public void setFilterPresenceValueB(Integer filterPresenceValue) {
        this.filterPresenceValueB = filterPresenceValue;
    }

    

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS
     * **********************************************
     */
    /**
     * ***********************************************************************************************
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return mainModule.allDevicesBooted();
    }


    /**
     * Returns the clampState of the clamp. If the clampState is being updated
     * and waiting for a response from a sensor, this methods waits until the
     * clampState is updated. If the clampState is not being updated, it returns
     * immediatly the clampState.
     *
     * @return clampState
     *
     */
    public FilterClampState getClampState() {

        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + ": getClampState interrupted while waiting for update.", ex);
                }

            }
            return clampState;

        } finally {
            lock.unlock();
        }
    }

    /**
     * This methods returns true if the clamp is locked. In the simulator, it is
     * overriden.
     *
     * @return true if the clamp is locked
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the clamp is locked")
    public boolean isLocked() {
        return this.lockStatus == LockStatus.LOCKED;
    }

    /**
     *
     * @return true if filter is engaged on the clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if filter is engaged on the clamp : filter presence sensors sees it ")
    public boolean isFilterEngaged() {
        return this.filterPresenceStatus == FilterPresenceStatus.LOCKABLE;
    }

    /**
     * update filterPresenceStatus from value returned by filterPresenceSensor
     */
    public void updateFilterPresenceStatus() {
        int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
        int mechaValue = newFilterPresenceSensorValue - this.filterPresenceOffset;
        if (mechaValue < this.filterPresenceMinValue) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + 
                    " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue + " must be >= " + this.filterPresenceMinValue, name);

        } else if (mechaValue < this.filterPresenceValueA) {
            this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;

        } else if (mechaValue < this.filterPresenceValueB) {
            this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;

        } else if (mechaValue < this.filterPresenceMaxValue){
            this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            
        } else {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, name + 
                    " ERROR new read value FOR FILTER POSITION SENSOR  = "
                    + newFilterPresenceSensorValue + " must be <=" + this.filterPresenceMaxValue, name);
        }
    }


    /**
     * Update lockStatus from value returned by lockSensor.
     *
     * @throws FcsHardwareException
     */
    public void updateLockStatus() {
        int mechaValue = lockSensor.getValue() - this.lockSensorOffset;
        if (mechaValue < lockSensorMinValue) {
            this.lockStatus = LockStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, 
                    name + " ERROR new read value FOR LOCK SENSOR  = " + mechaValue
                    + " should be >= " + lockSensorMinValue, name);
            
        } else if (mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB) {
            this.lockStatus = LockStatus.UNKNOWN;

        } else if (mechaValue <= lockSensorMaxValue) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
            this.raiseAlarm(CA_SENSOR_ERROR, 
                    name + " ERROR new read value FOR LOCK SENSOR  = " + mechaValue
                    + " should be <= " + lockSensorMaxValue, name);
        }
    }


    /**
     * Update clampState from values sent by sensors.
     * This method updates the clamp clampState regarding the value returned by
     * the filter presence sensor and the value returned by the method
     * isLocked(). When the update is completed, it sends a signal to threads
     * waiting to get the new value of clampState.
     * @throws FcsHardwareException
     */
    @Command(type=QUERY, description = " update Clamp State.")
    public void updateState() {
        lock.lock();

        try {
            updatingState = true;

            this.updateFilterPresenceStatus();

            this.updateLockStatus();

            this.computeClampState();

        } finally {
            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();

        }
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByCarouselClamp status = this.createStatusDataPublishedByClamp();
        s.publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }

    /**
     * Create an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByCarouselClamp createStatusDataPublishedByClamp() {
        StatusDataPublishedByCarouselClamp status = new StatusDataPublishedByCarouselClamp();
        status.setClampState(clampState);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterPositionSensorValue(filterPresenceSensor.getValue());
        status.setLockSensorValue(lockSensor.getValue());
        status.setLockStatus(lockStatus);
        return status;
    }

    /**
     * Compute the global state of the clamp given the lock sensor and the
     * presence filter sensor state. This has to be overriden for the clamp X-.
     *
     */
    private void computeClampState() {

        if (this.filterPresenceStatus == FilterPresenceStatus.ERROR
                || this.lockStatus == LockStatus.ERROR) {
            clampState = FilterClampState.ERROR;
            
        } else if (this.filterPresenceStatus == FilterPresenceStatus.NOT_LOCKABLE) {
            clampState = FilterClampState.UNLOCKABLE; 

        } else if (this.filterPresenceStatus == FilterPresenceStatus.LOCKABLE) {
            // a filter is in the socket

            if (this.lockStatus == LockStatus.LOCKED) {
                clampState = FilterClampState.CLAMPEDONFILTER;

            } else if (this.lockStatus == LockStatus.UNLOCKED) {
                clampState = FilterClampState.UNCLAMPEDONFILTER;
            } else {
                clampState = FilterClampState.UNDEFINED;
            }
        
        //TODO if nofilter do a compteClampStateNoFilter which is different for Xminus and Xplus
        } else if (this.filterPresenceStatus == FilterPresenceStatus.NOFILTER) {
        //nofilter for clampXplus (when no filter is detected)
        //impossible to have no filter and clamp LOCKED for clampXminus.

            if (this.lockStatus == LockStatus.LOCKED) {
                clampState = FilterClampState.READYTOCLAMP;

            } else if (this.lockStatus == LockStatus.UNLOCKED) {
                clampState = FilterClampState.UNCLAMPEDEMPTY;

            } else {
                clampState = FilterClampState.UNDEFINED;
            }
           

        } else {
            // NOT_LOCKABLE (UNDEFINED)
            clampState = FilterClampState.UNDEFINED;

        }
    }
   

    /**
     * Because clampXminus is different from clampXplus the way of computing
     * clampState is different when we release clamps. When a clampXminus is
     * released, (clampController is stopAction), its lockSensor doesn't detect
     * a change, so lockStatus stays UNLOCKED, therefore clampState doesn't
     * change and stays UNCLAMPEDEMPTY. When a clampXplus is released, its
     * lockSensor detects the change, and lockStatus updates to LOCKED.
     * Therefore clampState updates to READYTOCLAMP. So when we release a clamp,
     * we have to recompute clampState for clampXminus.
     *
     */
    private void computeClampStateWhenReleasing() {
        if (name.contains("Xminus") && this.clampState == FilterClampState.UNCLAMPEDEMPTY) {
            clampState = FilterClampState.READYTOCLAMP;
        }
    }

    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to
     * be released after each time it has been unlocked.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING3, type = Command.CommandType.ACTION,
            description = "Release clamp in order to get ready to clamp a filter again")
    public void release() {
        FCSLOG.info("Checking conditions for release clamp " + name + " on socket at standby position.");

        carousel.updateStateWithSensors();
        if (clampState != FilterClampState.UNCLAMPEDEMPTY) {
            throw new RejectedCommandException("Can't release a clamp if isn't unclamped and empty.");
        }

        FCSLOG.info("Releasing clamp " + name + " on socket at standby position.");

        this.executeAction(MobileItemAction.RELEASE, timeoutForReleasing);

    }

    /**
     * Unlock the clamp when a filter is locked by the clamp. This is used only
     * in engineering mode.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Unlock the clamp")
    public void unlock() {

        FCSLOG.info(name + ": "
                + "UNLOCK State1 = " + clampState.toString());
        carousel.updateStateWithSensors();

        if (clampState != FilterClampState.CLAMPEDONFILTER) {
            throw new RejectedCommandException("Can't unlock a clamp if it is already unlocked.");
        }

        if (autochanger == null) {
            throw new RejectedCommandException("NO AUTOCHANGER");
        }

        if (autochanger.isAtStandby() && !autochanger.isHoldingFilter()) {
            throw new RejectedCommandException("CANNOT UNLOCK A CLAMP if FILTER is not HELD by autochanger.");
        }
        this.executeAction(MobileItemAction.UNLOCK, timeoutForUnlocking);
    }

    @Override
    public void startAction(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCK) {
            this.controller.enable();
            this.controller.writeCurrent((short)this.currentToUnlock);

        } else if (action == MobileItemAction.RELEASE) {
            this.controller.stopAction();

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK "
                    + "or RELEASE");
        }

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action == MobileItemAction.UNLOCK) {
            return this.clampState == FilterClampState.UNCLAMPEDONFILTER;

        } else if (action == MobileItemAction.RELEASE) {
            computeClampStateWhenReleasing();
            return this.clampState == FilterClampState.READYTOCLAMP;

        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK "
                    + "or RELEASE");
        }
    }

    @Override
    @Deprecated
    public void postAction(MobileItemAction action) {
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateStateWithSensors();
    }

    /**
     * This methods read the thermometer, update the field temperature and
     * returns the value sent by the thermometer;
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public double updateTemperature() {
        //TODO implement Monitoring 
//        double temp = this.thermometer.readTemperature();
//        this.temperature = temp;
//        return temp;
        return 0;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "display configuration values for this clamp.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/filterPresenceSensor=");
        sb.append(this.filterPresenceSensor.getName());
        sb.append("#valueA=");
        sb.append(this.filterPresenceMinValue);
        sb.append("#valueB=");
        sb.append(this.filterPresenceValueA);
        sb.append("#valueC=");
        sb.append(this.filterPresenceValueB);
        sb.append("/lockSensor=");
        sb.append(this.lockSensor.getName());
        sb.append("#valueA=");
        sb.append(this.lockSensorValueA);
        sb.append("#valueB=");
        sb.append(this.lockSensorValueB);
        sb.append("#valueC=");
        sb.append(this.lockSensorMaxValue);
        return sb.toString();
    }

    //TODO check with Guillaume what is to be done to stop the actions.
    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.info("Current Command: " + s.getCurrentAction() + " " + s.getState());
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
//    @Override
//    public void postStart() {
//        ai814.initializeAndCheckHardware();
//    }
    

}
