package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is used with the final hardware and prototype in which we have
 * online clamps to hold the filter at online position. 
 * It's not used for Single Filter Test.
 * 
 * CAMERA PROTECTION SYSTEM
 * Signal comming from carousel :
 * CS : Carousel stopped on socket
 * CF0: No filter in carousel socket
 * CF1: Filer in carousel socket
 * CFC: Filter clamped on carousel socket
 * 
 * Signal coming from loader :
 * LPS: Loader at Storage position (loader is connected)
 * LRH: Loader at Handoff position, closed but not clamped
 * 
 *
 * @author virieux
 */
public class Autochanger implements HardwareController, FilterHolder, AlertRaiser, HasLifecycle {

    @LookupField(strategy = Strategy.TOP)
    private Subsystem subs;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;
    
    @LookupField(strategy = Strategy.TREE)
    private ConfigurationService configurationService;

    @LookupName
    private String name;

    @LookupField(strategy = Strategy.BYNAME)
    private BridgeToHardware tcpProxy;

    private final PlutoGatewayInterface plutoGateway;

    @LookupField(strategy = Strategy.BYNAME)
    private FilterManager filterManager;

    @LookupField(strategy = Strategy.BYNAME)
    private FilterIdentificator filterIdentificator;

    /**
     * CPS signal : loader_LPS
     */
    @LookupField(strategy = Strategy.BYNAME)
    private ComplementarySensors loaderConnectedSensors;
    
    /**
     * CPS signal : carousel_CS
     * Carousel stop on socket
     */
    @LookupField(strategy = Strategy.BYNAME)
    private ComplementarySensors carouselStoppedAtStandbySensors; 
    
    /**
     * CPS signal : carousel_CF0
     * No filter in carousel socket
     */
    @LookupField(strategy = Strategy.BYNAME)
    private ComplementarySensors carousel_CF0Sensors; 
    
    /**
     * CPS signal : carousel_CF1
     * Filter in carousel socket
     */
    @LookupField(strategy = Strategy.BYNAME)
    private ComplementarySensors carousel_CF1Sensors;    

    @LookupField(strategy = Strategy.CHILDREN)
    private AutochangerTwoTrucks autochangerTrucks;

    @LookupField(strategy = Strategy.CHILDREN)
    private AutochangerTwoLatches latches;

    @LookupField(strategy = Strategy.CHILDREN)
    private AutochangerThreeOnlineClamps onlineClamps;

    @LookupField(strategy = Strategy.TREE)
    private MainModule main;

    private int filterOnTrucksID = 0;

    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder carousel;

    @LookupField(strategy = Strategy.BYNAME)
    private FilterHolder loader;

    //si on lit la même valeur sur les 2 c'est OK
    //plage de valeurs acceptables pour l'horizontalité
    //max acceptable pour la différence entre les 2 inclinometres
    @LookupField(strategy = Strategy.BYNAME)
    private Inclinometer inclinometerXminus;

    @LookupField(strategy = Strategy.BYNAME)
    private Inclinometer inclinometerXplus;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    public Autochanger(PlutoGatewayInterface plutoGateway) {
        this.plutoGateway = plutoGateway;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * Return true if an error has been detected on sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }

    /**
     * returns trucks to be able to send commands to trucks in fcs subsystem.
     *
     * @return trucks
     */
    public AutochangerTwoTrucks getAutochangerTrucks() {
        return autochangerTrucks;
    }

    /**
     * returns onlineClamps to be able to send commands to onlineClamps in fcs
     * subsystem.
     *
     * @return onlineClamps
     */
    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    /**
     * If autochanger holds a filter, returns filterID, else returns 0.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filterID, else return 0.")
    public int getFilterOnTrucksID() {
        return filterOnTrucksID;
    }

    /**
     * Used for simulator only.
     *
     * @param filterOnTrucksID
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        this.filterOnTrucksID = filterOnTrucksID;
    }

    /**
     * set filter location to AUTOCHANGER.
     */
    public void setFilterOnAutochanger() {
        if (filterOnTrucksID != 0) {
            filterManager.setFilterLocation(filterOnTrucksID, FcsEnumerations.FilterLocation.AUTOCHANGER);
        }
    }

    /**
     * If autochanger holds a filter, return filter name else return null.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "If autochanger holds a filter, return filter name else return null.")
    public String getFilterOnTrucksName() {
        if (this.isHoldingFilter()) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return null;
        }
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean
     * is being updated and waits for a response from a sensor, this methods
     * waits until loaderConnected is updated. If the field loaderConnected is
     * not being updated, it returns immediatly the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.", ex);
                }

            }
            return loaderConnectedSensors.isOn() && !loaderConnectedSensors.isInError();

        } finally {
            lock.unlock();
        }
    }

    /**
     * TODO when autochanger is in standalone mode this method has to answer the
     * value of the fake carousel sensor.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby() {
        return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }

    /**
     * Return true if loader is connected and holding a filter. This command
     * doesn't read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if loader is connected and holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff() {
        return loader.isHoldingFilter();
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is
     * hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return tcpProxy.allDevicesBooted();
    }

    /**
     * @return if all autochanger CANopen devices are booted, identified and
     * nitialized and homing of the controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger CANopen devices are booted, identified and "
            + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return tcpProxy.allDevicesBooted() && isInitialized();
    }

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are
     * initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized()
                && latches.isInitialized() && onlineClamps.isInitialized();
    }

    @Override
    public void postStart() {
        FCSLOG.fine(name + " BEGIN postStart.");
        if (plutoGateway.isBooted()) {
            initializeGateway();
        } else {
            plutoGateway.raiseAlarmIfMissing();
        }
        FCSLOG.fine(name + " END postStart.");
    }

    private void initializeGateway() {
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsEnumerations.FcsAlert.HARDWARE_ERROR, name + " couldn't initialize gateway",
                    plutoGateway.getName(), ex);
        }
    }

    @Override
    public void checkStarted() {
        FCSLOG.info(name + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        tcpProxy.bootProcess();

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
            autochangerTrucks.initializeControllers();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " couldn't initialize autochanger", ex);
        }
        FCSLOG.info(name + " END checkStarted");
    }

    /**
     * Check if latches can be opened. If latches can't be openened because the
     * conditions for opening are not filled, this command throws a
     * RejectedCommandException. If latches are in error, it throws a
     * FcsHardwareException.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches() {
        FCSLOG.info(name + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new FcsHardwareException(name + ": latches are in ERROR state - can't unlockClamps latches.");

//        } else if(!this.getSubsystem().isInState(OperationalState.NORMAL)) {
//            /* for test bench we want to be able to open latches even if there is no filter*/
//            doRaiseWarning(name+":SECURITY","Security conditions bypassed"," in checkConditionsForOpeningLatches");
//            
        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't open latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");

        } else if (autochangerTrucks.isAtStandby()) {
            checkConditionsForOpeningLatchesAtStandby();

        } else if (autochangerTrucks.isAtHandoff()) {
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }

    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(name + ": latches are not intialized. "
                    + "Please initialize hardware first.");
        }
    }

    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position*/
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(name + " carousel is holding filter at STANDBY => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter "
                    + "- can't open latches.");
        }
    }

    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position*/
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(name + " loader is holding filter at HANDOFF => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter "
                    + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches() {

        FCSLOG.info(name + " checking conditions for closing latches.");

        updateStateWithSensors();
        String message;
        checkLatchesInitialized();

        if (latches.isInError()) {
            throw new RejectedCommandException(name + ": latches are in ERROR state - can't close latches.");

        } else if (isEmpty()) {
            message = name + ": no filter in autochanger - can't close latches.";
            throw new RejectedCommandException(message);

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(name + ": autochanger is not "
                    + "at handoff position neither at standby - can't close latches.");
        }
    }

    /**
     * log if actions on online clamps are allowed, throws an exception
     * otherwise.
     */
    @Command(type = Command.CommandType.QUERY, description = "log if actions on online clamps are allowed,"
            + "throws an exception otherwise.")
    public void checkConditionsForActioningOnlineClamps() {
        if (!isAtOnline()) {
            throw new RejectedCommandException(name + " actions not allowed on ONLINE clamps "
                    + "when trucks are not at ONLINE.");
        }
        if (isEmpty()) {
            throw new RejectedCommandException(name + " actions not allowed on ONLINE clamps "
                    + "when no filter on trucks.");
        }
        FCSLOG.info(name + " autochangerTrucks are at ONLINE, a filter is there : actions are allowed.");
    }

    /**
     * This methods checks that filter can be moved by trucks. Returns
     * immediately if autochanger trucks are empty. Throws
     * RejectedCommandException if a filter is in trucks AND held both by
     * autochanger and (loader or carousel).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion() {

        String msg = name + " can't move autochanger trucks because ";
        this.updateStateWithSensors();

        if (this.isSensorsInError()) {
            handleSensorsError();

        } else if (isEmpty()) {
            FCSLOG.info(name + " trucks are empty - can move");

        } else if (!this.onlineClamps.isOpened()) {
            throw new RejectedCommandException(name + " : a filter "
                    + "is in trucks and ONLINE clamps are NOT OPENED.");

        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);

        } else if (autochangerTrucks.isAtHandoff()) {
            FCSLOG.info(name + " : trucks are AT HANDOFF ");
            checkFilterSafetyAtHandoff(msg);

        } else if (this.autochangerTrucks.isAtOnline()) {
            FCSLOG.info(name + " : trucks are at ONLINE ");
            checkFilterSafetyAtOnline(msg);

        } else if (!latches.isHoldingFilter()) {
            throw new RejectedCommandException(msg + " latches are not CLOSED");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtStandby(String message) {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                    + "both carousel and autochanger are holding filter at STANDBY");

        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                    + "neither carousel nor autochanger are holding filter at STANDBY");
        } else {
            FCSLOG.info(name + " filter safe at STANDBY - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtHandoff(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                    + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                    + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message
                    + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message
                    + "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(name + " filter safe at HANDOFF - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE. It can
     * be move : - if no filter is at STANDBY (otherwise risk of collision) -
     * AND ONLINE clamps are opened, - AND latches are CLOSED.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtOnline(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message
                    + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                    + "- can't move trucks");
        }

        if (isHoldingFilter() && !onlineClamps.isOpened()) {
            throw new RejectedCommandException(message
                    + "onlineClamps have to be opened on filter at ONLINE - can't move trucks.");
        }

        if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(message
                    + "neither latches nor onlineClamps are holding filter at ONLINE");
        } else {
            FCSLOG.info(name + " filter safe at ONLINE - can move");
        }
    }

    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when
     * error is detected in sensors.
     *
     * @throws FcsHardwareException
     */
    private void handleSensorsError() {
        String msg = name + " error detected in sensors:";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        this.raiseAlarm(AC_SENSOR_ERROR, msg);
        throw new FcsHardwareException(msg);
    }

    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if a filter is in trucks and latches are CLOSED.")
    @Override
    public boolean isHoldingFilter() {
        return latches.isHoldingFilter();
    }

    /**
     * This method reads all the sensors and updates the trucks, latches and
     * online clamps state.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update clamp state in reading sensors.")
    @Override
    public void updateStateWithSensors() {

        plutoGateway.checkInitialized();

        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            updateState();
            updateFilterOnTrucksID();
        } finally {

            updatingState = false;
            stateUpdated.signalAll();
            lock.unlock();
            this.publishData();
        }
    }

    /**
     * update filterOnTrucksID
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "update filterOnTrucksID.")
    public void updateFilterOnTrucksID() {
        if (!latches.isEmpty()) {
            filterOnTrucksID = filterIdentificator.getFilterId();
        } else {
            filterOnTrucksID = 0;
        }
    }

    private void updateState() {
        this.autochangerTrucks.updateState();
        this.latches.updateState();
        this.onlineClamps.updateState();
        this.filterIdentificator.updateFilterId();
    }

    /**
     * Update FCS state and FCS readyness state and publish on the status bus.
     * Check that Autochanger hardware is ready to be operated and moved. This
     * means that : - all CAN open devices are booted, identified and
     * initialized, - initialization has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update FCS state and FCS readyness state and publishes on the status bus.")
    public void updateFCSStateToReady() {
        if (this.isCANDevicesReady() && latches.isInitialized()
                && autochangerTrucks.isInitialized() && onlineClamps.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /**
     * Return true if autochanger trucks are at HANDOFF. This command doesn't
     * read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoff();
    }

    /**
     * Return true if autochanger trucks are at STANDBY. This command doesn't
     * read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandby();
    }

    /**
     * return true if autochanger trucks are at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnline();
    }

    /**
     * Return true if there is no filter in the autochanger. This command
     * doesn't read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }

    void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * Close latches on filter at STANDBY. Identify filter at STANDBY in reading
     * identifier ID. Update filter location.
     *
     * @throws RejectedCommandException if autochanger trucks are not at
     * STANDBY.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, alias = "grabFilter",
            description = "Go to STANDBY to grab filter and stay at STANDBY.")
    public void grabFilterAtStandby() {
        if (!this.isAtStandby()) {
            throw new RejectedCommandException(name + ": can't grab a filter when trucks are not at STANDBY.");
        }
        FCSLOG.info(name + ": grabbing a filter at standby position.");
        latches.close();
        filterOnTrucksID = identifyFilterAtStandby();
        //todo update filterLocation has been done in method updateStateFromSensors so it's no need to do it here.
//        filterManager.setFilterLocation(filterOnTrucksID,FcsEnumerations.FilterLocation.AUTOCHANGER);
        FCSLOG.info(name + ": filter " + filterOnTrucksID + " is grabbed on autochanger");
    }

    /**
     * Open the latches at standby position. position.
     *
     * @throws RejectedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Ungrab filter at STANDBY")
    public void ungrabFilterAtStandby() {
        if (!this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(name + ": can't ungrab a filter at STANDBY when carousel is not holding filter.");
        }
        FCSLOG.info(name + ": ungrabing filter at standby position.");
        latches.open();
        //eject filter from autochanger
        //TODO don't do that here : it's the responsability of subsystem which locks on filter to set new location of filter
        synchronized (this) {
            filterManager.setFilterLocation(filterOnTrucksID, FcsEnumerations.FilterLocation.CAROUSEL);
            filterOnTrucksID = 0;
        }
    }

    /**
     * set a new value for the acquisition frequency of linear rail controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, description = "set a new value for the acquisition "
            + "frequency of linear rail controllers current, in milliseconds.")
    public void setFastAcqRateLinearRails(int rate) {
        configurationService.change("acTruckXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("acTruckXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategoriesAs("timers:fast");
    }

    /**
     * set a new value for the acquisition frequency of ONLINE clamps
     * controllers current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, description = "set a new value for the acquisition "
            + "frequency of ONLINE clamps controllers current, in milliseconds.")
    public void setFastAcqRateOnlineClamps(int rate) {
        configurationService.change("onlineClampXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampYminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategoriesAs("timers:fast");
    }

    /**
     * speed up online clamps current monitoring for the 3 clamps
     */
    @Command(type = Command.CommandType.QUERY)
    public void increaseCurrentMonitoring() {
        configurationService.loadCategories("timers:fast");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    @Command(type = Command.CommandType.QUERY)
    public void decreaseCurrentMonitoring() {
        configurationService.loadCategories("timers:slow");
    }
   

    /**
     * Read the filterID sensors at STANDBY to check filterID.
     *
     * @return
     */
    private int identifyFilterAtStandby() {
        return filterIdentificator.getFilterId();
    }

    /**
     * Creates an Object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutoChanger createStatusDataPublishedByAutoChanger() {
        StatusDataPublishedByAutoChanger status = new StatusDataPublishedByAutoChanger();
        status.setLoaderConnectedSensorValue(this.loaderConnectedSensors.isOn());
        status.setLoaderConnectedSensorsInError(this.loaderConnectedSensors.isInError());
        status.setLoaderHoldingFilterAtHandoff(this.isLoaderHoldingFilterAtHandoff());
        status.setLoaderHoldingFilterAtHandoffInError(updatingState);
        status.setCarouselHoldingFilter(this.isCarouselHoldingFilterAtStandby());
        status.setCarousel_CS(carouselStoppedAtStandbySensors.isOn());
        status.setCarousel_CF0(carousel_CF0Sensors.isOn());
        status.setCarousel_CF1(carousel_CF1Sensors.isOn());
        status.setCarousel_CS_InError(carouselStoppedAtStandbySensors.isInError());
        status.setCarousel_CF0_InError(carousel_CF0Sensors.isInError());
        status.setCarousel_CF1_InError(carousel_CF1Sensors.isInError());
        status.setInclinaisonXminus(inclinometerXminus.getInclinaison());
        status.setInclinaisonXplus(inclinometerXplus.getInclinaison());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData("autochangerGeneral",
                createStatusDataPublishedByAutoChanger()));
    }

}
