
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystems.fcs.drivers.PlutoGateway;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;

/**
 * This class simulates the Can Open Pluto Gateway we use to communicate with
 * the sensors in the whole size prototype.
 * It provides a method to force a sensor to be on or off in order to be able to simulate 
 * hardware state. 
 * For example, if we set to on the handoff position sensors on autochanger, we can simulate that
 * autochanger trucks are at HANDOFF position.
 *
 * @author virieux
 */
public class SimuPlutoGateway extends PlutoGateway implements HasLifecycle {

    /**
     * Build a new SimuPlutoGateway with a CANopen node ID and a serial number.
     * @param nodeID
     * @param serialNB 
     */
    public SimuPlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    /**
     * returns simulated readValues for the gateway.
     * In simulation we don't have a real gateway to read but we can force the readValues to simulate 
 various hardware states.
     * @return simulated readValues in hexa 
     */
    @Override
    public int[] readNewValues() {
        return readValues;
    }
    
    /**
     * Return new simulated hexavalues returned by PDO.
     * @return 
     */
    @Override
    public int[] readValuesFromPDO() {
        return readValues;
    }

    /**
     * Replace hevaValues in order to force simulated sensor to be on or off depending on the
     * argument on given as argument.
     * It changes in readValues byte sensor.getByteNumero[] it replaces the bit which correspond
     * to the sensor within the boolean on given as argument.
     *
     * @param sensor to be changed
     * @param on if true, simulates that sensor is on, else simulates that sensor if off.
     */
    public void replaceHexaValues(DigitalSensor sensor, boolean on) {

        lock.lock();

        try {
            updatingValues = true;
            replaceSensorValue(sensor, on);

        } finally {

            updatingValues = false;
            valueUpdated.signal();
            lock.unlock();
        }
    }

    private void replaceSensorValue(DigitalSensor sensor, boolean on) {
        int byteNo = sensor.getByteNumero();
        int hexaValue = this.readValues[byteNo];
        int newHexaValue;
        if (on) {
            newHexaValue = FcsUtils.force2one(hexaValue, sensor.getInputNumero());
        } else {
            newHexaValue = FcsUtils.force2zero(hexaValue, sensor.getInputNumero());
        }
        this.readValues[byteNo] = newHexaValue;
    }
    
}
