
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import org.lsst.ccs.subsystems.fcs.ForceSensor;
import org.lsst.ccs.subsystems.fcs.LoaderCarrier;
import org.lsst.ccs.subsystems.fcs.LoaderClamp;
import org.lsst.ccs.subsystems.fcs.RedondantSensors;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;

/**
 * This class is for the simulation of a pluto gateway for the loader.
 * @author virieux
 */
public class SimuLoaderPlutoGateway extends SimuPlutoGateway {

    @LookupField(strategy = Strategy.BYNAME)
    private LoaderCarrier carrier;
    @LookupField(strategy = Strategy.BYNAME)
    private LoaderClamp clamp;

    @LookupField(strategy = Strategy.BYNAME)
    private SimuLoaderCarrierController carrierController;
    
    @LookupField(strategy = Strategy.BYNAME)
    private SimuLoaderClampController hooksController;
    
    private DigitalSensor closeSensor1;
    private DigitalSensor openSensor1;
    private DigitalSensor closeSensor2;
    private DigitalSensor openSensor2;
    private DigitalSensor closeSensor3;
    private DigitalSensor openSensor3;
    private DigitalSensor closeSensor4;
    private DigitalSensor openSensor4;
    
    private DigitalSensor clampedStatusSensor;
    private ForceSensor forceSensor0;
    private ForceSensor forceSensor1;
    
    private RedondantSensors filterPresenceSensors;
    private RedondantSensors loaderOnCameraSensors;
    private BinarySensor storagePositionSensors;
    private BinarySensor handoffPositionSensors;
    private BinarySensor engagedPositionSensors;
    
    /**
     * Build a new SimuLoaderPlutoGateway
     * @param nodeID
     * @param serialNB 
     */
    public SimuLoaderPlutoGateway(int nodeID, String serialNB) {
        super(nodeID, serialNB);
    }

    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        this.closeSensor1 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor1");
        this.closeSensor2 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor2");
        this.closeSensor3 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor3");
        this.closeSensor4 = (DigitalSensor) lookup.getComponentByName("loaderCloseSensor4");
        this.openSensor1 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor1");
        this.openSensor2 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor2");
        this.openSensor3 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor3");
        this.openSensor4 = (DigitalSensor) lookup.getComponentByName("loaderOpenSensor4");
        this.clampedStatusSensor = (DigitalSensor) lookup.getComponentByName("clampedStatusSensor");
        this.forceSensor0 = (ForceSensor) lookup.getComponentByName("forceSensor0");
        this.forceSensor1 = (ForceSensor) lookup.getComponentByName("forceSensor1");
                
        this.loaderOnCameraSensors = (RedondantSensors) lookup.getComponentByName("loaderOnCameraSensors");
        this.filterPresenceSensors = (RedondantSensors) lookup.getComponentByName("loaderFilterPresenceSensors");
        this.storagePositionSensors = (RedondantSensors) lookup.getComponentByName("loaderStorageSensors");
        this.handoffPositionSensors = (BinarySensor) lookup.getComponentByName("loaderHandoffSensors");
        this.engagedPositionSensors = (BinarySensor) lookup.getComponentByName("loaderEngagedSensors");
       
        //Filter is closed && clamped and carrier is at storage position
        simulateLoaderClampIsClosed();
        simulateLoaderIsConnectedOnCamera();
        simulateClampedStatusSensorIsOn();
        simulateFilterIsOnLoader();
        simulateLoaderCarrierIsAtStorage();
    }
    
    protected void simuBinarySensorOn(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceHexaValues(((RedondantSensors) sensor).getSensor(), true);
            replaceHexaValues(((RedondantSensors) sensor).getSensorR(), true);
        } else {
            replaceHexaValues((DigitalSensor) sensor, true);
        }
    }
    
    protected void simuBinarySensorOff(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceHexaValues(((RedondantSensors) sensor).getSensor(), false);
            replaceHexaValues(((RedondantSensors) sensor).getSensorR(), false);
        } else {
            replaceHexaValues((DigitalSensor) sensor, false);
        }
    }
    
    protected void simuRedondantSensorsInError(RedondantSensors sensors) {
        replaceHexaValues(sensors.getSensor(), true);
        replaceHexaValues(sensors.getSensorR(), false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsClosed() {
        simulateHooksAreClosed();
        simulateClampedStatusSensorIsOff();
        hooksController.setPosition(clamp.getRelativePositionToClose());
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsClamped() {
        simulateHooksAreClosed();
        simulateClampedStatusSensorIsOn();
        hooksController.setPosition(clamp.getTargetPositionToClamp());
    }    
    
    private void simulateHooksAreClosed() {
        simulateHookIsClosed(closeSensor1, openSensor1);
        simulateHookIsClosed(closeSensor2, openSensor2);
        simulateHookIsClosed(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsOpened() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsOpened(closeSensor4, openSensor4);
        simulateClampedStatusSensorIsOff();
        hooksController.setPosition(clamp.getTargetPositionToOpen());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderClampIsInTravel() {
        simulateHookIsInTravel(closeSensor1, openSensor1);
        simulateHookIsInTravel(closeSensor2, openSensor2);
        simulateHookIsInTravel(closeSensor3, openSensor3);
        simulateHookIsInTravel(closeSensor4, openSensor4);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsUnknown() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }    
 
    /**
     * simulate that clamp is in error : if one hook is in error, clamp is in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampIsInError() {
        simulateHookIsInError(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsClosed(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceHexaValues(closeSensor, true);
        replaceHexaValues(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsOpened(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceHexaValues(closeSensor, false);
        replaceHexaValues(openSensor, true);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsInError(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceHexaValues(closeSensor, true);
        replaceHexaValues(openSensor, true);
    }    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHookIsInTravel(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceHexaValues(closeSensor, false);
        replaceHexaValues(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsConnectedOnCamera() {
        simuBinarySensorOn(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderIsNotConnectedOnCamera() {
        simuBinarySensorOff(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsOnLoader() {
        simuBinarySensorOn(filterPresenceSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsNotOnLoader() {
        simuBinarySensorOff(filterPresenceSensors);
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtStorage() {
        simuBinarySensorOn(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOff(engagedPositionSensors);
        carrierController.setPosition(carrier.getStoragePosition());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtHandoff() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOn(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        carrierController.setPosition(carrier.getHandoffPosition());
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsAtEngaged() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        carrierController.setPosition(carrier.getEngagedPosition());
    }    

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateLoaderCarrierIsInTravel() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateHandoffSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) handoffPositionSensors);
    }    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateStorageSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) storagePositionSensors);
    }    
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampedStatusSensorIsOn() {
        replaceHexaValues(clampedStatusSensor, true);
        simulateForceSensorsOn();
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateClampedStatusSensorIsOff() {
        replaceHexaValues(clampedStatusSensor, false);
        simulateForceSensorsOff();
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)   
    public void simulateForceSensorsOn() {
        simulateForceSensorClamped(forceSensor0, true);
        simulateForceSensorClamped(forceSensor1, true);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)   
    public void simulateForceSensorsOff() {
        simulateForceSensorClamped(forceSensor0, false);
        simulateForceSensorClamped(forceSensor1, false);
    }    
    
    public void simulateForceSensorClamped(ForceSensor forceSensor, boolean on) {
        int forceOn = 900;
        int forceOff = 0;
        if (on) {
            this.updateForceSensor(forceSensor, forceOn);
        } else {
            this.updateForceSensor(forceSensor, forceOff);
        }
    }
    
    private void updateForceSensor(ForceSensor forceSensor, int force) {
        int voltage = getVoltageFromForce(forceSensor, force);
        this.readValues[forceSensor.getByteNumero()] = voltage;
    }
    
    private static int getVoltageFromForce(ForceSensor forceSensor, int force) {
        double voltage = (force + forceSensor.getTransfert_Constante()) 
                / (100 * forceSensor.getTransfert_coeff());
        return (int) voltage;
    }
    
    
}
