
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.LoaderClamp;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is to simulate an EPOS Controller which controls the loader hooks.
 *
 * @author virieux
 */
public class SimuLoaderClampController extends SimuEPOSController {

    @LookupField(strategy=Strategy.BYNAME)
    private LoaderClamp clamp;

    private final SimuLoaderPlutoGateway plutoGateway;

    /**
     * Build a new SimuLoaderClampController
     * @param nodeID
     * @param serialNB
     * @param plutoGateway
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuLoaderClampController(
            int nodeID, String serialNB,
            SimuLoaderPlutoGateway plutoGateway,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming
    ) {
        super(nodeID, serialNB, 
                paramsForCurrent, paramsForProfilePosition, paramsForHoming);
        this.plutoGateway = plutoGateway;
    }
    
   

    @Override
    public void writeCurrent(int aValue)  {
        this.targetCurrent = aValue;
        int currentInitial = this.actualCurrent;
        FCSLOG.debug("=>initial current=" + actualCurrent);
        int stepsNB = 10;
        int step = (targetCurrent - this.actualCurrent) / stepsNB;

        for (int i = 1; i < stepsNB; i++) {

            this.actualCurrent = (short) (currentInitial + (i * step));
            clamp.updateCurrent();
            if (actualCurrent == clamp.getCurrentToClamp()) {
                FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
                this.position = clamp.getTargetPositionToClamp();
                this.plutoGateway.simulateLoaderClampIsClosed();

            } else if (actualCurrent == clamp.getCurrentToOpen()) {
                this.position = clamp.getTargetPositionToOpen();
                this.plutoGateway.simulateLoaderClampIsOpened();
            }
            clamp.publishData();
            FCSLOG.debug("i=" + i + ",actualCurrent=" + actualCurrent);
            try {
                Thread.sleep(500);
                if (clamp.getHaltRequired().get()) {
                    FCSLOG.debug(name + " STOP simulated clamp motion.");
                    return;
                }
            } catch (InterruptedException ex) {
                throw new FcsHardwareException(name + " sleep was interrupted.", ex);
            }

        }
        this.actualCurrent = aValue;
        if (actualCurrent == clamp.getCurrentToClamp()) {
            FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
            this.position = clamp.getTargetPositionToClamp();
            this.plutoGateway.simulateLoaderClampIsClamped();

        } else if (actualCurrent == clamp.getCurrentToOpen()) {
            this.position = clamp.getTargetPositionToOpen();
            this.plutoGateway.simulateLoaderClampIsOpened();
        }
        clamp.publishData();
    }

    @Override
    public void writeTargetPosition(int positionToReached) {
        this.targetPosition = positionToReached;
        int positionInitiale = this.position;
        FCSLOG.info("=>position intitiale=" + position);
        FCSLOG.info("=>targetPosition=" + targetPosition);
        FCSLOG.info("clamp.getTargetPositionToClamp()=" + clamp.getTargetPositionToClamp());
        FCSLOG.info("clamp.getRelativePositionToUnclamp()=" + clamp.getRelativePositionToUnclamp());
        int stepsNB = 10;
        int step = (targetPosition - this.position) / stepsNB;

        for (int i = 1; i < stepsNB; i++) {
            this.position = positionInitiale + (i * step);
            clamp.updatePosition();
            simulateSensors();
            FCSLOG.info("i=" + i + ",position=" + position);
            try {
                Thread.sleep(500);
            } catch (InterruptedException ex) {
                throw new RejectedCommandException(name + " sleep was interrupted.");
            }
        }
        this.position = targetPosition;
        FCSLOG.info("i=" + 10 + ",position=" + position);
        simulateSensors();
    }
    
    private void simulateSensors() {
        if (position == clamp.getTargetPositionToOpen()) {
            FCSLOG.debug("plutoGateway=" + plutoGateway.toString());
            this.plutoGateway.simulateLoaderClampIsOpened();
            
        } else if (position == clamp.getRelativePositionToClose()) {
            this.plutoGateway.simulateLoaderClampIsClosed();
            
        } else if (position == clamp.getTargetPositionToClamp()) {
            this.plutoGateway.simulateLoaderClampIsClamped();
            
        } else if (position == clamp.getTargetPositionToClamp() + clamp.getRelativePositionToUnclamp()) {
            this.plutoGateway.simulateLoaderClampIsClosed();            
            
        } else {
            this.plutoGateway.simulateLoaderClampIsInTravel();
        }
        clamp.publishData();
    }
    
}


