
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.subsystems.fcs.Carousel;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * To simulate the controller which rotates carousel.
 * @author virieux
 */
public class SimuCarouselController extends SimuEPOSController implements EPOSControllerForCarousel {
    
    @LookupField(strategy=Strategy.BYNAME)
    private Carousel carousel;
    
    /**
     * Build a new SimuCarouselController
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public SimuCarouselController(
            int nodeID, String serialNB, 
            Map<String, Integer> paramsForCurrent, 
            Map<String, Integer> paramsForProfilePosition, 
            Map<String, Integer> paramsForHoming) {
        super(nodeID, serialNB, 
                paramsForCurrent, paramsForProfilePosition, paramsForHoming);
    }
    
    /**
     * To initialize a simulated ssiPosition.
     * @param ssiPosition 
     */
    public void setSsiPosition(int ssiPosition) {
        this.ssiPosition = ssiPosition;
        this.position = ssiPosition;
    }    
    
    @Override
    public void init() {
        /* when simulation starts carousel position is set to socket1 standby position.*/
        setSsiPosition(carousel.getSocketByName("socket1").getStandbyPosition());
    }
    
    @Override
    public void enableAndWriteRelativePosition(int pos) {
        enable();
        changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
        writeTargetPosition(pos);
        writeControlWord(0x7F);
    }     
        
    
    /**
     * This methods does nothing but save the relativePosition given as argument.
     * @param relativePosition 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position.")
    @Override
    public void writeTargetPosition(int relativePosition) throws FcsHardwareException {
        FCSLOG.info(name + "=> carousel is rotating.");
        FCSLOG.info(name + "=> position initiale=" + this.position);
        FCSLOG.info(name + "=> relative target position=" + relativePosition);
        int iterNB = 10;
        int step = relativePosition / iterNB;
        FCSLOG.info("step=" + step);
        int initialPosition = carousel.getPosition();
        try {
            for (int i = 1; i < iterNB; i++) {
                setSsiPosition(initialPosition + (i * step));
                FCSLOG.info(name + " i=" + i + ", position=" + position);
                try {
                    Thread.sleep(500);
                    if (carousel.getHaltRequired().get()) {
                        FCSLOG.fine(name + " STOP simulated carousel motion.");
                        return;
                    }
                } catch (InterruptedException ex) {
                    throw new FcsHardwareException(name + " sleep was interrupted.",ex);
                }
            }
            setSsiPosition(initialPosition + relativePosition);
            carousel.updatePosition();
        } catch (FcsHardwareException ex) {
            FCSLOG.error(name+" should not raise an Exception:",ex);
        }
        
    }
    
    
}
