
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MAX;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SENSOR14BITS_MIN;
import org.lsst.ccs.subsystems.fcs.drivers.CanCBXAI814;

/**
 * This class simulates a filter presence sensor for the carousel.
 * In the carousel the new values of the filter presence sensor are read from a PDO.
 * @author virieux
 */
public class SimuCarouselClampFilterPresenceSensor extends SimuCanOpenSensor14bits implements AutochangerTruckPositionListener {
    
    int acTrucksStandbyPosition = 992000;
    
    /**
     * Create a new simulated filter presence sensor for the Carousel.
     * @param adc
     * @param numOfAnalogInput 
     */
    public SimuCarouselClampFilterPresenceSensor( 
            CanCBXAI814 adc, int numOfAnalogInput) {
        super(adc, numOfAnalogInput);
    }
    
    @Override
    public void start() {
        simulateFilterIsLockable();
    }
    
    /**
     * We observe SimuAutochangerPlutoGateway.
     * When it changes trucks position we are notified, we can then simulate that carousel is holding filter or not.
     * @param newPos the new trucks position
     */
    @Override
    public void onTruckPositionChange(int newPos) {
        FCSLOG.finest(name+" :old Value=" + this.printValue());
        if (this.isAtStandby()) {
            FCSLOG.finest(name + " is AT STANDBY.");
        }
        if (s.isInState(PhaseState.OPERATIONAL) && this.isAtStandby()
                && autochanger.isHoldingFilter()) {
            if (newPos <  acTrucksStandbyPosition - 500) {
                /**
                 * filter is on autochanger trucks.
                 * So if position is not acTrucksStandbyPosition, clamp filter presence sensor doesn't detect presence filter.
                 */
                simulateNoFilter();
                
            } else if (newPos == acTrucksStandbyPosition) {
                simulateFilterIsLockable();
            }
        }
        FCSLOG.finest(name+" :new Value=" + this.printValue());
    }
    
    /**
     * Simulates the filter is seen by the sensor.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsLockable() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, clamp.getFilterPresenceValueA(), 
                clamp.getFilterPresenceValueB());
    }

    /**
     * Simulates the filter is seen by the sensor but is not Lockable.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsNotLockable() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, clamp.getFilterPresenceValueB(), 
                clamp.getFilterPresenceValueC());
    }

    /**
     * Simulates no filter is seen by the sensor.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateNoFilter() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, clamp.getFilterPresenceValueC(), SENSOR14BITS_MAX);
    }

    /**
     * Simulates the sensor is in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public void simulateFilterIsInError() {
        ((SimuCanOpenADC) adc).simulateSensorValue(this, SENSOR14BITS_MIN, clamp.getFilterPresenceValueA());
    }
    

}
