
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The CommandDispenser provides methods and internal classes to stored the commands
 * we send to the tcpProxy and to wait for the response. 
 * It is used by the method call() of the tcpProxy to stored the initial commands and then to
 * retrieve the response. 
 * The response is updated by the tcpProxy readingThread when it arrives from the tcp Socket. 
 * The CommandToRegister object has to be removed from the map by the client classes.
 *
 * @author virieux
 */
public class CommandDispenser {
    
    /**
    * A Logger for CommandDispenser.
    */
    public static final Logger LOCAL_LOG = Logger.getLogger("org.lsst.ccs.subsystems.fcs.CommandDispenser");


    /**
     * An internal class for the command to be registered.
     * It provides a method to store the response of the command : updateResponse.
     * And a method to retrieve the response : getResponse
     * It has a lock because many threads can store or retrieve a command at the same time.
     */
    protected static class CommandToRegister {

        private final String initialCommand;
        private volatile String response;
        private final Lock lock = new ReentrantLock();
        private final Condition responseArrived = lock.newCondition();

        /**
         * Creates a CommandToRegister from the text command.
         * @param aTextCommand 
         */
        public CommandToRegister(String aTextCommand) {
            this.initialCommand = aTextCommand;
            this.response = null;
        }

        /**
         * Update the response of the command from the String theResponse.
         * @param theResponse 
         */
        public void updateResponse(String theResponse) {
            lock.lock();
            try {
                this.response = theResponse;
                
                responseArrived.signal();
            } finally {
                lock.unlock();
            }
        }

        /**
         * Return the response to the initial command. If the response is
         * not yet arrived, it waits for it.
         *
         * @param timeout we don't want to wait the response more than timeout
         * @return this.response when it's arrived
         * @throws
         * org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException
         */
        public String getResponse(long timeout)  {

            lock.lock();

            try {
                while (this.response == null) {
                    try {
                        boolean responseOK = responseArrived.await(timeout, TimeUnit.MILLISECONDS);
                        if (!responseOK) {
                            throw new CanOpenCallTimeoutException("timeout expired for command:" + initialCommand);
                        }
                        if (this.response == null) {
                            LOCAL_LOG.error("no response for " + initialCommand);
                            throw new CanOpenCallTimeoutException("response=null for command:" + initialCommand);
                        }

                    } catch (InterruptedException ex) {
                        LOCAL_LOG.error("Interrupted while waiting for a response for command :  " + initialCommand, ex);
                    }
                }
                LOCAL_LOG.finest("Return response: " + response + " for command: " + this.toString());
                return this.response;
            } finally {
                lock.unlock();
            }
        }

        @Override
        public String toString() {
            return initialCommand + "/" + response;
        }
    }

    /* A HashMap to store the CommandToRegister*/
    private final HashMap<String, CommandToRegister> map = new HashMap<>();

    /**
     * Registers a command and returns a token to be used to retrieve the command in the map.
     * The token is the key of the map.
     * The command is stored as a CommandToRegister in the value of the map.
     *
     * @param command
     * @return
     */
    public synchronized String register(String command) {
        if (command.isEmpty()) {
            throw new IllegalArgumentException("null Command");
        }
        String key;
        String nodeID;
        String[] words = command.split(",");
        String commandWord = words[0];
        if (words.length > 1) {
            nodeID = words[1];
            key = commandWord + nodeID;
        } else {
            key = commandWord;
        }

        LOCAL_LOG.finest("command word:" + commandWord);
        LOCAL_LOG.finest("key: " + key);
        LOCAL_LOG.finest("Command to register: " + command);
        map.put(key, new CommandToRegister(command));
        return key;
    }

    /**
     * Registers the response for a command. The command is retrieve by its token given as argument,
     * and the response is given as argument too.
     *
     * @param token
     * @param response
     * @return
     */
    public String registerResponse(String token, String response) {
        LOCAL_LOG.finest("Registering token response: " + token + " " + response);
        if (response == null) {
            throw new IllegalArgumentException("null Response");
        }
        if (!isTokenUsed(token)) {
            throw new IllegalArgumentException("invalid Token: " + token);
        }
        CommandToRegister commandToRegister;
        //the actions on the map have to be synchronized.
        synchronized (this) {
            commandToRegister = map.get(token);
        }
        //we can't update the response in the synchronized block because the
        //thread waiting for the response has acquired a lock on the object CommandRegister.
        commandToRegister.updateResponse(response);
        return token;
    }

    /**
     * Returns the response to a command. The command is retrieved by the token given as augument.
     * This methods waits for the response coming from the field bus.
     * If the response is not received during a timeout given as argument, a CanOpenCallTimeoutException
     * is thrown.
     * @param token
     * @param timeout
     * @return
     * @throws CanOpenCallTimeoutException 
     */
    public String getCommandResponse(String token, long timeout) {
        if (!isTokenUsed(token)) {
            throw new IllegalArgumentException("invalid Token: " + token);
        }

        CommandToRegister commandToRegister;
        synchronized (this) {
            commandToRegister = map.get(token);
        }
        return commandToRegister.getResponse(timeout);
    }

    /**
     * Remove a token.
     * @param token 
     */
    public synchronized void remove(String token) {
        LOCAL_LOG.finest("removing token = " + token);
        map.remove(token);
    }

    /**
     * Tells if the token is used.
     * <B>beware</B> : in a multihreaded environment there may be race condition
     * where this method returns true but where an Exception is fired upon call
     * of callback Method.
     *
     * @param token
     * @return
     */
    public synchronized boolean isTokenUsed(String token) {
        if (token == null) {
            return false;
        }
        return map.containsKey(token);
    }
    

}
